#' Generalized assumption-free bound
#'
#' `GAFbound()` returns a list with the GAF upper and lower bounds. The
#' sensitivity parameters can be inserted directly or as output from
#' `sensitivityparametersM()`.
#'
#' @param whichEst Input string. Defining the causal estimand of interest.
#'   Available options are as follows. (1) Risk ratio in the total
#'   population: `"RR_tot"`, (2) Risk difference in the total population:
#'   `"RD_tot"`, (3) Risk ratio in the subpopulation: `"RR_sub"`, (4) Risk
#'   difference in the subpopulation: `"RD_sub"`.
#' @param sens Possible method to input sensitivity parameters. `sens` can 
#'   be the output from sensitivityparametersM(), a data.frame with columns 
#'   'parameter' and 'value', or a name list with correct names (e.g. 
#'   `"RR_UY_T1"`,  `"RR_UY_T0"`, etc.). If not supplied, parameters can be 
#'   entered manually as specified below.
#' @param M Possible method to input sensitivity parameter. Must be between 0 
#' and 1, larger than m and smaller than max_t P(Y=1|T=t,I_s=1).
#' @param m Possible method to input sensitivity parameter. Must be between 0 
#' and 1, smaller than M and larger than min_t P(Y=1|T=t,I_s=1).
#' @param outcome Input vector. A binary outcome variable. Either the data
#' vector (length>=3) or two conditional outcome probabilities with
#' P(Y=1|T=1,I_s=1) and P(Y=1|T=0,I_s=1) as first and second element.
#' @param treatment Input vector. A binary treatment variable. Either the data
#' vector (length>=3) or two conditional treatment probabilities with
#' P(T=1|I_s=1) and P(T=0|I_s=1) as first and second element.
#' @param selection Input vector or input scalar. A binary selection variable or
#'   a selection probability. Can be omitted for subpopulation estimands.
#'
#' @return A list containing the upper and lower GAF bounds.
#' @export
#' @examples
#'
#' # Example with selection indicator variable.
#' y = c(0, 0, 0, 0, 1, 1, 1, 1)
#' tr = c(0, 0, 1, 1, 0, 0, 1, 1)
#' sel = c(0, 1, 0, 1, 0, 1, 0, 1)
#' Mt = 0.8
#' mt = 0.2
#' GAFbound(whichEst = "RR_tot", M = Mt, m = mt, outcome = y, treatment = tr,
#'  selection = sel)
#'
#' # Example with selection probability.
#' selprob = mean(sel)
#' GAFbound(whichEst = "RR_tot", M = Mt, m = mt, outcome = y[sel==1],
#'  treatment = tr[sel==1], selection = selprob)
#'
#' # Example with subpopulation and no selection variable or probability.
#' Ms = 0.7
#' ms = 0.1
#' GAFbound(whichEst = "RR_sub", M = Ms, m = ms, outcome = y, treatment = tr)
#'
#' # Example with simulated data.
#' n = 1000
#' tr = rbinom(n, 1, 0.5)
#' y = rbinom(n, 1, 0.2 + 0.05 * tr)
#' sel = rbinom(n, 1, 0.4 + 0.1 * tr + 0.3 * y)
#' Mt = 0.5
#' mt = 0.05
#' GAFbound(whichEst = "RD_tot", M = Mt, m = mt, outcome = y, treatment = tr,
#'  selection = sel)
#'  
# Example specifying the sensitivity parameters from sensitivityparametersM().
#' # Risk ratio in the subpopulation. DGP from the zika example.
#' V = matrix(c(1, 0, 0.85, 0.15), ncol = 2)
#' U = matrix(c(1, 0, 0.5, 0.5), ncol = 2)
#' Tr = c(-6.2, 1.75)
#' Y = c(-5.2, 5.0, -1.0)
#' S = matrix(c(1.2, 2.2, 0.0, 0.5, 2.0, -2.75, -4.0, 0.0), ncol = 4)
#' probT1 = 0.286
#' probT0 = 0.004
#' senspar = sensitivityparametersM(whichEst = "RR_sub", whichBound = "GAF", 
#'  Vval = V,  Uval = U, Tcoef = Tr, Ycoef = Y, Scoef = S, Mmodel = "L",
#'  pY1_T1_S1 = probT1, pY1_T0_S1 = probT0)
#'  
#' GAFbound(whichEst = "RR_sub", sens = senspar, outcome = c(probT1, probT0),
#'  treatment = c(0.01, 0.99))
#'
#' @references Zetterstrom, Stina. "Bounds for selection bias using outcome
#'  probabilities" Epidemiologic Methods 13, no. 1 (2024): 20230033
#'
GAFbound <- function(whichEst, sens = NULL, M, m, outcome, treatment, selection = NULL)
{
  
  if (!is.null(sens)) {
    # Case 1: Output from sensitivityparametersM()
    if (inherits(sens, "sensparams")) {
      params <- stats::setNames(as.numeric(sens$value), sens$parameter)
      # Case 2: Data frame with columns 'parameter' and 'value'
    } else if (is.data.frame(sens)) {
      params <- stats::setNames(as.numeric(sens$value), sens$parameter)
      # Case 3: Named list with correct names
    } else if (is.list(sens)) {
      params <- unlist(sens)
    } else {
      stop("'sens' must be a named list (with correct names), data.frame (with columns 'parameter' and 'value'), or 'sensparams' object.")
    }

    # Extracting the sensitivity parameters from the input list.
    if(whichEst == "RR_tot" | whichEst == "RD_tot")
    {
      M = params[["M_T"]]
      m = params[["m_T"]]
    } else{
      M = params[["M_S"]]
      m = params[["m_S"]]
    }
    
  }
  
  if(whichEst == "RR_tot" | whichEst == "RD_tot")
  {
    if(is.null(selection[1])){stop('The argument "selection" must be specified for total population estimands.')}
  }


  if(is.null(selection[1])){selection = 1} # Give arbitrary value in case of NULL.

  if((m < 0 | m > 1 | M < 0 | M > 1))
    stop("M and m cannot be smaller than 0 or larger than 1.")
  if(m >= M)
    stop("M must be larger than m.")

  # Calculate the GAF bound.
  bound = calcGAFbound(whichEst, M, m, outcome, treatment, selection, "GAF")
  bound = round(bound, 2)
  # Output.
  heading = c("GAF lower bound", "GAF upper bound")
  values = list(bound[1], bound[2])
  returnDat = matrix(cbind(heading, values), ncol = 2)
  return(returnDat)
}
