% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BS3.R
\name{BS3}
\alias{BS3}
\title{The Birnbaum-Saunders family - Bourguignon & Gallardo (2022)}
\usage{
BS3(mu.link = "log", sigma.link = "logit")
}
\arguments{
\item{mu.link}{defines the mu.link, with "log" link as the default 
for the mu parameter.}

\item{sigma.link}{defines the sigma.link, with "logit" link as the default 
for the sigma.}
}
\value{
Returns a gamlss.family object which can be used to fit a 
BS3 distribution in the \code{gamlss()} function.
}
\description{
The function \code{BS3()} defines The Birnbaum-Saunders, 
a two parameter distribution, for a \code{gamlss.family} object 
to be used in GAMLSS fitting 
using the function \code{gamlss()}.
}
\details{
The Birnbaum-Saunders with parameters \code{mu} and \code{sigma}
has density given by

\eqn{f(x|\mu,\sigma) = \frac{(1-\sigma)y+\mu}{2\sqrt{2\pi\mu\sigma(1-\sigma)}y^{3/2}} \exp{\left[ \frac{-1}{2\sigma} \left( \frac{(1-\sigma)y}{\mu} + \frac{\mu}{(1-\sigma)y} - 2 \right) \right]} }

for \eqn{x>0}, \eqn{\mu>0} and \eqn{0<\sigma<1}. In this 
parameterization 
\eqn{Mode(X)=\mu} and 
\eqn{Var(X)=(\mu\sigma)^2(1+5\sigma^2/4)}.
}
\examples{
# Example 1
# Generating some random values with
# known mu and sigma
y <- rBS3(n=50, mu=2, sigma=0.2)

# Fitting the model
require(gamlss)
mod1 <- gamlss(y~1, sigma.fo=~1, family=BS3)

# Extracting the fitted values for mu and sigma
# using the inverse link function
exp(coef(mod1, what="mu"))
exp(coef(mod1, what="sigma"))

# Example 2
# Generating random values for a regression model

# A function to simulate a data set with Y ~ BS3
\dontrun{
gendat <- function(n) {
  x1 <- runif(n)
  x2 <- runif(n)
  mu <- exp(1.45 - 3 * x1)
  inv_logit <- function(x) 1 / (1 + exp(-x))
  sigma <- inv_logit(2 - 1.5 * x2)
  y <- rBS3(n=n, mu=mu, sigma=sigma)
  data.frame(y=y, x1=x1, x2=x2)
}

set.seed(1234)
dat <- gendat(n=100)

mod2 <- gamlss(y~x1, sigma.fo=~x2, 
               family=BS3, data=dat,
               control=gamlss.control(n.cyc=100))

summary(mod2)
}

# Example 3
# The response variable is the ratio between the average
# rent per acre planted with alfalfa and the corresponding 
# average rent for other agricultural uses. The density of
# dairy cows (X2, number per square mile) is the explanatory variable. 
library(alr4)
data("landrent")

landrent$ratio <- landrent$Y / landrent$X1

with(landrent, plot(x=X2, y=ratio))

mod3 <- gamlss(ratio~X2, sigma.fo=~X2, 
               data=landrent, family=BS3)

summary(mod3)
logLik(mod3)

}
\references{
Bourguignon, M., & Gallardo, D. I. (2022). A new look at the 
Birnbaum–Saunders regression model. Applied Stochastic Models in 
Business and Industry, 38(6), 935-951.
}
\seealso{
\link{dBS3}.
}
