#' @title Weighted Average Stress Index (WASI)
#' @description
#' A composite measure that computes the weighted mean for each genotype across
#' multiple stress indices, accounting for whether higher or lower values are better.
#'
#' @param data A data frame containing genotype IDs (`Gen`) and stress index values
#'  (`GMP`, `HM`, `MP`, `MRP`, `PYR`, `RSI`, `SSPI`, `STI`, `TOL`, `YI`, `YSI`).
#' @param decimals Integer; number of decimal places to use for Excel-style data.
#'  All indices are rounded to this precision before ranking and again for the
#'  index × rank products (default `5`).
#'
#' @return A data frame with `Gen` and its computed `WASI` (sorted in descending order).
#'
#' @details
#' The Weighted Average Stress Index for genotype *i* is
#' \deqn{WASI_i = \frac{\sum_{j=1}^{k} (X_{ij} \times R_{ij})}{\sum_{j=1}^{k} R_{ij}},}
#' where \eqn{X_{ij}} is the value of genotype *i* for index *j*, and \eqn{R_{ij}}
#' is its rank, determined by whether higher or lower values are favorable.
#' Ranks use Excel-like behavior (`ties.method = "min"`). Indices are rounded to
#' `decimals` places prior to ranking and multiplication to better match Excel-style
#' calculations.
#'
#' @examples
#' df <- data.frame(
#' Gen = paste0("G", 1:5),
#' GMP  = c(1:5),
#' HM   = c(6:10),
#' MP   = c(11:15),
#' MRP  = c(16:20),
#' PYR  = c(21:25),
#' RSI  = c(26:30),
#' SSPI = c(31:35),
#' STI  = c(0.1, 0.2, 0.3, 0.4, 0.5),
#' TOL  = c(41:45),
#' YI   = c(0.6, 0.7, 0.8, 0.9, 1.0),
#' YSI  = c(0.2, 0.3, 0.4, 0.5, 0.6)
#' )
#' WASI(df)
#'
#' @export
WASI <- function(data, decimals = 5) {
  stopifnot("Gen" %in% names(data))
  lower_better  <- c("TOL","SSPI","PYR","RSI")
  higher_better <- c("STI","YI","YSI","MP","GMP","HM","MRP")
  indices <- c(lower_better, higher_better)

  # Column presence
  miss <- setdiff(indices, names(data))
  if (length(miss)) stop("Missing columns: ", paste(miss, collapse = ", "))

  # Numeric & NA checks
  nonnum <- vapply(data[indices], function(x) !is.numeric(x), logical(1))
  if (any(nonnum)) stop("Non-numeric columns: ", paste(names(nonnum)[nonnum], collapse = ", "))
  has_na <- vapply(data[indices], function(x) any(is.na(x)), logical(1))
  if (any(has_na)) stop("NA values in: ", paste(names(has_na)[has_na], collapse = ", "))

  # Round indices first
  idx_round <- as.data.frame(lapply(data[indices], function(x) round(x, decimals)))

  # computed on the rounded values
  rank_mat <- sapply(indices, function(ind) {
    x <- idx_round[[ind]]
    if (ind %in% higher_better) {
      rank(-x, ties.method = "min")   # higher = better
    } else {
      rank( x, ties.method = "min")   # lower  = better
    }
  })

  # Weighted average (round each product to 'decimals' before summing)
  prod_mat <- as.matrix(idx_round) * rank_mat
  prod_mat <- round(prod_mat, decimals)

  weighted_sum <- rowSums(prod_mat)
  rank_sum     <- rowSums(rank_mat)

  WASI_val <- round(weighted_sum / rank_sum, decimals)

  out <- data.frame(Gen = data$Gen, WASI = WASI_val, row.names = NULL)
  out[order(out$WASI, decreasing = TRUE), , drop = FALSE]
}
