#' Optimal Binning using Local Polynomial Density Binning (LPDB)
#'
#' Performs supervised discretization of continuous numerical variables using a
#' novel approach that combines non-parametric density estimation with
#' information-theoretic optimization. The algorithm first identifies natural
#' clusters and boundaries in the feature distribution using local polynomial
#' density estimation, then refines the bins to maximize predictive power.
#'
#' @param feature A numeric vector representing the continuous predictor variable.
#'   Missing values (NA) should be handled prior to binning.
#' @param target An integer vector of binary outcomes (0/1) corresponding to
#'   each observation in \code{feature}. Must have the same length as \code{feature}.
#' @param min_bins Integer. The minimum number of bins to produce. Must be \eqn{\ge} 2.
#'   Defaults to 3.
#' @param max_bins Integer. The maximum number of bins to produce. Must be \eqn{\ge}
#'   \code{min_bins}. Defaults to 5.
#' @param bin_cutoff Numeric. The minimum fraction of total observations required
#'   for a bin to be considered valid. Bins smaller than this threshold are merged.
#'   Value must be in (0, 1). Defaults to 0.05.
#' @param max_n_prebins Integer. The maximum number of initial candidate cut points
#'   to generate during the density estimation phase. Defaults to 20.
#' @param polynomial_degree Integer. The degree of the local polynomial used for
#'   density estimation (note: currently approximated via KDE). Defaults to 3.
#' @param enforce_monotonic Logical. If \code{TRUE}, the algorithm forces the
#'   Weight of Evidence (WoE) trend to be strictly monotonic. Defaults to \code{TRUE}.
#' @param convergence_threshold Numeric. The threshold for determining convergence
#'   during the iterative merging process. Defaults to 1e-6.
#' @param max_iterations Integer. Safety limit for the maximum number of merging
#'   iterations. Defaults to 1000.
#'
#' @return A list containing the binning results:
#'   \itemize{
#'     \item \code{id}: Integer vector of bin identifiers.
#'     \item \code{bin}: Character vector of bin labels in interval notation.
#'     \item \code{woe}: Numeric vector of Weight of Evidence for each bin.
#'     \item \code{iv}: Numeric vector of Information Value contribution per bin.
#'     \item \code{count}: Integer vector of total observations per bin.
#'     \item \code{count_pos}: Integer vector of positive cases.
#'     \item \code{count_neg}: Integer vector of negative cases.
#'     \item \code{event_rate}: Numeric vector of the target event rate in each bin.
#'     \item \code{centroids}: Numeric vector of the geometric centroids of the final bins.
#'     \item \code{cutpoints}: Numeric vector of upper boundaries (excluding Inf).
#'     \item \code{total_iv}: The total Information Value of the binned variable.
#'     \item \code{monotonicity}: Character string indicating the final WoE trend ("increasing", "decreasing", or "none").
#'   }
#'
#' @details
#' The \strong{Local Polynomial Density Binning (LPDB)} algorithm is a two-stage process:
#'
#' \enumerate{
#'   \item \strong{Density-Based Initialization:}
#'   \itemize{
#'     \item Estimates the probability density function \eqn{f(x)} of the \code{feature}
#'           using Kernel Density Estimation (KDE), which approximates local polynomial regression.
#'     \item Identifies \emph{critical points} on the density curve, such as local minima
#'           and inflection points. These points often correspond to natural boundaries
#'           between clusters or modes in the data.
#'     \item Uses these critical points as initial candidate cut points to form pre-bins.
#'   }
#'   \item \strong{Supervised Refinement:}
#'   \itemize{
#'     \item Calculates WoE and IV for each pre-bin.
#'     \item Enforces monotonicity by merging bins that violate the trend (determined
#'           by the correlation between bin centroids and WoE values).
#'     \item Merges bins with frequencies below \code{bin_cutoff}.
#'     \item Iteratively merges bins to meet the \code{max_bins} constraint, choosing
#'           merges that minimize the loss of total Information Value.
#'   }
#' }
#'
#' This method is particularly powerful for complex, multi-modal distributions where
#' standard quantile or equal-width binning might obscure important structural breaks.
#'
#' @seealso \code{\link{ob_numerical_kmb}}, \code{\link{ob_numerical_jedi}}
#'
#' @examples
#' # Example: Binning a tri-modal distribution
#' set.seed(123)
#' # Feature with three distinct clusters
#' feature <- c(rnorm(300, mean = -3), rnorm(400, mean = 0), rnorm(300, mean = 3))
#' # Target depends on these clusters
#' target <- rbinom(1000, 1, plogis(feature))
#'
#' result <- ob_numerical_lpdb(feature, target,
#'   min_bins = 3,
#'   max_bins = 5
#' )
#'
#' print(result$bin) # Should ideally find cuts near -1.5 and 1.5
#' print(result$monotonicity)
#'
#' @export
ob_numerical_lpdb <- function(feature, target, min_bins = 3, max_bins = 5,
                              bin_cutoff = 0.05, max_n_prebins = 20,
                              polynomial_degree = 3, enforce_monotonic = TRUE,
                              convergence_threshold = 1e-6, max_iterations = 1000) {
  # Type Validation
  if (!is.numeric(feature)) {
    warning("Feature converted to numeric for processing.")
    feature <- as.numeric(feature)
  }

  if (!is.integer(target)) {
    target <- as.integer(target)
  }

  # Dimension Check
  if (length(feature) != length(target)) {
    stop("Length of 'feature' and 'target' must match.")
  }

  # NA Check
  if (any(is.na(feature))) {
    warning("Feature contains NA values. These will be excluded during density estimation.")
  }

  # .Call Interface
  .Call("_OptimalBinningWoE_optimal_binning_numerical_lpdb",
    target,
    feature,
    as.integer(min_bins),
    as.integer(max_bins),
    as.numeric(bin_cutoff),
    as.integer(max_n_prebins),
    as.integer(polynomial_degree),
    as.logical(enforce_monotonic),
    as.numeric(convergence_threshold),
    as.integer(max_iterations),
    PACKAGE = "OptimalBinningWoE"
  )
}
