#' Calculate Agronomic Efficiency (AE)
#' @description
#' The contribution of fertilizer N towards yield, compared to a non-fertilized control.
#' Formula: AE = (YieldF - Yield0) / FertN
#' @param YieldF A numeric vector for yield in fertilized conditions.
#' @param Yield0 A numeric vector of non-fertilized control yield values.
#' @param FertN A numeric value or vector for fertilizer N input.
#' @return A numeric vector of AE values.
#' @examples
#' YieldF <- c(10, 12, 15)
#' Yield0 <- c(5, 5, 5)
#' FertN <- 50
#' AE(YieldF, Yield0, FertN)
#' @export
AE <- function(YieldF, Yield0, FertN) {
  # Error handling for vectors
  if(length(YieldF) != length(Yield0) && length(Yield0) != 1) {
    warning("YieldF and Yield0 have different lengths. Ensure data is paired correctly.")
  }
  
  # Check for zero division
  if (any(FertN == 0)) {
    stop("FertN cannot be zero.")
  }
  
  # Calculation
  AE_val <- (YieldF - Yield0) / FertN
  return(AE_val)
}