#' Posterior mixture weight calculation
#'
#' @description
#' Computes posterior updated mixture weights for a two-component normal–normal model
#' using the standard logit-additive update. The *prior* mixing weight is either a
#' fixed weight \eqn{w \in (0,1)} or a dynamic mixing weight as for eNAP prior:
#' \eqn{Z = n_{eff}^{-1/4}|y_{\mathrm{dir}} - y_{\mathrm{ind}}| / s_{\mathrm{link}}}:
#' where, \eqn{n_{eff}= s_{\mathrm{link}}^{-1}}
#'
#' \eqn{w_(Z) = 1/exp(a + b \log (Z+1)), \quad a<0, \; b > 0.}
#'
#'
#' @param w Scalar. If `w > 1`, use the ADAPTIVE branch (logistic prior on log-Z).
#'   If `0 < w < 1`, use a fixed prior weight equal to `w`.
#' @param a,b Parameters used in the elastic function for dynamic mixing weight. Must satisfy `a < 0` and  `b > 0`.
#' @param s_EC2,s_EC1,s_C2C1 Sampling variances for direct evidence (E vs. C2 trial), and edges of indirect evidence (E vs. C1 trial and C2 vs. C1 trial).
#' @param mu0,tau0 Mean and variance for the vague component.
#' @param y_EC2,y_EC1,y_C2C1 Estimated log-HR for E vs. C2 tria, E vs. C1 trial, C2 vs. C1 trial, respectively
#' @param eps Numeric scalar used for small-value clipping (default `1e-12`).
#'
#' @return A numeric vector of posterior weights in `(0,1)` reflecting realized borrowing fraction of the informative component.
#'
#' @details
#' - **Fixed prior mixing weight (Robust NMAP Prior):** requires `0 < w < 1`.
#' - **Adaptive branch (Adaptive NMAP Prior):** triggered by `w > 1`, requires `a<0` and `b > 0`.
#'   This corresponds to a decreasing prior weight as the inconsistency grows.
#' - All variance/SD arguments may be given as scalars or vectors; scalars are recycled.
#'
#' @examples
#' y_EC2  <- -0.5; y_EC1  <- -0.8; y_C2C1 <- -0.3
#' s_EC2 <- 0.2; s_EC1 <- 0.18; s_C2C1 <- 0.18
#' 
#' # Fixed mixing weight 0.5
#' post_w(w = 0.5, a = NA, b = NA,s_EC2,s_EC1,s_C2C1,
#'        y_EC2,y_EC1,y_C2C1)
#'
#' # Dynamic weight with elastic function of (a=-4.6, b=3):
#' post_w(w = 2, a = -2.5, b = 10,s_EC2,s_EC1,s_C2C1,
#'        y_EC2,y_EC1,y_C2C1)
#'
#' @export
#' @importFrom stats plogis qlogis
post_w <- function(w, a, b, s_EC2,s_EC1,s_C2C1, y_EC2,y_EC1=-0.5,y_C2C1=-0.5,tau0=1000, mu0=0,eps = 1e-12) {
  if (length(w) != 1 || !is.finite(w)) stop("`w` must be a single finite scalar.")
  
  if (any(!is.finite(s_EC2))  || any(s_EC2  <= 0)) stop("`s_EC2` must be positive and finite.")
  if (any(!is.finite(s_EC1))  || any(s_EC1  <= 0)) stop("`s_EC1` must be positive and finite.")
  if (any(!is.finite(s_C2C1))  || any(s_C2C1  <= 0)) stop("`s_C2C1` must be positive and finite.")
  if (any(!is.finite(tau0)) || any(tau0 <= 0)) stop("`tau0` must be positive and finite.")
  
  s_tot<-sum(s_EC2,s_EC1,s_C2C1)
  n_eff<-1/s_tot
  Z<-pmax(n_eff^(-0.25)*abs(y_EC2-(y_EC1-y_C2C1))/sqrt(s_tot),eps)
  
  # Prior mixing weight
  if (w > 1) {
    if (!is.finite(a) || !is.finite(b) || b <= 0 || a > 0)
      stop("Invalid tuning parameters (a, b) for eNAP: require a<0 and b>0.")
    w_prior <- 1 / (1 + exp(a + b * log(Z+1)))
  } else {
    if (!(w > 0 && w < 1)) stop("Fixed `w` must be in (0,1).")
    w_prior <- w
  }
  
  # Helper function
  logit <- function(p) log(p/(1 - p))
  ilogit <- function(x) 1/(1 + exp(-x))
  
  # Derive
  V0=s_EC2 + tau0
  v1=1/(1/tau0+1/s_EC1)
  v2=1/(1/tau0+1/s_C2C1)
  V_ind  <- v1+v2
  mu_ind<-v1*(mu0/tau0+y_EC1/s_EC1)-(v2*(mu0/tau0+y_C2C1/s_C2C1))
  logit_w_post<-logit(w_prior)+0.5*log(V0/(s_EC2+V_ind)) - 0.5*( (y_EC2-mu_ind)^2/(s_EC2+V_ind) - (y_EC2-mu0)^2/V0)  

  ilogit(logit_w_post)
}
