#' Conduct posterior inference with NAP-based priors
#'
#' @description
#' Draw posterior via MCMC (JAGS) with derived NAP priors from NAP_prior function
#' both setting (one external trial/multiple external trials) and NAP method (NAP/mNAP/eNAP) 
#' will be determined by the provided NAP_prior object. If using eNAP, make sure the 
#' tuning parameter used to derive NAP_prior are calibrated by tune_param_eNAP function.
#' @param NAP_prior An object returned by \code{\link{NAP_prior}()} containing the full
#'   the NAP prior (and if eNAP without assumed direct effects, calibrated tuning parameters (a,b)) 
#' @param y_EC2 Numeric scalar. Direct estimate \eqn{y_{EC2}} (e.g., log-HR) for
#'   \eqn{E} vs \eqn{C2}.
#' @param s_EC2 Positive numeric scalar. Sampling variance \eqn{s^2_{EC2}} for \eqn{y_{EC2}}.
#' @param iter Total MCMC iterations per chain (default \code{2000}).
#' @param chains Number of MCMC chains (default \code{4}).
#' @param model Either a length-1 character string containing JAGS model code or
#'   a path to a JAGS model file. If \code{NULL}, a package default will be used.
#'  
#' @return A list of class \code{"NAP_posterior_result"} with elements:
#' \itemize{
#'   \item \code{posterior_sum}: data frame with posterior summaries for \eqn{\theta_{E,C2}}
#'         (mean, sd, 95\% interval, \code{prob_E_better}), and the prior/posterior
#'         weights (\code{prior_weight}, \code{post_weight}).
#'   \item \code{enap_prior}: For eNAP only: data frame describing the eNAP prior with calculated data-dependent weight:
#'         columns for \code{NAP (Informative)} and \code{Vague}, rows for
#'         \code{Mixing Weight}, \code{Mean}, \code{Variance}, and
#'         \code{ESS (events)} if available.
#'   \item \code{jags_fit}: the \code{R2jags} fit object.
#' }#'
#' @examples
#' \donttest{
#' # Create a NAP_prior object
#' my_naprior <- NAP_prior(
#'   weight_mtd = "fixed", w = 0.50,       # fixed mixture weight
#'   y_EC1  = -0.36, s_EC1  = 0.16^2,
#'   y_C2C1 = -0.30, s_C2C1 = 0.14^2,      # single external trial
#'   tau0   = 1000
#' )
#'
#' # Calculate posterior
#' out <- NAP_posterior(
#'   NAP_prior = my_naprior,
#'   y_EC2 = -0.20, s_EC2 = 0.18^2,
#'   iter = 1000, chains = 2
#' )
#' out$posterior_sum
#' out$enap_prior
#' }
#'
#' @export
#' @importFrom R2jags jags 

NAP_posterior <- function(NAP_prior = NULL,
                        y_EC2,  s_EC2,
                        iter   = 2000,
                        chains = 4,
                        model  = NULL) {
  

  ## ---- validate shared
  req_num <- function(x, nm) if (!is.numeric(x) || length(x)!=1L || !is.finite(x)) stop("`", nm, "` must be a finite scalar.")
  req_pos <- function(x, nm) { req_num(x, nm); if (x <= 0) stop("`", nm, "` must be > 0.") }
  
  req_num(y_EC2, "y_EC2");   req_pos(s_EC2, "s_EC2")
  if (!inherits(NAP_prior,"NAP_prior")) stop("`NAP_prior` must be output object from NAP_prior function.")
  
  d<-NAP_prior$details
  weight_med<-NAP_prior$weight_mode
  is_RE <- NAP_prior$mode=="Multiple external trials"
  
  ## ---- choose model default (if not supplied)
 
  if (is.null(model)) {
    model <- .get_pkg_model(is_RE)   # <- returns the JAGS model TEXT
  }

  y_C2C1   <- d$y_inputs$y_C2C1_original
  s_C2C1  <-  d$s_inputs$s_C2C1_original
  y_EC1 <- d$y_inputs$y_EC1
  s_EC1 <- d$s_inputs$s_EC1
  
  
  ## ---- prior weight
  if (identical(weight_med, "fixed")) {
    weight <- d$w_inform
  } else  {
    # FE adaptive
    y_C2C1   <- if(is_RE) d$y_inputs$y_C2C1_RE else y_C2C1
    s_C2C1  <- if(is_RE) d$s_inputs$s_C2C1_RE else s_C2C1
    a<-d$a
    b<-d$b
    n_eff<-1/sum(s_EC2,s_C2C1,s_EC1)
    Z        <- n_eff^(-0.25)*abs(y_EC2 - (y_EC1 - y_C2C1))/sqrt(sum(s_EC2,s_C2C1,s_EC1))
    weight   <- 1 / (1 + exp(a + b * log(Z+1)))
  } 
  ## ---- resolve model to a file path for JAGS
  model_path <- .model_path_from(model)
  
  ## ---- build JAGS data & run
  if (!is_RE) {
    # FE
    data_mix <- list(
      y_ind  = c(y_EC1, y_C2C1),
      se_ind = c(sqrt(s_EC1), sqrt(s_C2C1)),
      y_dir  = y_EC2, se_dir = sqrt(s_EC2),
      w      = weight
    )
    par_save <- c("theta_EC2", "z")
  } else {
    # RE
    data_mix <- list(
      y_dir = y_EC2, se_dir = sqrt(s_EC2),
      y_EC1 = y_EC1, se_EC1 = sqrt(s_EC1),
      m_h = length(y_C2C1),
      y_ext = as.numeric(y_C2C1),
      se_ext = as.numeric(sqrt(s_C2C1)),
      w = weight
    )
    par_save <- c("theta_EC2","z","sigma","d_C2C1")
  }
  
  fit <- R2jags::jags(
    data        = data_mix,
    parameters.to.save = par_save,
    model.file  = model_path,
    n.chains    = chains,
    n.iter      = iter,
    n.burnin    = min(floor(iter/2),max(1000, floor(iter/2)))
  )
  
  sims   <- fit$BUGSoutput$sims.list
  theta  <- sims$theta_EC2
  zbar   <- mean(sims$z)
  
  ## ---- tidy output
  out <- data.frame(
    post_mean      = mean(theta),
    post_sd        = stats::sd(theta),
    low95          = stats::quantile(theta, 0.025),
    hi95           = stats::quantile(theta, 0.975),
    prob_E_better  = mean(theta < 0),
    prior_weight   = weight,
    post_weight    = zbar,
    sigma_hat      = if (is_RE & !is.null(sims$sigma))  mean(sims$sigma)  else NA_real_,
    row.names = NULL,
    check.names = FALSE
  )
  
  jags_fit    = fit
  
  if(identical(weight_med, "adaptive")){
    enap_prior  = data.frame(
      `NAP (Informative)` = c(weight, d$mu_ind, d$V_ind,d$ESS_informative),
      Vague       = c( 1-weight,  d$mu0,    d$tau0, NA_real_),
      row.names   = c("Mixing Weight", "Mean", "Variance","ESS (events)"),
      check.names = FALSE
    )
    res <- list(
      posterior_sum   = out,
      enap_prior  = enap_prior,
      jags_fit    = fit
    )
  }else{
    res <- list(
      posterior_sum   = out,
      jags_fit    = fit
    )
  }
  
  class(res) <- c("NAP_posterior_result", class(res))
  return(res)
}
#' @export
print.NAP_posterior_result <- function(x, ...) {
  cat("\nPosterior summary (theta_EC2):\n")
  print(x$posterior_sum, row.names = FALSE)
  if (!is.null(x$enap_prior)) {
    cat("\neNAP prior with derived data-dependent weight:\n")
    print(x$enap_prior)
  }
  invisible(x)
}

#' Turn a single `model` input (object with JAGS text OR path) into a file path
#' @keywords internal
.model_path_from <- function(model) {
  if (is.character(model) && length(model) == 1L) {
    if (file.exists(model)) {
      return(normalizePath(model, winslash = "/", mustWork = TRUE))
    }
    out <- file.path(tempdir(), "model_RE_from_text.txt")
    writeLines(model, out, useBytes = TRUE)
    return(out)
  }
  stop("`model` must be either (i) a length-1 character string containing JAGS code ",
       "(e.g., your packaged `jags_model_RE` object), or (ii) a path to a .txt file.")
}
#' @keywords internal
.get_pkg_model <- function(is_RE, pkg = "NAPrior") {
  ns <- asNamespace(pkg)
  nm <- if (is_RE) "jags_model_RE" else "jags_model_FE"
  txt <- get0(nm, envir = ns, inherits = FALSE)
  if (!is.character(txt) || length(txt) != 1L || !nzchar(txt)) {
    stop(sprintf("Internal model `%s` not found or not a length-1 character string.", nm), call. = FALSE)
  }
  txt
}