#' @title Introduces Missing Not at Random (MNAR) Values into Data Sets
#'
#' @description
#' This function introduces missing values under the Missing Not at Random (MNAR) mechanism into previously generated data sets (e.g., those produced by sim.skewed() or sim.normal()).
#' Under the MNAR mechanism, the probability of missingness depends on the observed values of the variable itself.
#' Specifically, the target variable is first sorted in decreasing order. Based on the specified percentage of missingness, 90 percents of missing values are assigned randomly among the highest values, while the remaining 10 percents are assigned randomly among the rest of the sample.
#' For example, with a sample size of 300 and a target of 20 percents missingness (60 cases), the variable is sorted in descending order.
#' Missing values are then introduced in 54 cases (90 percents of 60) from the top of the distribution, while the remaining 6 cases (10 percents of 60) are randomly chosen from the lower 240 observations.
#' The missing values are represented by NA in the output files. New data sets containing missing values are saved as separate files, preserving the originals.
#' Additionally, a file named "MNAR_List.dat" is created, which contains the names of all data sets with MNAR missingness.
#'
#' @author Fatih Orcan
#' @importFrom utils read.table write.table
#' @param misg A numeric vector of 0s and 1s specifying which items will contain missing values.
#' A value of 0 indicates the item will not include missingness, while 1 indicates missing values will be introduced.
#' If omitted, all items are treated as eligible for missingness.
#' @param perct The percentage of missingness to be applied (default = 10 percents).
#' @param dataList The file name containing the list of previously generated data sets (e.g., "Data_List.dat"), either created by this package or by external software.
#' @param f.loc The directory path where both the original data sets and the "dataList" file are located.
#' @export
#' @examples
#'
#' # Step 1: Generate data sets
#'
#' fc<-fcors.value(nf=3, cors=c(1,.5,.6,.5,1,.4,.6,.4,1))
#' fl<-loading.value(nf=3, fl.loads=c(.5,.5,.5,0,0,0,0,0,0,0,0,.6,.6,.6,0,0,0,0,0,0,0,0,.4,.4))
#' floc<-tempdir()
#' sim.normal(nd=10, ss=100, fcors=fc, loading<-fl,  f.loc=floc)
#'
#'  # Step 2: Introduce MNAR missing values
#'
#' mis.items<-c(1,1,1,0,0,0,0,0)
#' dl<-"Data_List.dat"  # must be located in the working directory
#' MNAR.data(misg = mis.items, perct = 20, dataList = dl, f.loc=floc)

MNAR.data<-function(misg=NULL, perct=10, dataList="Data_List.dat", f.loc){

  data.names<-as.matrix(read.table(paste(f.loc, "/", dataList,sep=""), header = FALSE))
  misg.names<-data.names
  veri<-read.table(paste(f.loc,"/", data.names[1,],sep=""))
  colnames(veri)<-c("ID", paste("x",seq(1:(dim(veri)[2]-1)),sep=""))

  nd<-dim(data.names)[1]
  for(i in 1:nd){
    veri<-read.table(paste(f.loc,"/", data.names[i,],sep=""))

    misg.names[i,]<-c(paste("MNAR_",i,".dat", sep=""))
    nitem<-dim(veri)[2]-1
    ss<-dim(veri)[1]

    MNAR.data<-veri
    for(j in 1:nitem){
      if(misg[j]==0){
        MNAR.data[,j+1]<-veri[,j+1]}
      else if(misg[j]==1){
        mis.ss<-(perct/100)*ss

        MNAR.data[sample(order(veri[,j+1],decreasing = T)[1:mis.ss],size=mis.ss*.9),j+1]<-NA
        MNAR.data[sample(order(veri[,j+1],decreasing = T)[(mis.ss+1):ss],size=mis.ss*.1),j+1]<-NA
        message(paste("MNAR_",i,".dat was completed", sep=""))
        }

      else {stop("Please use only 0s or 1s to indicated missingness")}}
    write.table(MNAR.data, file= paste(f.loc, "/MNAR_",i,".dat", sep=""), sep = "\t",
                col.names = FALSE, row.names = FALSE,quote = FALSE)
  }
  write.table(misg.names,file=paste(f.loc,"/MNAR_List.dat", sep = ""),
              col.names = FALSE, row.names = FALSE, quote = FALSE)
  message("Done!...")
}
