\name{HMMVarianceMatrix}
\alias{HMMVarianceMatrix}
\title{Variance-Covariance Matrix for Hidden Markov Models}
\description{
Computes an approximate variance-covariance matrix of the parameter estimates
from a fitted Hidden Markov Model (HMM) using a numerical Hessian of the log-likelihood.
This function reformulates the HMM as a special case of a Hidden Semi-Markov Model (HSMM)
with geometric dwell-time distributions, and calls \code{\link{HSMMVarianceMatrix}}.
}
\usage{
HMMVarianceMatrix(x, HMM, obsdist, h = 1e-5, method = "central",
                  verbose = TRUE)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}
\item{HMM}{A fitted HMM object (typically the output from \code{\link{findmleHMM}}),
containing estimated parameters \code{Pi}, \code{delta}, and observation parameters.}
\item{obsdist}{Character string. Observation distribution. Supported distributions:
"norm", "pois", "weibull", "nbinom", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}
\item{h}{Numeric. Step size for finite-difference approximation of the Hessian.
Default is \code{1e-5}.}
\item{method}{Character string. Method for numerical Hessian computation.
Options are \code{"central"} (default, central differences) or \code{"forward"} (forward differences).}
\item{verbose}{Logical, if \code{TRUE} (default), progress messages are printed to the console.
Set to \code{FALSE} to suppress informational output.}
}
\value{
A variance-covariance matrix of the HMM parameter estimates, computed as the
inverse of the observed information matrix (negative Hessian of the log-likelihood),
adjusted to be positive-definite.
}
\details{
This function is a wrapper around \code{\link{HSMMVarianceMatrix}}, treating an HMM
as a special case of an HSMM with geometric dwell-time distributions. The Hessian is
computed numerically, inverted, and adjusted for positive-definiteness to provide
a variance-covariance matrix for the estimated parameters.
}
\examples{
# Example with 2-state Normal HMM
J <- 2
Pi <- matrix(c(0.9, 0.1,
               0.2, 0.8), nrow = 2, byrow = TRUE)
delta <- c(0.5, 0.5)
obspar <- list(mean = c(0, 3), sd = c(1, 1))
# Simulate data
sim_data <- generateHMM(n = 100, J = J, obsdist = "norm",
                        obspar = obspar, Pi = Pi, delta = delta)
# Fit HMM
HMM_fit <- findmleHMM(x = sim_data$x, J = J,
                      obsdist = "norm", obspar = obspar,
                      Pi = Pi, delta = delta)
# Compute variance-covariance matrix
vcov_matrix <- HMMVarianceMatrix(x = sim_data$x, HMM = HMM_fit,
                                 obsdist = "norm")
vcov_matrix
}
\seealso{
\code{\link{generateHMM}}, for simulating HMM data.
\code{\link{HSMMVarianceMatrix}}, for the general hidden semi-Markov case.
\code{\link{confintHMM}}, for bootstrap confidence intervals of HMM parameters. \cr
\code{\link{findmleHMM}}, for fitting Hidden Markov Models.
}
\author{Aimee Cody}

