exceedanceplotHSMMgev <- function(x, HSMM, threshold, dwelldist, M = NA,
                                  varcov = NULL, B = 10000, level = 0.95,
                                  time_structure = NULL, shift = FALSE,
                                  plot_title = "Exceedance Probabilities Over Time",
                                  save_plot = FALSE, filename = NULL,
                                  width = 12, height = 8, dpi = 300,
                                  verbose = TRUE) {

  nk <- length(HSMM$delta)

  # Extract GEV parameters from HSMM structure with log transformation for scale
  gev.param <- c(
    HSMM$observationparameters$loc,
    log(HSMM$observationparameters$scale),
    HSMM$observationparameters$shape
  )

  # Compute or use provided variance-covariance matrix (observation parameters only)
  if (is.null(varcov)) {
    if (verbose) message("Computing numerical variance-covariance matrix...")
    gev.varc <- HSMMVarianceMatrix(
      x = x, HSMM = HSMM, M = M, obsdist = "gev", dwelldist = dwelldist, verbose=verbose
    )
    # Extract only observation parameter portion of variance-covariance matrix
    gev.varc <- gev.varc[1:(nk*length(HSMM$observationparameters)), 1:(nk*length(HSMM$observationparameters))]
  } else {
    gev.varc <- varcov
  }

  # Decode most likely state sequence using HSMM global decoding
  vky <- globaldecodeHSMM(
    x = x, M = M, HSMM=HSMM, obsdist = "gev", dwelldist = dwelldist
  )

  if (verbose) message("Computing exceedance probabilities by state...")
  # Initialize storage for exceedance probabilities: [estimate, lower CI, upper CI] x states
  pe.wk.state <- matrix(0, 3, nk)

  # Compute exceedance probabilities for each state
  for (jj in 1:nk) {
    if (verbose && (jj == 1 || jj %% max(1, nk %/% 5) == 0)) {
      message(sprintf("Processing state %d / %d", jj, nk))
    }

    # Compute P(X <= threshold) for current state using GEV distribution
    pe.wk.state[1, jj] <- pevd(
      threshold,
      loc   = HSMM$observationparameters$loc[jj],
      scale = HSMM$observationparameters$scale[jj],
      shape = HSMM$observationparameters$shape[jj],
      type = "GEV",
      lower.tail = TRUE
    )

    # Bootstrap confidence intervals
    pe.wk.bt <- rep(0, B)
    for (b in 1:B) {
      if (verbose && b %% max(1, B %/% 10) == 0) {
        message(sprintf("State %d: Bootstrap sample %d / %d", jj, b, B))
      }

      # Generate bootstrap parameter sample from multivariate normal
      gev.param.bt <- rmnorm(n = 1, mean = gev.param, varcov = gev.varc)
      loc.bt   <- gev.param.bt[1:nk]
      scale.bt <- exp(gev.param.bt[(nk+1):(2*nk)])
      shape.bt <- gev.param.bt[(2*nk+1):(3*nk)]

      # Compute exceedance probability for bootstrap sample
      pe.wk.bt[b] <- pevd(
        threshold,
        loc   = loc.bt[jj],
        scale = scale.bt[jj],
        shape = shape.bt[jj],
        type = "GEV",
        lower.tail = TRUE
      )
    }

    # Store confidence interval bounds
    alpha <- (1 - level) / 2
    pe.wk.state[2:3, jj] <- quantile(pe.wk.bt, probs = c(alpha, 1 - alpha))
  }

  # Map state-specific exceedance probabilities to observation time series
  pe.wk.all <- matrix(0, 3, length(x))
  for (k in seq_along(x)) {
    pe.wk.all[, k] <- pe.wk.state[, vky[k]]
  }

  # Set default time structure if not provided
  if (is.null(time_structure)) {
    time_structure <- list(
      unit = "year",
      observations_per_unit = 52,
      start_point = 1
    )
  }
  time_unit <- time_structure$observations_per_unit

  # Aggregate weekly probabilities to annual exceedance probabilities
  n_periods <- floor(length(x) / time_unit)
  pe.yr.all <- matrix(0, 3, n_periods)

  for (j in 1:n_periods) {
    # Compute annual non-exceedance probability as product of weekly probabilities
    tem <- 1
    for (i in ((j - 1) * time_unit + 1):(j * time_unit)) {
      tem <- tem * pe.wk.state[, vky[i]]
    }
    # Convert to annual exceedance probability: P(exceed in year) = 1 - P(not exceed all weeks)
    pe.yr.all[, j] <- 1 - tem
  }

  # Create enhanced time information for plotting
  time_info <- createEnhancedTimeInfo(n_periods, time_structure)

  # Set up plot output
  if (save_plot) {
    if (is.null(filename)) {
      stop("filename must be specified when save_plot = TRUE")
    }
    png(filename, width = width, height = height, units = "in", res = dpi)
    on.exit(dev.off(), add = TRUE)
  }

  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar), add = TRUE)

  par(mar = c(4.5, 4.5, 3, 1))

  # Create main exceedance probability plot with confidence intervals
  plot(time_info$labels, pe.yr.all[1, ], type = "l",
       ylim = c(min(pe.yr.all), max(pe.yr.all)),
       xlab = time_info$x_label,
       ylab = paste("Prob exceeding", threshold),
       cex.lab = 1.5, cex.axis = 1.5,
       main = plot_title, cex.main = 1.2)

  # Add confidence interval lines (dashed)
  lines(time_info$labels, pe.yr.all[2, ], lty = "dashed", col = "black")
  lines(time_info$labels, pe.yr.all[3, ], lty = "dashed", col = "black")

  if (save_plot && verbose) {
    message("Plot saved to: ", filename)
  }

  # Return computed results for further analysis
  invisible(list(
    exceedance_probs = pe.yr.all,
    time_info = time_info,
    decoded_states = vky,
    time_unit = time_structure$observations_per_unit,
    threshold = threshold
  ))
}

createEnhancedTimeInfo <- function(n_periods, time_structure) {
  # Validate required fields based on unit type
  required_fields <- c("unit", "observations_per_unit")
  if (time_structure$unit == "custom") {
    required_fields <- c("conversion_factor", "unit_name")
  }
  missing_fields <- setdiff(required_fields, names(time_structure))
  if (length(missing_fields) > 0) {
    stop("Missing required fields in time_structure: ", paste(missing_fields, collapse = ", "))
  }

  # Set conversion parameters and axis labels
  if (time_structure$unit == "custom") {
    conversion_factor <- time_structure$conversion_factor
    unit_name <- time_structure$unit_name
    x_label <- paste("Time (", unit_name, "s)", sep = "")
  } else {
    conversion_factor <- time_structure$observations_per_unit
    unit_name <- time_structure$unit

    # Create appropriate axis labels for common time units
    if (unit_name == "year") {
      x_label <- "Time (years)"
    } else if (unit_name == "day") {
      x_label <- "Time (days)"
    } else if (unit_name == "hour") {
      x_label <- "Time (hours)"
    } else if (unit_name == "week") {
      x_label <- "Time (weeks)"
    } else if (unit_name == "month") {
      x_label <- "Time (months)"
    } else {
      x_label <- paste("Time (", unit_name, "s)", sep = "")
    }
  }

  duration <- n_periods

  # Determine time range from start/end points
  has_start <- !is.null(time_structure$start_point)
  has_end <- !is.null(time_structure$end_point)

  if (has_start && has_end) {
    start_point <- time_structure$start_point
    end_point <- time_structure$end_point
    time_labels <- seq(start_point, end_point, length.out = n_periods)
  } else if (has_start && !has_end) {
    start_point <- time_structure$start_point
    time_labels <- seq(start_point, start_point + n_periods - 1, by = 1)
    end_point <- time_labels[length(time_labels)]
  } else if (!has_start && has_end) {
    end_point <- time_structure$end_point
    start_point <- end_point - n_periods + 1
    time_labels <- seq(start_point, end_point, by = 1)
  } else {
    # Default to starting at 1
    start_point <- 1
    time_labels <- seq(start_point, start_point + n_periods - 1, by = 1)
    end_point <- time_labels[length(time_labels)]
  }

  return(list(
    labels = time_labels,
    unit = ifelse(time_structure$unit == "custom", "custom", time_structure$unit),
    conversion_factor = conversion_factor,
    unit_name = unit_name,
    x_label = x_label,
    duration = duration,
    start_point = start_point,
    end_point = end_point
  ))
}

