% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Functions.R
\name{generate_data}
\alias{generate_data}
\title{Generate multivariate time series from the proposed model}
\usage{
generate_data(
  mu0,
  deltas,
  Sig_eta,
  Sig_nu,
  Phi,
  Sig_e1,
  errortype,
  df = 10,
  number_cps,
  lengthofeachpart
)
}
\arguments{
\item{mu0}{Numeric vector of length \eqn{p}. The initial mean vector
\eqn{\boldsymbol{\mu}_0}.}

\item{deltas}{A list of numeric vectors, each representing the jump
magnitude \eqn{\boldsymbol{\delta}_t} at a change point.}

\item{Sig_eta}{Numeric \eqn{p \times p} covariance matrix
\eqn{\Sigma_{\boldsymbol{\eta}}} of the random walk innovation.}

\item{Sig_nu}{Numeric \eqn{p \times p} covariance matrix
\eqn{\Sigma_{\boldsymbol{\nu}}} of the VAR(1) innovation.}

\item{Phi}{Numeric \eqn{p \times p} autoregressive coefficient matrix
\eqn{\Phi}.}

\item{Sig_e1}{Numeric \eqn{p \times p} initial-state covariance matrix
of the VAR(1) process.}

\item{errortype}{Character; either "n" (Gaussian) or "t" (Student's t)
specifying the distribution of the innovations.}

\item{df}{Degrees of freedom for the t-distribution (used only when
`errortype = "t"`). Default is 10.}

\item{number_cps}{Integer; number of change points (\eqn{m}).}

\item{lengthofeachpart}{Integer; number of observations between
consecutive change points (\eqn{\tau_{k+1} - \tau_k}).}
}
\value{
A numeric matrix of dimension \eqn{n \times p}, with
\eqn{n = (number\_cps+1)\,lengthofeachpart}, containing the simulated
observations \eqn{\{\mathbf{y}_t\}_{t=1}^n}.
}
\description{
Simulates a multivariate time series following the proposed model structure,
where the mean component evolves as a random walk with abrupt shifts,
overlaid by a stationary VAR(1) process to account for temporal and
cross-sectional correlations.

Specifically, at each time point \eqn{t = 1, \ldots, n}, the data are
generated as
\deqn{\mathbf{y}_t = \boldsymbol{\mu}_t + \boldsymbol{\epsilon}_t,}
where, for \eqn{t = 2, \ldots, n},
\deqn{\boldsymbol{\mu}_t = \boldsymbol{\mu}_{t-1} + \boldsymbol{\eta}_t + \boldsymbol{\delta}_t,}
and
\deqn{\boldsymbol{\epsilon}_t = \Phi \boldsymbol{\epsilon}_{t-1} + \boldsymbol{\nu}_t.}

Here, \eqn{\boldsymbol{\eta}_t} denotes the random walk innovation with
covariance \eqn{\Sigma_{\boldsymbol{\eta}}}, and
\eqn{\boldsymbol{\nu}_t} is the VAR(1) innovation with covariance
\eqn{\Sigma_{\boldsymbol{\nu}}}. The vector
\eqn{\boldsymbol{\delta}_t} is nonzero only at change points.
}
\details{
The total length of the time series is given by
\eqn{n = (number\_cps + 1) \times lengthofeachpart}, so that the specified
change points partition the data into equally sized segments. When
\eqn{\Sigma_{\boldsymbol{\eta}} = 0}, the model reduces to a piecewise
constant mean process with no random walk component. When \eqn{\Phi = 0},
the process reduces to a random walk model without vector autoregressive
dependence. If both \eqn{\Sigma_{\boldsymbol{\eta}} = 0} and \eqn{\Phi = 0},
the model simplifies to the classical piecewise constant setting commonly
used in multiple change point analysis. The two innovation components are
generated independently.

The innovations \eqn{\boldsymbol{\eta}_t} and \eqn{\boldsymbol{\nu}_t} are
drawn either from a multivariate normal distribution (when
\code{errortype = "n"}) using \code{\link[MASS]{mvrnorm}}, or from a
multivariate Student's t distribution (when \code{errortype = "t"}) using
\code{\link[SimDesign]{rmvt}}.
}
\examples{
set.seed(123)
p <- 3
mu0 <- rep(0, p)
deltas <- list(c(3, 0, -3), c(-2, 4, 0))
Sig_eta <- diag(0.01, p)
Sig_nu  <- random_Signu(p, 0)
Phi <- random_Phi(p, p)
Sig_e1 <- get_Sig_e1_approx(Sig_nu, Phi)

Y <- generate_data(mu0, deltas, Sig_eta, Sig_nu, Phi, Sig_e1,
                   errortype = "n", number_cps = 2, lengthofeachpart = 100)
dim(Y)

}
