% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/barycenter_simulation.R
\name{simulate_finite_ot_barycenter_test}
\alias{simulate_finite_ot_barycenter_test}
\alias{simulate_limit_ot_barycenter_test_null}
\alias{simulate_limit_ot_barycenter_test_alt}
\title{Simulations for \code{ot_barycenter_test}}
\usage{
simulate_finite_ot_barycenter_test(
  mu,
  costm,
  n,
  w = NULL,
  num.sim = 1000,
  solver = ot_test_lp_solver()
)

simulate_limit_ot_barycenter_test_null(
  mu,
  costm,
  n = NULL,
  delta = NULL,
  w = NULL,
  num.sim = 1000,
  solver = ot_test_lp_solver(),
  mean = NULL
)

simulate_limit_ot_barycenter_test_alt(
  mu,
  costm,
  delta,
  w = NULL,
  num.sim = 1000,
  solver = ot_test_lp_solver()
)
}
\arguments{
\item{mu}{matrix (row-wise) or list containing \eqn{K} probability vectors.}

\item{costm}{semi-metric cost matrix \eqn{c \in \mathbb{R}^{N \times N}}.}

\item{n}{vector of samples sizes.}

\item{w}{weight vector \eqn{w \in \mathbb{R}^K_+}.}

\item{num.sim}{number of samples to draw from the limiting null distribution.}

\item{solver}{the LP solver to use, see \code{\link{ot_test_lp_solver}}.}

\item{delta}{vector of asymptotic sample size ratios.}

\item{mean}{mean of the Gaussians appearing in the limiting distribution. Must be of the same structure as \code{mu}.}
}
\value{
A vector containing the simulated samples.
}
\description{
Perform simulations for the test statistic used in \code{\link{ot_barycenter_test}}.
}
\details{
See \code{\link{ot_barycenter_test}} for the definition of the test statistic and more details.

\code{simulate_finite_ot_barycenter_test} simulates from the finite sample distribution.

\code{simulate_limit_ot_barycenter_test_null} and \code{simulate_limit_ot_barycenter_test_alt} simulate from the limiting distribution
under the null or alternative, respectively.

All these simulations can be done in parallel via \code{\link[future:plan]{future::plan}} and the progress can be shown with \code{\link[progressr:with_progress]{progressr::with_progress}}.
}
\examples{

# enable txt progressbar
progressr::handlers("txtprogressbar")
# enable parallel computation
if (requireNamespace("future")) {
    future::plan(future::multisession)
}

K <- 3
N <- 2
costm <- cost_matrix_lp(1:N)

# use higher values for better approximation
num.sim <- 10
n <- rep(300, K)

delta <- rep(1 / K, K)

# under null
mu <- matrix(1 / N, K, N, TRUE)

# to run this, a LP solver must be available for ROI (ROI.plugin.glpk by default)
if (requireNamespace("ROI.plugin.glpk")) {
    solver <- ot_test_lp_solver("glpk")
    set.seed(123)
    # show progress bar
    progressr::with_progress({
        lhs <- simulate_finite_ot_barycenter_test(mu, costm, n, num.sim = num.sim, solver = solver)
        rhs <- simulate_limit_ot_barycenter_test_null(mu, costm, delta = delta, num.sim = num.sim,
                                                      solver = solver)
    })
    h1 <- density(lhs)
    h2 <- density(rhs)
    plot(h1$x, h1$y, xlim = range(h1$x, h2$x), ylim = range(h1$y, h2$y), type = "l",
         col = "red", xlab = "x", ylab = "density", main = "KDEs")
    lines(h2$x, h2$y, col = "blue")
    legend("topright", c("Finite", "Limit"), col = c("red", "blue"), pch = 15)
}

# under alternative
mu[2, ] <- 1:N / sum(1:N)

if (requireNamespace("ROI.plugin.glpk")) {
    solver <- ot_test_lp_solver("glpk")
    set.seed(123)
    lhs <- simulate_finite_ot_barycenter_test(mu, costm, n, num.sim = num.sim, solver = solver)
    rhs <- simulate_limit_ot_barycenter_test_alt(mu, costm, delta, num.sim = num.sim,
                                                 solver = solver)
    h1 <- density(lhs)
    h2 <- density(rhs)
    plot(h1$x, h1$y, xlim = range(h1$x, h2$x), ylim = range(h1$y, h2$y), type = "l",
         col = "red", xlab = "x", ylab = "density", main = "KDEs")
    lines(h2$x, h2$y, col = "blue")
    legend("topright", c("Finite", "Limit"), col = c("red", "blue"), pch = 15)
}
\dontshow{
## R CMD check: make sure any open connections are closed afterward
if (requireNamespace("future") && !inherits(future::plan(), "sequential")) future::plan(future::sequential)
}
}
