\name{cpost_stat}
\alias{cpost_stat}
\title{Estimation of the scedasis function}
\usage{
cpost_stat(N, x, xs, xg, bw, k, C = 5L)
}
\arguments{
\item{N}{integer, number of samples to draw from the distribution of the concomitant covariate}

\item{x}{one-dimensional vector of in-sample covariate in [0,1]}

\item{xs}{one-dimensional vector of concomitant covariate}

\item{xg}{one-dimensional vector of length m containing the grid of in-sample and possibly out-sample covariate in [0,1]}

\item{bw}{double, bandwidth for the computation of the kernel}

\item{k}{integer, number of exceedances for the generalized Pareto}

\item{C}{integer, hyperparameter entering the posterior distribution of the law of the concomitant covariate. Default: 5}
}
\value{
an \code{N} by \code{m} matrix containing the values of the posterior samples of the scedasis function (rows) for each value of \code{xg} (columns)
}
\description{
Kernel-based method for the estimation of the scedasis function. Given the values of the complete and concomitant covariate, defined as \eqn{X \mid Y > t}, with \eqn{t} being the threshold, it returns
a matrix containing a posterior sample of the scedasis function for each covariate value.
}
\examples{
\dontrun{
# generate data
set.seed(1234)
n <- 500
samp <- evd::rfrechet(n, 0, 1:n, 4)
# set effective sample size and threshold
k <- 50
threshold <- sort(samp, decreasing = TRUE)[k+1]
# preliminary mle estimates of scale and shape parameters
mlest <- evd::fpot(
  samp,
  threshold,
  control = list(maxit = 500))
# empirical bayes procedure
proc <- estPOT(
  samp,
  k = k,
  pn = c(0.01, 0.005),
  type = "continuous",
  method = "bayesian",
  prior = "empirical",
  start = as.list(mlest$estimate),
  sig0 = 0.1)
# conditional predictive density estimation
yg <- seq(0, 50, by = 2)
nyg <- length(yg)
# estimation of scedasis function
# setting
M <- 1e3
C <- 5
alpha <- 0.05
bw <- .5
nsim <- 5000
burn <- 1000
# create covariate
# in sample obs
n_in = n
# number of years ahead
nY = 1
n_out = 365 * nY
# total obs
n_tot = n_in + n_out
# total covariate (in+out sample period)
x <- seq(0, 1, length = n_tot)
# in sample grid dimension for covariate
ng_in <- 150
xg <- seq(0, x[n_in], length = ng_in)
# in+out of sample grid
xg <- c(xg, seq(x[n_in + 1], x[(n_tot)], length = ng_in))
# in+out sample grid dimension
nxg <- length(xg)
xg <- array(xg, c(nxg, 1))
# in sample observations
samp_in <- samp[1:n_in]
ssamp_in <- sort(samp_in, decreasing = TRUE, index = TRUE)
x_in <- x[1:n_in] # in sample covariate
xs <- x_in[ssamp_in$ix[1:k]]
# in sample concomitant covariate
# estimate scedasis function over the in and out of sample period
res_stat <- apply(
  xg,
  1,
  cpost_stat,
  N = nsim - burn,
  x = x_in,
  xs = xs,
  bw = bw,
  k = k,
  C = C
)
}
}
