% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mqdedist.R
\name{mqdedist}
\alias{mqdedist}
\title{Minimum Quantile Distance Fit of Univariate Distributions.}
\source{
Based on the function mledist of the R package: \pkg{fitdistrplus}
  
  Delignette-Muller ML and Dutang C (2015), \emph{fitdistrplus: An R Package 
  for Fitting Distributions}. Journal of Statistical Software, 64(4), 1-34.
  
  Functions \code{checkparam} and \code{startargdefault} are needed and 
  were copied from the same package (fitdistrplus version: 1.0-9).
}
\usage{
mqdedist(
  data,
  distr,
  probs = (1:length(data) - 0.5)/length(data),
  qtype = 5,
  dist = "euclidean",
  start = NULL,
  fix.arg = NULL,
  optim.method = "default",
  lower = -Inf,
  upper = Inf,
  custom.optim = NULL,
  weights = NULL,
  silent = TRUE,
  gradient = NULL,
  ...
)
}
\arguments{
\item{data}{A numeric vector with the observed values for non-censored data.}

\item{distr}{A character string \code{"name"} naming a distribution for which
the corresponding quantile function \code{qname} and the corresponding 
density distribution \code{dname} must be classically defined.}

\item{probs}{A numeric vector of the probabilities for which the minimum 
quantile distance estimation is done. \eqn{p[k] = (k - 0.5) / n} (default).}

\item{qtype}{The quantile type used by the R \code{\link{quantile}} function 
to compute the empirical quantiles. Type 5 (default), i.e. \eqn{x[k]} is 
both the \eqn{k}th order statistic and the type 5 sample quantile of 
\eqn{p[k] = (k - 0.5) / n}.}

\item{dist}{The distance measure between observed and theoretical quantiles 
to be used. This must be one of "euclidean" (default), "maximum", or 
"manhattan". Any unambiguous substring can be given.}

\item{start}{A named list giving the initial values of parameters of the 
named distribution or a function of data computing initial values and 
returning a named list. This argument may be omitted (default) for some 
distributions for which reasonable starting values are computed (see the 
'details' section of \code{\link[fitdistrplus]{mledist}}).}

\item{fix.arg}{An optional named list giving the values of fixed parameters 
of the named distribution or a function of data computing (fixed) parameter
values and returning a named list. Parameters with fixed value are thus NOT
estimated.}

\item{optim.method}{\code{"default"} (see details) or optimization method to pass to 
\code{\link{optim}}.}

\item{lower}{Left bounds on the parameters for the \code{"L-BFGS-B"} method 
(see \code{\link{optim}}) or the \code{\link{constrOptim}} function (as an 
equivalent linear constraint).}

\item{upper}{Right bounds on the parameters for the \code{"L-BFGS-B"} method 
(see \code{\link{optim}}) or the \code{\link{constrOptim}} function (as an 
equivalent linear constraint).}

\item{custom.optim}{A function carrying the optimization (see details).}

\item{weights}{An optional vector of weights to be used in the fitting 
process. Should be \code{NULL} or a numeric vector with strictly positive 
numbers. If non-\code{NULL}, weighted mqde is used, otherwise ordinary 
mqde.}

\item{silent}{A logical to remove or show warnings when bootstrapping.}

\item{gradient}{A function to return the gradient of the optimization 
objective function for the \code{"BFGS"}, \code{"CG"} and \code{"L-BFGS-B"}
methods. If it is \code{NULL}, a finite-difference approximation will be 
used, see \code{\link{optim}}.}

\item{\dots}{Further arguments passed to the \code{\link{optim}}, 
\code{\link{constrOptim}} or \code{custom.optim} function.}
}
\value{
\code{mqdedist} returns a list with following components,
  
  \item{estimate}{ the parameter estimates.}
  
  \item{convergence}{ an integer code for the convergence of 
  \code{\link{optim}} defined as below or defined by the user in the 
  user-supplied optimization function.
  
  \code{0} indicates successful convergence.
  
  \code{1} indicates that the iteration limit of \code{\link{optim}} has been
  reached.
  
  \code{10} indicates degeneracy of the Nealder-Mead simplex.
  
  \code{100} indicates that \code{\link{optim}} encountered an internal 
  error. }
  
  \item{value}{the value of the optimization objective function at the solution found.}
  
  \item{hessian}{ a symmetric matrix computed by \code{\link{optim}} as an
  estimate of the Hessian at the solution found or computed in the
  user-supplied optimization function. }
  
  \item{probs}{ the probability vector on which observed and theoretical quantiles were calculated. }
  
  \item{dist}{ the name of the distance between observed and theoretical quantiles used. }
  
  \item{optim.function }{ the name of the optimization function used.  }
  
  \item{fix.arg}{ the named list giving the values of parameters of the named
  distribution that must kept fixed rather than estimated by maximum
  likelihood or \code{NULL} if there are no such parameters. }
  
  \item{loglik}{ the log-likelihood. }
  
  \item{optim.method}{when \code{\link{optim}} is used, the name of the
  algorithm used, \code{NULL} otherwise.}
  
  \item{fix.arg.fun}{the function used to set the value of \code{fix.arg} or
  \code{NULL}.}
  
  \item{weights}{the vector of weights used in the estimation process or 
  \code{NULL}.}
  
  \item{counts}{A two-element integer vector giving the number of calls to
  the log-likelihood function and its gradient respectively. This excludes
  those calls needed to compute the Hessian, if requested, and any calls to
  log-likelihood function to compute a finite-difference approximation to the
  gradient. \code{counts} is returned by \code{\link{optim}} or the
  user-supplied optimization function, or set to \code{NULL}.}
  
  \item{optim.message}{A character string giving any additional information 
  returned by the optimizer, or \code{NULL}. To understand exactly the 
  message, see the source code.}
}
\description{
Fit of univariate distributions for non-censored data using 
  minimum quantile distance estimation (mqde), which can also be called 
  maximum quantile goodness-of-fit estimation.
}
\details{
The \code{mqdedist} function carries out the minimum quantile 
  distance estimation numerically, by minimization of a distance between 
  observed and theoretical quantiles.
  
  The optimization process is the same as 
  \code{\link[fitdistrplus]{mledist}}, see the 'details' section of that 
  function.
  
  Optionally, a vector of \code{weights} can be used in the fitting process. 
  By default (when \code{weights=NULL}), ordinary mqde is carried out, 
  otherwise the specified weights are used to compute a weighted distance.
  
  We believe this function should be added to the package 
  \pkg{fitdistrplus}. Until it is accepted and incorporated into that
  package, it will remain in the package \pkg{BMT}. This function is 
  internally called in \code{\link{BMTfit.mqde}}.
}
\examples{
# (1) basic fit of a normal distribution 
set.seed(1234)
x1 <- rnorm(n = 100)
mean(x1); sd(x1)
mqde1 <- mqdedist(x1, "norm")
mqde1$estimate

# (2) defining your own distribution functions, here for the Gumbel 
# distribution for other distributions, see the CRAN task view dedicated 
# to probability distributions
dgumbel <- function(x, a, b) 1/b*exp((a-x)/b)*exp(-exp((a-x)/b))
pgumbel <- function(q, a, b) exp(-exp((a-q)/b))
qgumbel <- function(p, a, b) a-b*log(-log(p))
mqde1 <- mqdedist(x1, "gumbel", start = list(a = 10, b = 5))
mqde1$estimate

# (3) fit a discrete distribution (Poisson)
set.seed(1234)
x2 <- rpois(n = 30, lambda = 2)
mqde2 <- mqdedist(x2, "pois")
mqde2$estimate

# (4) fit a finite-support distribution (beta)
set.seed(1234)
x3 <- rbeta(n = 100, shape1 = 5, shape2 = 10)
mqde3 <- mqdedist(x3, "beta")
mqde3$estimate

# (5) fit frequency distributions on USArrests dataset.
x4 <- USArrests$Assault
mqde4pois <- mqdedist(x4, "pois")
mqde4pois$estimate
mqde4nbinom <- mqdedist(x4, "nbinom")
mqde4nbinom$estimate

# (6) weighted fit of a normal distribution 
set.seed(1234)
w1 <- runif(100)
weighted.mean(x1, w1)
mqde1 <- mqdedist(x1, "norm", weights = w1)
mqde1$estimate

}
\references{
LaRiccia, V. N. (1982). \emph{Asymptotic Properties of Weighted 
  $L^2$ Quantile Distance Estimators}. The Annals of Statistics, 10(2), 
  621-624.
  
  Torres-Jimenez, C. J. (2017, September), \emph{Comparison of estimation methods 
  for the BMT distribution}. ArXiv e-prints.
}
\seealso{
\code{\link{mpsedist}}, \code{\link[fitdistrplus]{mledist}}, 
  \code{\link[fitdistrplus]{mmedist}}, \code{\link[fitdistrplus]{qmedist}}, 
  \code{\link[fitdistrplus]{mgedist}}, \code{\link{optim}}, 
  \code{\link{constrOptim}}, and \code{\link{quantile}}.
}
\author{
Camilo Jose Torres-Jimenez [aut,cre] \email{cjtorresj@unal.edu.co}
}
\keyword{distribution}
