% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_BKP.R, R/plot_DKP.R
\name{plot}
\alias{plot}
\alias{plot.BKP}
\alias{plot.DKP}
\title{Plot Fitted BKP or DKP Models}
\usage{
\method{plot}{BKP}(x, only_mean = FALSE, n_grid = 80, dims = NULL, ...)

\method{plot}{DKP}(x, only_mean = FALSE, n_grid = 80, dims = NULL, ...)
}
\arguments{
\item{x}{An object of class \code{"BKP"} or \code{"DKP"}, typically returned
by \code{\link{fit_BKP}} or \code{\link{fit_DKP}}.}

\item{only_mean}{Logical. If \code{TRUE}, only the predicted mean surface is
plotted for 2D inputs (applies to both BKP and DKP models for mean
visualization). Default is \code{FALSE}.}

\item{n_grid}{Positive integer specifying the number of grid points per
dimension for constructing the prediction grid. Larger values produce
smoother and more detailed surfaces, but increase computation time. Default
is \code{80}.}

\item{dims}{Integer vector indicating which input dimensions to plot. Must
have length 1 (for 1D) or 2 (for 2D). If \code{NULL} (default), all
dimensions are used when their number is <= 2.}

\item{...}{Additional arguments passed to internal plotting routines
(currently unused).}
}
\value{
This function is called for its side effects and does not return a
value. It produces plots visualizing the predicted probabilities, credible
intervals, and posterior summaries.
}
\description{
Visualizes fitted \code{BKP} (binary) or \code{DKP}
(multi-class) models according to the input dimensionality. For 1D inputs,
it shows predicted class probabilities with credible intervals and observed
data. For 2D inputs, it generates contour plots of posterior summaries. For
higher-dimensional inputs, users must specify which dimensions to plot.
}
\details{
The plotting behavior depends on the dimensionality of the input covariates:

\itemize{
\item \strong{1D inputs:}
\itemize{
\item For \code{BKP} (binary/binomial data), the function plots the posterior mean curve with a shaded 95\% credible interval, overlaid with the observed proportions (\eqn{y/m}).
\item For \code{DKP} (categorical/multinomial data), it plots one curve per class, each with a shaded credible interval and the observed class frequencies.
\item For classification tasks, an optional curve of the maximum posterior class probability can be displayed to visualize decision confidence.
}

\item \strong{2D inputs:}
\itemize{
\item For both BKP and DKP models, the function generates contour plots over a 2D prediction grid.
\item Users can choose to plot only the predictive mean surface (\code{only_mean = TRUE}) or a set of four summary plots (\code{only_mean = FALSE}):
\enumerate{
\item Predictive mean
\item 97.5th percentile (upper bound of 95\% credible interval)
\item Predictive variance
\item 2.5th percentile (lower bound of 95\% credible interval)
}
\item For DKP, these surfaces are generated separately for each class.
\item For classification tasks, predictive class probabilities can also be visualized as the maximum posterior probability surface.
}

\item \strong{Input dimensions greater than 2:}
\itemize{
\item The function does not automatically support visualization and will terminate with an error.
\item Users must specify which dimensions to visualize via the \code{dims} argument (length 1 or 2).
}
}
}
\examples{
# ============================================================== #
# ========================= BKP Examples ======================= #
# ============================================================== #

#-------------------------- 1D Example ---------------------------
set.seed(123)

# Define true success probability function
true_pi_fun <- function(x) {
  (1 + exp(-x^2) * cos(10 * (1 - exp(-x)) / (1 + exp(-x)))) / 2
}

n <- 30
Xbounds <- matrix(c(-2,2), nrow=1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)
y <- rbinom(n, size = m, prob = true_pi)

# Fit BKP model
model1 <- fit_BKP(X, y, m, Xbounds=Xbounds)

# Plot results
plot(model1)


#-------------------------- 2D Example ---------------------------
set.seed(123)

# Define 2D latent function and probability transformation
true_pi_fun <- function(X) {
  if(is.null(nrow(X))) X <- matrix(X, nrow=1)
  m <- 8.6928
  s <- 2.4269
  x1 <- 4*X[,1]- 2
  x2 <- 4*X[,2]- 2
  a <- 1 + (x1 + x2 + 1)^2 *
    (19- 14*x1 + 3*x1^2- 14*x2 + 6*x1*x2 + 3*x2^2)
  b <- 30 + (2*x1- 3*x2)^2 *
    (18- 32*x1 + 12*x1^2 + 48*x2- 36*x1*x2 + 27*x2^2)
  f <- log(a*b)
  f <- (f- m)/s
  return(pnorm(f))  # Transform to probability
}

n <- 100
Xbounds <- matrix(c(0, 0, 1, 1), nrow = 2)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)
y <- rbinom(n, size = m, prob = true_pi)

# Fit BKP model
model2 <- fit_BKP(X, y, m, Xbounds=Xbounds)

# Plot results
plot(model2, n_grid = 50)

# ============================================================== #
# ========================= DKP Examples ======================= #
# ============================================================== #

#-------------------------- 1D Example ---------------------------
set.seed(123)

# Define true class probability function (3-class)
true_pi_fun <- function(X) {
  p1 <- 1/(1+exp(-3*X))
  p2 <- (1 + exp(-X^2) * cos(10 * (1 - exp(-X)) / (1 + exp(-X)))) / 2
  return(matrix(c(p1/2, p2/2, 1 - (p1+p2)/2), nrow = length(p1)))
}

n <- 30
Xbounds <- matrix(c(-2, 2), nrow = 1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(150, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model1 <- fit_DKP(X, Y, Xbounds = Xbounds)

# Plot results
plot(model1)


#-------------------------- 2D Example ---------------------------
set.seed(123)

# Define latent function and transform to 3-class probabilities
true_pi_fun <- function(X) {
  if (is.null(nrow(X))) X <- matrix(X, nrow = 1)
  m <- 8.6928; s <- 2.4269
  x1 <- 4 * X[,1] - 2
  x2 <- 4 * X[,2] - 2
  a <- 1 + (x1 + x2 + 1)^2 *
    (19 - 14*x1 + 3*x1^2 - 14*x2 + 6*x1*x2 + 3*x2^2)
  b <- 30 + (2*x1 - 3*x2)^2 *
    (18 - 32*x1 + 12*x1^2 + 48*x2 - 36*x1*x2 + 27*x2^2)
  f <- (log(a*b)- m)/s
  p1 <- pnorm(f) # Transform to probability
  p2 <- sin(pi * X[,1]) * sin(pi * X[,2])
  return(matrix(c(p1/2, p2/2, 1 - (p1+p2)/2), nrow = length(p1)))
}

n <- 100
Xbounds <- matrix(c(0, 0, 1, 1), nrow = 2)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(150, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model2 <- fit_DKP(X, Y, Xbounds = Xbounds)

# Plot results
plot(model2, n_grid = 50)

}
\references{
Zhao J, Qing K, Xu J (2025). \emph{BKP: An R Package for Beta
Kernel Process Modeling}.  arXiv.
https://doi.org/10.48550/arXiv.2508.10447.
}
\seealso{
\code{\link{fit_BKP}} and \code{\link{fit_DKP}} for fitting BKP and
DKP models, respectively; \code{\link{predict.BKP}} and
\code{\link{predict.DKP}} for generating predictions from fitted BKP and
DKP models.
}
\keyword{BKP}
\keyword{DKP}
