#' Create a binary.prediction object
#'
#' @param bipartition The classes predictions (bipartition values), only 0 and 1
#' @param probability The probability/confidence of a prediction, between 0..1
#' @return An object of type binary.prediction
utiml_binary_prediction <- function(bipartition, probability) {
  res <- list(bipartition = bipartition, probability = probability)
  class(res) <- "binary.prediction"
  res
}

#' Create a data.frame from original mldr data for a single label
#'
#' @param mdata The mldr dataset
#' @param label.name The name of the label
#' @param extra.columns New columns to be added in the new dataset
#' @return a data.frame where the label is the last column
utiml_create_binary_data <- function (mdata, label.name, extra.columns = NULL) {
  if (is.null(extra.columns)) {
    cbind(mdata$dataset[mdata$attributesIndexes], mdata$dataset[label.name])
  }
  else {
    cbind(mdata$dataset[mdata$attributesIndexes],
          extra.columns,
          mdata$dataset[label.name])
  }
}

#' Create Dynamically the model for Binary Relevance Methods
#'
#' This method is used in the methods to call dinamicaly the base classifier
#'
#' @param dataset An object generated by the \code{\link{utiml_prepare_data}}
#'  method
#' @param ... Others arguments passed to the base method
#' @return The model dinamically generated
utiml_create_model <- function(dataset, ...) {
  # Call dynamic multilabel model with merged parameters
  model <- do.call(mltrain, c(list(object = dataset), ...))
  attr(model, "dataset") <- dataset$mldataset
  attr(model, "label") <- dataset$labelname

  model
}

#' Create a predictive multi-label result
#'
#' This function select the correct result and organize them in a prediction
#'  matrix where the columns are the labels and the rows are the test examples.
#'  If probability is \code{TRUE} the values contain the labels probabilities,
#'  otherwise the values are the predictive value '0' or '1'.
#'
#' @param predictions The list of binary.prediction obtained from union of all
#'  binary predictions.
#' @param probability A logical value. If \code{TRUE} the predicted values are
#'  the score between 0 and 1, otherwise the values are bipartition 0 or 1.
#' @return A matrix containing the probabilistic values or just predictions.
#'  If the matrix contains the probabilistic values then an attribute called
#'  'classes' contains the bipartitions values. Otherwise, if the matrix
#'  contains the bipartitions values then an attribute called 'probs' cantains
#'  the probabilities
utiml_predict <- function (predictions, probability) {
  bipartitions <- do.call(cbind, lapply(predictions, function(lblres) {
    lblres$bipartition
  }))

  probabilities <- do.call(cbind, lapply(predictions, function(lblres) {
    lblres$probability
  }))

  multilabel_prediction(bipartitions, probabilities, probability)
}

#' Dinamically call the prediction function
#'
#' @param model A model generated by \code{\link{utiml_create_model}} method
#' @param newdata The data to be predicted
#' @param ... Others arguments passed to the base method
#'
#' @return A binary.prediction object
utiml_predict_binary_model <- function(model, newdata, ...) {
  result <- do.call(mlpredict, c(list(model = model, newdata = newdata), ...))

  if (any(rownames(result) != rownames(newdata))) {
    where <- paste(attr(model, "dataset"), "/", attr(model, "label"))
    warning(cat("The order of the predicted instances from", where,
                "are wrong!\n", sep=' '))
  }

  #Because the factores is necessary first convert to character
  bipartition <- as.numeric(as.character(result$prediction))
  probability <- result$probability

  zeros <- bipartition == 0
  probability[zeros] <- 1 - probability[zeros]

  names(bipartition) <- names(probability) <- rownames(result)
  utiml_binary_prediction(bipartition, probability)
}

#' Prepare a Transformed Multi-Label Data to be processed
#'
#' @param dataset A data.frame with the data (the last column must be the class
#'  column)
#' @param classname The name of specific class of the object
#' @param mldataset The name of multi-label dataset
#' @param mlmethod The name of the multi-label method that will process this
#'  dataset
#' @param base.method The name of the base method that will process this dataset
#' @param ... Extra parameters for adding in the dataset object
#' @return A \code{mltransformation} object. This is used as a list and
#' contains at least five values:
#'  \describe{
#'    \item{data}{A data.frame with the train data, where the columns are
#'    the attributes and the rows are the examples.}
#'    \item{labelname}{The name of the class column.}
#'    \item{labelindex}{The column index of the class.}
#'    \item{mldataset}{The name of multi-label dataset.}
#'    \item{mlmethod}{The name of the multi-label method.}
#'  }
utiml_prepare_data <- function(dataset, classname, mldataset, mlmethod,
                               base.method, ...) {
  label <- colnames(dataset)[ncol(dataset)]

  # Convert the class column as factor
  dataset[, label] <- as.factor(dataset[, label])

  # Create object
  object <- list(
    data = dataset,
    labelname = label,
    labelindex = ncol(dataset),
    mldataset = mldataset,
    mlmethod = mlmethod
  )

  extra <- list(...)
  for (nextra in names(extra)) {
    object[[nextra]] <- extra[[nextra]]
  }

  basename <- paste("base", base.method, sep = "")
  class(object) <- c(classname, basename, "mltransformation")

  object
}

#' Summary method for mltransformation
#' @param object A transformed dataset
#' @param ... additional arguments affecting the summary produced.
#' @export
summary.mltransformation <- function(object, ...) {
  summary(object$data, ...)
}
