% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/misc_and_utility.R
\name{umx_score_scale}
\alias{umx_score_scale}
\title{Score a psychometric scale by summing normal and reversed items.}
\usage{
umx_score_scale(
  base = NULL,
  pos = NULL,
  rev = NULL,
  min = 1,
  max = NULL,
  data = NULL,
  score = c("total", "proportionCorrect", "errors", "mean", "max", "factor"),
  name = NULL,
  na.rm = TRUE,
  minManifests = NA,
  alpha = FALSE,
  mapStrings = NULL,
  correctAnswer = NULL,
  omegaNfactors = 1,
  digits = 2,
  verbose = FALSE,
  suffix = ""
)
}
\arguments{
\item{base}{String common to all item names.}

\item{pos}{The positive-scored item numbers.}

\item{rev}{The reverse-scored item numbers.}

\item{min}{Minimum legal response value (default = 1). Not implemented for values other than 1 so far...}

\item{max}{Maximum legal response value (also used to compute reversed item values).}

\item{data}{The data frame}

\item{score}{Score total (default), proportionCorrect, errors, mean, max, or factor scores}

\item{name}{The name of the scale to be returned. Defaults to "\code{base}_score"}

\item{na.rm}{Whether to delete NAs when computing scores (Default = TRUE) Note: Choice affects mean!}

\item{minManifests}{How many missing items to tolerate for an individual (when score = factor)}

\item{alpha}{print Reliability (omega and Cronbach's alpha) (TRUE)}

\item{mapStrings}{Recoding input like "No"/"Maybe"/"Yes" into numeric values (0,1,2)}

\item{correctAnswer}{Use when scoring items with one correct response (1/0).}

\item{omegaNfactors}{Number of factors for the omega reliability (default = 1)}

\item{digits}{Rounding for omega etc. (default 2)}

\item{verbose}{Whether to print the whole omega output (FALSE)}

\item{suffix}{(if dealing with, e.g. "_T1")}
}
\value{
\itemize{
\item scores
}
}
\description{
Use this function to generate scores as the appropriate sum of responses to the normal and reversed items in a scale.

Items must be named on the pattern \code{basename + N + suffix}, where \code{base} is the prefix common to all item (column) names, N is item number in the scale, and suffix an optional trail (like "_T1").

\code{pos} and \code{rev} are vectors of the item numbers for the normal and reverse-scored item numbers.

To reverse items, the function uses \code{max} and \code{min} as the lowest and highest possible response scores to compute how to reverse items.

\emph{note}: \code{min} defaults to 1.
}
\details{
In the presence of NAs, \code{score= "mean"} and \code{score = "totals"} both return NA unless na.rm = TRUE.
\code{score = "max"}, ignores NAs no matter what.
}
\examples{
library(psych)
library(psychTools)
data(bfi)

# ==============================
# = Score Agreeableness totals =
# ==============================

# Handscore subject 1
# A1(R)+A2+A3+A4+A5 = (6+1)-2 +4+3+4+4 = 20

tmp = umx_score_scale(base = "A", pos = 2:5, rev = 1, max = 6, data= bfi, name = "A")
tmp[1, namez(tmp, "A",ignore.case = FALSE)]
#  A1 A2 A3 A4 A5  A
#  2  4  3  4  4  20

# ====================
# = Request the mean =
# ====================
tmp = umx_score_scale(name = "A", base = "A", 
   pos = 2:5, rev = 1, max = 6, data= bfi, score="mean")
tmp$A[1] # = 4

# ========================
# = Request factor score =
# ========================
\dontrun{
tmp = umx_score_scale(name = "A", base = "A", pos = 2:5, rev = 1,
   max = 6, score = "factor", minManifests = 4, data= bfi)
#            g
# A2 0.6574826
# A3 0.7581274
# A4 0.4814788
# A5 0.6272332
# A1 0.3736021

# ==================
# = Request alpha  =
# ==================

tmp=umx_score_scale(base="A", pos=2:5, rev=1, max=6, data=bfi, alpha=TRUE)
# omega t = 0.72
}

# ==================
# = na.rm = TRUE ! =
# ==================
tmpDF = bfi
tmpDF[1, "A1"] = NA
tmp = umx_score_scale("A", pos = 2:5, rev = 1, max = 6, data= tmpDF, score="mean")
tmp$A_score[1] # 3.75

tmp= umx_score_scale("A", pos= 2:5, rev= 1, max = 6, data = tmpDF,
   score="mean", na.rm=FALSE)
tmp$A_score[1] # NA (reject cases with missing items)

# ===============
# = Score = max =
# ===============
tmp = umx_score_scale("A", pos = 2:5, rev = 1, max = 6,
  data = bfi, name = "A", score = "max")
tmp$A[1] # Subject 1 max = 5 (reversed) item 1

# Default scale name
tmp = umx_score_scale("E", pos = 3:5, rev = 1:2, max = 6, 
   data= tmp, score = "mean", na.rm = FALSE)
tmp$E_score[1]

# Using @BillRevelle's psych package: More diagnostics, including alpha
scores= psych::scoreItems(items = bfi, min = 1, max = 6, keys = list(
	E = c("-E1","-E2", "E3", "E4", "E5"),
	A = c("-A1", "A2", "A3", "A4", "A5")
))
summary(scores)
scores$scores[1, ]
#  E   A 
# 3.8 4.0 

# Compare output
# (note, by default psych::scoreItems replaces NAs with the sample median...)
RevelleE = as.numeric(scores$scores[,"E"])
RevelleE == tmp[,"E_score"]

# =======================
# = MapStrings examples =
# =======================
mapStrings = c(
   "Very Inaccurate", "Moderately Inaccurate", 
   "Slightly Inaccurate", "Slightly Accurate",
   "Moderately Accurate", "Very Accurate")
bfi$As1 = factor(bfi$A1, levels = 1:6, labels = mapStrings)
bfi$As2 = factor(bfi$A2, levels = 1:6, labels = mapStrings)
bfi$As3 = factor(bfi$A3, levels = 1:6, labels = mapStrings)
bfi$As4 = factor(bfi$A4, levels = 1:6, labels = mapStrings)
bfi$As5 = factor(bfi$A5, levels = 1:6, labels = mapStrings)
bfi= umx_score_scale(name="A" , base="A", pos=2:5, rev=1, max=6, data=bfi)
bfi= umx_score_scale(name="As", base="As", pos=2:5, rev=1, mapStrings = mapStrings, data= bfi)
}
\references{
\itemize{
\item Revelle, W. (2022) psych: Procedures for Personality and Psychological Research, Northwestern University, Evanston, Illinois, USA, \url{https://CRAN.R-project.org/package=psych} Version = 2.2.9.
}
\itemize{
\item McNeish, D. (2018). Thanks coefficient alpha, we’ll take it from here. \emph{Psychological Methods}, \strong{23}, 412-433. \doi{10.1037/met0000144}.
}
}
\seealso{
Other Data Functions: 
\code{\link{noNAs}()},
\code{\link{prolific_anonymize}()},
\code{\link{prolific_check_ID}()},
\code{\link{prolific_read_demog}()},
\code{\link{umxFactor}()},
\code{\link{umxHetCor}()},
\code{\link{umx_as_numeric}()},
\code{\link{umx_cont_2_quantiles}()},
\code{\link{umx_lower2full}()},
\code{\link{umx_make_MR_data}()},
\code{\link{umx_make_TwinData}()},
\code{\link{umx_make_fake_data}()},
\code{\link{umx_make_raw_from_cov}()},
\code{\link{umx_merge_randomized_columns}()},
\code{\link{umx_polychoric}()},
\code{\link{umx_polypairwise}()},
\code{\link{umx_polytriowise}()},
\code{\link{umx_read_lower}()},
\code{\link{umx_rename}()},
\code{\link{umx_reorder}()},
\code{\link{umx_select_valid}()},
\code{\link{umx_stack}()},
\code{\link{umx_strings2numeric}()},
\code{\link{umx}}
}
\concept{Data Functions}
