% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DSF.R
\name{DSF}
\alias{DSF}
\alias{reset_stream.DSF}
\alias{get_points.DSF}
\alias{update.DSF}
\alias{close_stream.DSF}
\title{Data Stream Filter Base Classes}
\usage{
DSF(...)

\method{reset_stream}{DSF}(dsd, pos = 1)

\method{get_points}{DSF}(x, n = 1L, info = TRUE, ...)

\method{update}{DSF}(object, dsd = NULL, n = 1L, return = "data", ...)

\method{close_stream}{DSF}(dsd, ...)
}
\arguments{
\item{...}{Further arguments passed on.}

\item{dsd}{a stream object of class \link{DSD}.}

\item{pos}{position in the stream.}

\item{x, object}{a \code{DSF} object.}

\item{n}{number of points to get/use for the update.}

\item{info}{return additional columns with information about the data point (e.g., a known cluster assignment).}

\item{return}{a character string indicating what update returns. The only
value is currently  \code{"data"} to return the transformed data.
possible values depend on the \code{DST}.}
}
\description{
Abstract base classes for all data stream filter (DSF) classes.
Data stream filters transform a data stream (\link{DSD}).
}
\details{
The \code{DSF} class cannot be instantiated, but it serve as a base
class from which other DSF classes inherit.

Data stream filters transform a \link{DSD} data stream. \code{DSF} can be used in two ways.
\enumerate{
\item \strong{DSD Adapter:}
When a data stream (\code{dsd}) is specified in the constructor, then the DSF acts as an a
adapter for a connected data stream. The DSF implementations inherit the
interface from \link{DSD} and provide:
\itemize{
\item \code{\link[=get_points]{get_points()}} get the transformed points.
\item \code{\link[=reset_stream]{reset_stream()}} reset the underlying stream
\item \code{\link[=close_stream]{close_stream()}} close the underlying stream
}
\item \strong{Stream Transformer:}
When no data stream (\code{dsd}) is specified in the constructor, then the DSF acts like a
\link{DST} data stream task and provides:
\itemize{
\item \code{\link[=update]{update()}} to transform the points from a specified \code{DSD}.
It is convenient to use the pipe (\link[magrittr:pipe]{magrittr::\%>\%}) to apply one or more filters to data
streams (see Examples section).
}
}
}
\section{Methods (by generic)}{
\itemize{
\item \code{reset_stream(DSF)}: reset the attached stream if reset is supported.

\item \code{get_points(DSF)}: DSD-like interface to get points if the DSF was created with an attached stream.

\item \code{update(DSF)}: updates with data and returns the filtered data.

\item \code{close_stream(DSF)}: close the attached stream if close is supported.

}}
\examples{
DSF()

# Example 1: Use as a DSD adapter
stream <- DSD_Gaussians(k = 3, d = 2) \%>\%
  DSF_Func(func = function(x) cbind(x, Xsum = x$X1 + x$X2))
stream

get_points(stream, n = 5)

# Example 2: Use as a stream transformer
trans <- DSF_Func(func = function(x) cbind(x, Xsum = x$X1 + x$X2))
trans

update(trans, stream, n = 5)

# Example 3: Use as a DST preprocessor
clusterer <- DSF_Func(func = function(x) cbind(x, X1_squared = x$X1^2)) \%>\%
               DST_Runner(DSC_Kmeans(k = 3))
clusterer

update(clusterer, stream, n = 100)

# Example 5: Specify a complete pipeline DSD -> DSF -> DST
pipeline <- DSD_Gaussians(k = 3, d = 2) \%>\%
               DSF_Func(func = function(x) cbind(x, X1_squared = x$X1^2)) \%>\%
               DST_Runner(DSC_Kmeans(k = 3))
pipeline

update(pipeline, n = 100)
plot(pipeline$dst)
}
\seealso{
Other DSF: 
\code{\link{DSF_Convolve}()},
\code{\link{DSF_Downsample}()},
\code{\link{DSF_ExponentialMA}()},
\code{\link{DSF_Func}()},
\code{\link{DSF_Scale}()},
\code{\link{DSF_dplyr}()}

Other DSD: 
\code{\link{DSD_BarsAndGaussians}()},
\code{\link{DSD_Benchmark}()},
\code{\link{DSD_Cubes}()},
\code{\link{DSD_Gaussians}()},
\code{\link{DSD_MG}()},
\code{\link{DSD_Memory}()},
\code{\link{DSD_Mixture}()},
\code{\link{DSD_NULL}()},
\code{\link{DSD_ReadDB}()},
\code{\link{DSD_ReadStream}()},
\code{\link{DSD_Target}()},
\code{\link{DSD_UniformNoise}()},
\code{\link{DSD_mlbenchData}()},
\code{\link{DSD_mlbenchGenerator}()},
\code{\link{DSD}()},
\code{\link{animate_data}()},
\code{\link{close_stream}()},
\code{\link{get_points}()},
\code{\link{plot.DSD}()},
\code{\link{reset_stream}()}
}
\author{
Michael Hahsler
}
\concept{DSD}
\concept{DSF}
