#' @title Expressions with statistics for tidy regression data frames
#' @name tidy_model_expressions
#'
#' @param ... Currently ignored.
#' @param data A tidy data frame from regression model object (see
#'   `statsExpressions::tidy_model_parameters()`).
#' @param statistic Which statistic is to be displayed (either `"t"` or `"f"`or
#'   `"z"` or `"chi"`) in the expression.
#' @inheritParams oneway_anova
#'
#' @details
#' When any of the necessary numeric column values (`estimate`, `statistic`,
#' `p.value`) are missing, for these rows, a `NULL` is returned instead of an
#' expression with empty strings.
#'
#' @note
#'
#' This is an **experimental** function and may change in the future. Please do
#' not use it yet in your workflow.
#'
#' @examples
#' set.seed(123)
#'
#' # tidy dataframe
#' df <- tidy_model_parameters(lm(wt ~ am * cyl, mtcars))
#'
#' # create a column containing expressions
#' tidy_model_expressions(df, statistic = "t")
#' @export
tidy_model_expressions <- function(data,
                                   statistic = NULL,
                                   k = 2L,
                                   effsize.type = "omega",
                                   ...) {
  # standardize the statistic naming
  statistic <- substring(tolower(statistic), 1L, 1L)

  # if any of the necessary numeric columns are missing, there shouldn't be an
  # expression corresponding to that row
  #
  # convert the necessary columns to character type for expression
  df <- data %>%
    filter(if_all(
      .cols = c(matches("estimate|statistic|std.error|p.value")),
      .fns = ~ !is.na(.)
    )) %>%
    .data_to_char(k)

  # effect size text for the expression (common for t, z, and chi^2)
  es.text <- list(quote(widehat(italic(beta))))

  # t-statistic --------------------------------

  if (statistic == "t") {
    df %<>% mutate(
      expression = case_when(
        df.error %in% c("", "Inf") ~ glue("list({es.text}=='{estimate}', italic(t)=='{statistic}', italic(p)=='{p.value}')"),
        TRUE ~ glue("list({es.text}=='{estimate}', italic(t)('{df.error}')=='{statistic}', italic(p)=='{p.value}')")
      )
    )
  }

  # z-statistic ---------------------------------

  if (statistic == "z") {
    df %<>% mutate(expression = glue("list({es.text}=='{estimate}', italic(z)=='{statistic}', italic(p)=='{p.value}')"))
  }

  # chi^2-statistic -----------------------------

  if (statistic == "c") {
    df %<>% mutate(expression = glue("list({es.text}=='{estimate}', italic(chi)^2*('{df.error}')=='{statistic}', italic(p)=='{p.value}')"))
  }

  # f-statistic ---------------------------------

  if (statistic == "f") {
    # which effect size is needed?
    if (effsize.type == "eta") es.text <- list(quote(widehat(italic(eta)[p]^2)))
    if (effsize.type == "omega") es.text <- list(quote(widehat(italic(omega)[p]^2)))

    df %<>% mutate(expression = glue("list({es.text}=='{estimate}', italic(F)('{df}', '{df.error}')=='{statistic}', italic(p)=='{p.value}')"))
  }

  # add the `expression` column to the original data frame
  #
  # the rows for which no expression was created will have `NA`s in this column,
  # which should instead be replaced with `NULL`, which when parsed in the plotting
  # context, will not show anything instead of empty string
  left_join(data, select(df, term, expression), by = "term") %>%
    .glue_to_expression()
}


#' @keywords internal
#' @noRd
.glue_to_expression <- function(data) {
  data %>%
    rowwise() %>%
    mutate(expression = list(parse_expr(expression))) %>%
    ungroup() %>% # convert from `expression` to `language`
    mutate(expression = case_when(
      is.na(unlist(expression)) ~ list(NULL),
      TRUE ~ unlist(expression)
    ))
}
