% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sp.runs.test.R
\name{sp.runs.test}
\alias{sp.runs.test}
\title{Compute the global spatial runs test.}
\usage{
sp.runs.test(formula = NULL, data = NULL, fx = NULL,
listw = listw, alternative = "two.sided" ,
distr = "asymptotic", nsim = NULL,control = list())
}
\arguments{
\item{formula}{a symbolic description of the factor (optional).}

\item{data}{an (optional) data frame or a sf object containing the variable to testing for.}

\item{fx}{a factor (optional).}

\item{listw}{A neighborhood list (type knn or nb) or a W matrix that indicates the order of the elements in each  $m_i-environment$
(for example of inverse distance). To calculate the number of runs in each $m_i-environment$, an order must
be established, for example from the nearest neighbor to the furthest one.}

\item{alternative}{a character string specifying the alternative hypothesis, must be one
of "two.sided" (default), "greater" or "less".}

\item{distr}{A string. Distribution of the test "asymptotic" (default) or "bootstrap"}

\item{nsim}{Number of permutations to obtain confidence intervals (CI).
Default value is NULL to don`t get CI of number of runs.}

\item{control}{List of additional control arguments.}
}
\value{
A object of the \emph{htest} and \emph{sprunstest} class
  \tabular{ll}{
    \code{data.name} \tab a character string giving the names of the data.\cr
    \code{method} \tab the type of test applied ().\cr
    \code{SR} \tab total number of runs \cr
    \code{dnr} \tab empirical distribution of the number of runs \cr
    \code{statistic} \tab Value of the homogeneity runs statistic. Negative sign indicates global homogeneity \cr
    \code{alternative} \tab a character string describing the alternative hypothesis. \cr
    \code{p.value} \tab p-value of the SRQ \cr
    \code{pseudo.value} \tab the pseudo p-value of the SRQ test if nsim is not NULL\cr
    \code{MeanNeig} \tab Mean of the Maximum number of neighborhood \cr
    \code{MaxNeig} \tab Maximum number of neighborhood \cr
    \code{listw} \tab The list of neighborhood \cr
    \code{nsim} \tab number of boots (only for boots version) \cr
    \code{SRGP} \tab nsim simulated values of statistic. \cr
    \code{SRLP} \tab matrix with the number of runs for eacl localization. \cr
    }
}
\description{
This function compute the global spatial runs test for spatial independence of a
categorical spatial data set.
}
\details{
The order of the neighborhoods ($m_i-environments$) is critical to obtain the test. \cr
To obtain the number of runs observed in each $m_i-environment$, each element must be associated
with a set of neighbors ordered by proximity. Three kinds of lists can be included to identify $m_i-environments$:

\tabular{ll}{
    \code{knn} \tab Objects of the class knn that consider the neighbors in order of proximity.\cr
    \code{nb} \tab If the neighbors are obtained from an sf object, the code internally
    will call the function \code{\link{nb2nb_order}} it will order them in order of proximity of the centroids.\cr
    \code{matrix} \tab
    If a object of matrix class based in the inverse of the distance in introduced as argument, the function \code{\link{nb2nb_order}} will
    also be called internally to transform the object the class matrix to a matrix of the class nb with ordered neighbors. \cr
    }

Two alternative sets of arguments can be included in this function to compute the spatial runs test:

  \tabular{ll}{
    \code{Option 1} \tab A factor (fx) and a list of neighborhood (\code{listw}) of the class knn. \cr
    \code{Option 2} \tab A sf object (data) and formula to specify the factor. A list of neighborhood (listw) \cr
    }
}
\section{Control arguments}{

  \tabular{ll}{
    \code{seedinit} \tab Numerical value for the seed (only for boot version). Default value seedinit=123 \cr
      }
}

\examples{

# Case 1: SRQ test based on factor and knn
\donttest{
n <- 100
cx <- runif(n)
cy <- runif(n)
x <- cbind(cx,cy)
listw <- spdep::knearneigh(cbind(cx,cy), k=3)
p <- c(1/6,3/6,2/6)
rho <- 0.5
fx <- dgp.spq(listw = listw, p = p, rho = rho)
srq <- sp.runs.test(fx = fx, listw = listw)
print(srq)
plot(srq)

# Boots Version
control <- list(seedinit = 1255)
srq <- sp.runs.test(fx = fx, listw = listw, distr = "bootstrap" , nsim = 299, control = control)
print(srq)
plot(srq)

# Case 2: SRQ test with formula, a sf object (points) and knn
data("FastFood.sf")
x <- sf::st_coordinates(sf::st_centroid(FastFood.sf))
listw <- spdep::knearneigh(x, k=4)
formula <- ~ Type
srq <- sp.runs.test(formula = formula, data = FastFood.sf, listw = listw)
print(srq)
plot(srq)
# Version boots
srq <- sp.runs.test(formula = formula, data = FastFood.sf, listw = listw,
distr = "bootstrap", nsim = 199)
print(srq)
plot(srq)

# Case 3: SRQ test (permutation) using formula with a sf object (polygons) and nb
library(sf)
fname <- system.file("shape/nc.shp", package="sf")
nc <- sf::st_read(fname)
listw <- spdep::poly2nb(as(nc,"Spatial"), queen = FALSE)
p <- c(1/6,3/6,2/6)
rho = 0.5
co <- sf::st_coordinates(sf::st_centroid(nc))
nc$fx <- dgp.spq(listw = listw, p = p, rho = rho)
plot(nc["fx"])
formula <- ~ fx
srq <- sp.runs.test(formula = formula, data = nc, listw = listw,
distr = "bootstrap", nsim = 399)
print(srq)
plot(srq)

# Case 4: SRQ test (Asymptotic) using formula with a sf object (polygons) and nb
data(provinces_spain)
sf::sf_use_s2(FALSE)
listw <- spdep::poly2nb(provinces_spain, queen = FALSE)
provinces_spain$Coast <- factor(provinces_spain$Coast)
levels(provinces_spain$Coast) = c("no","yes")
plot(provinces_spain["Coast"])
formula <- ~ Coast
srq <- sp.runs.test(formula = formula, data = provinces_spain, listw = listw)
print(srq)
plot(srq)

# Boots version
srq <- sp.runs.test(formula = formula, data = provinces_spain, listw = listw,
distr = "bootstrap", nsim = 299)
print(srq)
plot(srq)

# Case 5: SRQ test based on a distance matrix (inverse distance)
N <- 100
cx <- runif(N)
cy <- runif(N)
data <- as.data.frame(cbind(cx,cy))
data <- sf::st_as_sf(data,coords = c("cx","cy"))
n = dim(data)[1]
dis <- 1/matrix(as.numeric(sf::st_distance(data,data)),ncol=n,nrow=n)
diag(dis) <- 0
dis <- (dis < quantile(dis,.10))*dis
p <- c(1/6,3/6,2/6)
rho <- 0.5
fx <- dgp.spq(listw = dis , p = p, rho = rho)
srq <- sp.runs.test(fx = fx, listw = dis)
print(srq)
plot(srq)

srq <- sp.runs.test(fx = fx, listw = dis, data = data)
print(srq)
plot(srq)

# Boots version
srq <- sp.runs.test(fx = fx, listw = dis, data = data, distr = "bootstrap", nsim = 299)
print(srq)
plot(srq)

# Case 6: SRQ test based on a distance matrix (inverse distance)
data("FastFood.sf")
sf::sf_use_s2(FALSE)
n = dim(FastFood.sf)[1]
dis <- 1000000/matrix(as.numeric(sf::st_distance(FastFood.sf,FastFood.sf)), ncol = n, nrow = n)
diag(dis) <- 0
dis <- (dis < quantile(dis,.005))*dis
p <- c(1/6,3/6,2/6)
rho = 0.5
co <- sf::st_coordinates(sf::st_centroid(FastFood.sf))
FastFood.sf$fx <- dgp.spq(p = p, listw = dis, rho = rho)
plot(FastFood.sf["fx"])
formula <- ~ fx

# Boots version
srq <- sp.runs.test(formula = formula, data = FastFood.sf, listw = dis,
distr = "bootstrap", nsim = 299)
print(srq)
plot(srq)
}

}
\seealso{
\code{\link{local.sp.runs.test}}, \code{\link{dgp.spq}}, \code{\link{Q.test}},
}
\author{
\tabular{ll}{
  Fernando López  \tab \email{fernando.lopez@upct.es} \cr
  Román Mínguez  \tab \email{roman.minguez@uclm.es} \cr
  Antonio Páez \tab \email{paezha@gmail.com} \cr
  Manuel Ruiz \tab \email{manuel.ruiz@upct.es} \cr
  }
  @references
  \itemize{
    \item Ruiz, M., López, F., and Páez, A. (2021).
    A test for global and local homogeneity of categorical data based on spatial runs.
      \emph{Working paper}.
  }
}
