% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sperrorest.R
\name{sperrorest}
\alias{sperrorest}
\title{Perform spatial error estimation and variable importance assessment}
\usage{
sperrorest(
  formula,
  data,
  coords = c("x", "y"),
  model_fun,
  model_args = list(),
  pred_fun = NULL,
  pred_args = list(),
  smp_fun = partition_cv,
  smp_args = list(),
  train_fun = NULL,
  train_param = NULL,
  test_fun = NULL,
  test_param = NULL,
  err_fun = err_default,
  imp_variables = NULL,
  imp_permutations = 1000,
  imp_sample_from = c("test", "train", "all"),
  importance = !is.null(imp_variables),
  distance = FALSE,
  do_gc = 1,
  progress = "all",
  benchmark = FALSE,
  mode_rep = c("future", "sequential", "loop"),
  mode_fold = c("sequential", "future", "loop"),
  verbose = 0
)
}
\arguments{
\item{formula}{A formula specifying the variables used by the \code{model}. Only
simple formulas without interactions or nonlinear terms should be used,
e.g. \code{y~x1+x2+x3} but not \code{y~x1*x2+log(x3)}. Formulas involving interaction
and nonlinear terms may possibly work for error estimation but not for
variable importance assessment, but should be used with caution.
The formula \code{y~...} is not supported, but \code{y~1} (i.e. no predictors) is.}

\item{data}{a \code{data.frame} with predictor and response variables. Training
and test samples will be drawn from this data set by \code{train_fun} and
\code{test_fun}, respectively.}

\item{coords}{vector of length 2 defining the variables in \code{data} that
contain the x and y coordinates of sample locations.}

\item{model_fun}{Function that fits a predictive model, such as \code{glm} or
\code{rpart}. The function must accept at least two arguments, the first one
being a formula and the second a data.frame with the learning sample.}

\item{model_args}{Arguments to be passed to \code{model_fun} (in addition to the
\code{formula} and \code{data} argument, which are provided by {sperrorest})}

\item{pred_fun}{Prediction function for a fitted model object created by
\code{model}. Must accept at least two arguments: the fitted \code{object} and a
\code{data.frame} \code{newdata} with data on which to predict the outcome.}

\item{pred_args}{(optional) Arguments to \code{pred_fun} (in addition to the
fitted model object and the \code{newdata} argument, which are provided by
{sperrorest}).}

\item{smp_fun}{A function for sampling training and test sets from \code{data}.
E.g. \link{partition_kmeans} for spatial cross-validation using spatial
\emph{k}-means clustering.}

\item{smp_args}{(optional) Arguments to be passed to \code{smp_fun}.}

\item{train_fun}{(optional) A function for resampling or subsampling the
training sample in order to achieve, e.g., uniform sample sizes on all
training sets, or maintaining a certain ratio of positives and negatives in
training sets. E.g. \link{resample_uniform} or \link{resample_strat_uniform}.}

\item{train_param}{(optional) Arguments to be passed to \code{resample_fun}.}

\item{test_fun}{(optional) Like \code{train_fun} but for the test set.}

\item{test_param}{(optional) Arguments to be passed to \code{test_fun}.}

\item{err_fun}{A function that calculates selected error measures from the
known responses in \code{data} and the model predictions delivered by
\code{pred_fun}. E.g. \link{err_default} (the default).}

\item{imp_variables}{(optional; used if \code{importance = TRUE}). Variables for
which permutation-based variable importance assessment is performed. If
\code{importance = TRUE} and \code{imp_variables} == \code{NULL}, all variables in
\code{formula} will be used.}

\item{imp_permutations}{(optional; used if \code{importance = TRUE}). Number of
permutations used for variable importance assessment.}

\item{imp_sample_from}{(default: \code{"test"}): specified if the permuted feature
values should be taken from the test set, the training set (a rather unlikely
choice), or the entire sample (\code{"all"}). The latter is useful in
leave-one-out resampling situations where the test set is simply too small
to perform any kind of resampling. In any case importances are
always estimates on the test set. (Note that resampling with replacement is
used if the test set is larger than the set from which the permuted values
are to be taken.)}

\item{importance}{logical (default: \code{FALSE}): perform permutation-based
variable importance assessment?}

\item{distance}{logical (default: \code{FALSE}): if \code{TRUE}, calculate mean
nearest-neighbour distances from test samples to training samples using
\link{add.distance.represampling}.}

\item{do_gc}{numeric (default: 1): defines frequency of memory garbage
collection by calling \link{gc}; if \verb{< 1}, no garbage collection; if \verb{>= 1}, run
a \link{gc} after each repetition; if \verb{>= 2}, after each fold.}

\item{progress}{character (default: \code{all}): Whether to show progress
information (if possible). Default shows repetition, fold and (if enabled)
variable importance progress. Set to \code{"rep"} for repetition information
only or \code{FALSE} for no progress information.}

\item{benchmark}{(optional) logical (default: \code{FALSE}): if \code{TRUE}, perform
benchmarking and return \code{sperrorestbenchmark} object.}

\item{mode_rep, mode_fold}{character (default: \code{"future"} and \code{"sequential"},
respectively): specifies whether to parallelize the execution at the repetition
level, at the fold level, or not at all.
Parallel execution uses \code{future.apply::future_lapply()} (see details below).
It is only possible to parallelize at the repetition level or at
the fold level.
The \code{"loop"} option uses a \code{for} loop instead of an \code{lappy}
function; this option is for debugging purposes.}

\item{verbose}{Controls the amount of information printed while processing.
Defaults to 0 (no output).}
}
\value{
A list (object of class {sperrorest}) with (up to) six components:
\itemize{
\item error_rep: \code{sperrorestreperror} containing
predictive performances at the repetition level
\item error_fold: \code{sperroresterror} object containing predictive
performances at the fold level
\item represampling: \link{represampling} object
\item importance: \code{sperrorestimportance} object containing
permutation-based variable importances at the fold level
\item benchmark: \code{sperrorestbenchmark} object containing
information on the system the code is running on, starting and
finishing times, number of available CPU cores and runtime performance
\item package_version: \code{sperrorestpackageversion} object containing
information about the {sperrorest} package version
}
}
\description{
{sperrorest} is a flexible interface for multiple types of
parallelized spatial and non-spatial cross-validation and bootstrap error
estimation and parallelized permutation-based assessment of spatial variable
importance.
}
\details{
Custom predict functions passed to \code{pred_fun}, which consist of
multiple child functions, must be defined in one function.
}
\section{Parallelization}{


Running in parallel is supported via package \CRANpkg{future}.
Have a look at \code{vignette("future-1-overview", package = "future")}.
In short: Choose a backend and specify the number of workers, then call
\code{sperrorest()} as usual. Example:\if{html}{\out{<div class="sourceCode r">}}\preformatted{future::plan(future.callr::callr, workers = 2)
sperrorest()
}\if{html}{\out{</div>}}

Parallelization at the repetition is recommended when using
repeated cross-validation. If the 'granularity' of parallelized
function calls is too fine, the overall runtime will be very
poor since the overhead for passing arguments and handling
environments becomes too large. Use fold-level parallelization
only when the processing time of individual folds is very
large and the number of repetitions is small or equals 1.

Note that nested calls to \code{future} are not possible.
Therefore a sequential \code{sperrorest} call should be used for
hyperparameter tuning in a nested cross-validation.
}

\examples{

## ------------------------------------------------------------
## Classification tree example using non-spatial partitioning
## ------------------------------------------------------------

# Muenchow et al. (2012), see ?ecuador
fo <- slides ~ dem + slope + hcurv + vcurv + log.carea + cslope

library(rpart)
mypred_part <- function(object, newdata) predict(object, newdata)[, 2]
ctrl <- rpart.control(cp = 0.005) # show the effects of overfitting
# show the effects of overfitting
fit <- rpart(fo, data = ecuador, control = ctrl)

### Non-spatial cross-validation:
mypred_part <- function(object, newdata) predict(object, newdata)[, 2]
nsp_res <- sperrorest(
  data = ecuador, formula = fo,
  model_fun = rpart,
  model_args = list(control = ctrl),
  pred_fun = mypred_part,
  progress = TRUE,
  smp_fun = partition_cv,
  smp_args = list(repetition = 1:2, nfold = 3)
)
summary(nsp_res$error_rep)
summary(nsp_res$error_fold)
summary(nsp_res$represampling)
# plot(nsp_res$represampling, ecuador)

### Spatial cross-validation:
sp_res <- sperrorest(
  data = ecuador, formula = fo,
  model_fun = rpart,
  model_args = list(control = ctrl),
  pred_fun = mypred_part,
  progress = TRUE,
  smp_fun = partition_kmeans,
  smp_args = list(repetition = 1:2, nfold = 3)
)
summary(sp_res$error_rep)
summary(sp_res$error_fold)
summary(sp_res$represampling)
# plot(sp_res$represampling, ecuador)

smry <- data.frame(
  nonspat_training = unlist(summary(nsp_res$error_rep,
    level = 1
  )$train_auroc),
  nonspat_test = unlist(summary(nsp_res$error_rep,
    level = 1
  )$test_auroc),
  spatial_training = unlist(summary(sp_res$error_rep,
    level = 1
  )$train_auroc),
  spatial_test = unlist(summary(sp_res$error_rep,
    level = 1
  )$test_auroc)
)
boxplot(smry,
  col = c("red", "red", "red", "green"),
  main = "Training vs. test, nonspatial vs. spatial",
  ylab = "Area under the ROC curve"
)
}
\references{
Brenning, A. 2012. Spatial cross-validation and bootstrap for
the assessment of prediction rules in remote sensing: the R package
'sperrorest'.
2012 IEEE International Geoscience and Remote Sensing Symposium (IGARSS),
23-27 July 2012, p. 5372-5375.
\url{https://ieeexplore.ieee.org/document/6352393}

Brenning, A. 2005. Spatial prediction models for landslide hazards: review,
comparison and evaluation. Natural Hazards and Earth System Sciences,
5(6), 853-862. \doi{10.5194/nhess-5-853-2005}

Brenning, A., S. Long & P. Fieguth. 2012. Detecting rock glacier
flow structures using Gabor filters and IKONOS imagery.
Remote Sensing of Environment, 125, 227-237.
\doi{10.1016/j.rse.2012.07.005}

Russ, G. & A. Brenning. 2010a. Data mining in precision agriculture:
Management of spatial information. In 13th International Conference on
Information Processing and Management of Uncertainty, IPMU 2010; Dortmund;
28 June - 2 July 2010. Lecture Notes in Computer Science, 6178 LNAI: 350-359.

Russ, G. & A. Brenning. 2010b. Spatial variable importance assessment for
yield prediction in Precision Agriculture. In Advances in Intelligent
Data Analysis IX, Proceedings, 9th International Symposium,
IDA 2010, Tucson, AZ, USA, 19-21 May 2010.
Lecture Notes in Computer Science, 6065 LNCS: 184-195.
}
