\name{C_fSolD}
\Rdversion{1.3}
\encoding{UTF-8}
\alias{fSolD}

\title{
  Daily apparent movement of the Sun from the Earth
}
\description{
  Compute the daily apparent movement of the Sun from the Earth. This movement is mainly described (for the simulation of photovoltaic systems) by the declination angle, the sunrise angle and the daily extra-atmospheric irradiation.
}
\usage{
fSolD(lat, BTd, method = 'michalsky')
}

\arguments{
  \item{lat}{Latitude (degrees) of the point of the Earth where calculations are needed. It is positive for locations above the Equator.}
  \item{BTd}{Daily temporal base, a \code{POSIXct} object which  may be
    the result of \code{\link{fBTd}}.}
  \item{method}{\code{character}, method for the sun geometry
    calculations to be chosen from 'cooper', 'spencer', 'michalsky' and
    'strous'. See references for details.}
}

\value{A \code{data.table} object with these components:
  \item{lat}{Latitude (degrees)}
  \item{decl}{Declination angle (radians) for each day of year in \code{dn} or \code{BTd}}
  \item{eo}{Factor of correction due the eccentricity of orbit of the Earth around the Sun.}
  \item{ws}{Sunrise angle (in radians) for each day of year. Due to the convention which considers that the solar hour angle is negative before midday, this angle is negative.}
  \item{Bo0d}{Extra-atmospheric daily irradiation (watt-hour per
    squared meter) incident on a horizontal surface}
  \item{EoT}{Equation of Time.}
}

\references{\itemize{ \item{Cooper, P.I., Solar Energy, 12, 3
    (1969). "The Absorption of Solar Radiation in Solar Stills"}
    \item{Spencer, Search 2 (5), 172,
    \url{https://www.mail-archive.com/sundial@uni-koeln.de/msg01050.html}}
    \item{Strous:
    \url{https://www.aa.quae.nl/en/reken/zonpositie.html}}
    \item{Michalsky, J., 1988: The Astronomical Almanac's algorithm for
    approximate solar position (1950-2050), Solar Energy 40, 227-235}
  \item Perpiñán, O, Energía Solar Fotovoltaica, 2015.
    (\url{https://oscarperpinan.github.io/esf/})
    \item Perpiñán, O. (2012), "solaR: Solar Radiation and Photovoltaic
  Systems with R", Journal of Statistical Software, 50(9), 1-32,
  \doi{10.18637/jss.v050.i09}
    }}


\author{
  Oscar Perpiñán Lamigueiro,
  Francisco Delgado López.
}
\examples{
BTd <- fBTd(mode = 'serie')

lat <- 37.2
fSolD(lat, BTd[100])
fSolD(lat, BTd[100], method = 'strous')
fSolD(lat, BTd[100], method = 'spencer')
fSolD(lat, BTd[100], method = 'cooper')

lat <- -37.2
fSolD(lat, BTd[283])

#Solar angles along the year
SolD <- fSolD(lat, BTd = fBTd())

library(lattice)
xyplot(SolD)

#Calculation of the daylength for several latitudes
library(latticeExtra)

Lats <- c(-60, -40, -20, 0, 20, 40, 60)
NomLats <- ifelse(Lats > 0, paste(Lats,'N', sep = ''),
                  paste(abs(Lats), 'S', sep = ''))
NomLats[Lats == 0] <- '0'

BTd <- fBTd(mode = 'serie')
mat <- matrix(nrow = length(BTd), ncol = length(Lats))
colnames(mat) <- NomLats
WsZ <- data.table(Dates = BTd, mat)

for (i in seq_along(Lats)){
    SolDaux <- fSolD(lat = Lats[i], BTd = fBTd(mode = 'serie'));
    WsZ[,i+1] <- r2h(2*abs(SolDaux$ws))}

p = xyplot(`60S` + `40S` + `20S` + `0` + `20N` + `40N` + `60N` ~ Dates, data = WsZ, type = "l",
           ylab = expression(omega[s] * (h)))
plab = p+glayer(panel.text(x[1], y[1], NomLats[group.number], pos = 2))
print(plab)
}

\keyword{utilities}

