% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/buildsmdtape.R, R/manifoldtransform.R,
%   R/tapell.R, R/tapesmo.R
\name{buildsmdtape}
\alias{buildsmdtape}
\title{Build CppAD Tapes for Score Matching}
\usage{
buildsmdtape(
  start,
  tran = "identity",
  end = start,
  ll,
  ytape,
  usertheta,
  bdryw = "ones",
  acut = 1,
  thetatape_creator = function(n) {     seq(length.out = n) },
  verbose = FALSE
)
}
\arguments{
\item{start}{The starting manifold. Used for checking that \code{tran} and \code{man} match.}

\item{tran}{The name of a transformation. Available transformations are
\itemize{
\item ``sqrt''
\item ``alr''
\item ``clr''
\item ``none'' or `identity'
}}

\item{end}{The name of the manifold that \code{tran} maps \code{start} to. Available manifolds are:
\itemize{
\item ``sph'' unit sphere
\item ``Hn111'' hyperplane normal to the vector \eqn{1, 1, 1, 1, ...}
\item ``sim'' simplex
\item ``Euc'' Euclidean space
}}

\item{ll}{The name of an inbuilt improper log-likelihood function to tape (which also specifies the parametric model family). On Linux operating systems a custom log-likelihood function created by \code{\link[=customll]{customll()}} can also be used; the \code{ll} should operate on the untransformed (i.e. starting) manifold.}

\item{ytape}{An example measurement value to use for creating the tapes. In the natural (i.e. \code{start}) manifold of the log-likelihood function.
Please ensure that \code{ytape} is the interior of the manifold and non-zero.}

\item{usertheta}{A vector of parameter elements for the likelihood function. \code{NA} elements will become \emph{dynamic parameters}. Other elements will be fixed at the provided value. The length of \code{usertheta} must be the correct length for the log-likelihood - \strong{no checking is conducted}.}

\item{bdryw}{The name of the boundary weight function. "ones" for manifolds without boundary. For the simplex and positive orthant of the sphere, "prodsq" and "minsq" are possible - see \code{\link[=ppi]{ppi()}} for more information on these.}

\item{acut}{A parameter passed to the boundary weight function \code{bdryw}. Ignored for \code{bdryw = "ones"}.}

\item{thetatape_creator}{A function that accepts an integer \code{n}, and returns a vector of \code{n} length. The function is used to fill in the \code{NA} elements of \code{usertheta} when building the tapes. Please ensure that the values filled by \code{thetatape_creator} lead to plausible parameter vectors for the chosen log-likelihood.}

\item{verbose}{If \code{TRUE} more details are printed when taping. These details are for debugging and will likely be comprehensible only to users familiar with the source code of this package.}
}
\value{
A list of:
\itemize{
\item an \code{\link{ADFun}} object containing a tape of an improper likelihood with \eqn{z} on the \code{end} manifold as the independent variable
\item an \code{\link{ADFun}} object containing a tape of the score matching discrepancy function with the non-fixed parameters as the independent variable, and the measurements on the \code{end} manifold as the dynamic parameter.
\item some information about the tapes
}
}
\description{
For a parametric model family, the function \code{buildsmdtape()} generates \code{CppAD} tapes (called \code{ADFun}s) for the improper log-likelihood (without normalising constant) of the family and the score matching discrepancy function \eqn{A(z) + B(z) + C(z)} (defined in \code{\link{scorematchingtheory}}).
Three steps are performed by \code{buildsmdtape()}: first an object that specifies the manifold and any transformation to another manifold is created; then a tape of the log-likelihood (without normalising constant) is created; finally a tape of \eqn{A(z) + B(z) + C(z)} is created.
}
\details{
The improper log-likelihood (without normalising constant) must be implemented in \verb{C++} and is selected by name. Similarly the transforms of the manifold must be implemented in \verb{C++} and selected by name.

When using, \code{CppAD} one first creates \emph{tapes} of functions. These tapes can then be used for evaluating the function and its derivatives, and generating further tapes through argument swapping, differentiation and composition.
The taping relies on specifying typical argument values for the functions (see \strong{Introduction to CppAD Tapes} below).
Tapes can have both \emph{independent} variables and \emph{dynamic} parameters.
The differentiation with \code{CppAD} occurs with respect to the independent variables.
Tapes of tapes are possible, including tapes that swap the independent and dynamic variables - this is how this package differentiates with respect to a dynamic variables (see \code{\link[=tapeSwap]{tapeSwap()}}).

To build a tape for the score matching discrepancy function, the package first tapes the map from a point \eqn{z} on the \code{end} manifold to the value of the improper log-likelihood, where the independent variable is the \eqn{z}, the dynamic parameter is a vector of the parameters to estimate, and the remaining model parameters are fixed and not estimated.
This tape is then used to generate a tape for the score matching discrepancy function where the parameters to estimate are the independent variable.

Only some combinations of \code{start}, \code{tran} and \code{end} are available because \code{tran} must map between \code{start} and \code{end}.
These combinations of \code{start}-\code{tran}-\code{end} are currently available:
\itemize{
\item sim-sqrt-sph
\item sim-identity-sim
\item sim-alr-Euc
\item sim-clr-Hn111
\item sph-identity-sph
\item Euc-identity-Euc
}

Currently available improper log-likelihood functions are:
\itemize{
\item dirichlet
\item ppi
\item vMF
\item Bingham
\item FB
}
}
\section{Introduction to CppAD Tapes}{
This package uses version 2022000.2 of the algorithmic differentiation library \code{CppAD} \insertCite{bell2023cp}{scorematchingad} to build score matching estimators.
Full help for \code{CppAD} can be found at \url{https://cppad.readthedocs.io/}.

Differentiation proceeds by \emph{taping} the basic (\emph{atomic}) operations performed on the independent variables and dynamic parameters. The atomic operations include multiplication, division, addition, sine, cosine, exponential and many more.
Example values for the variables and parameters are used to conduct this taping, so care must be taken with any conditional (e.g. if-then) operations, and \href{https://cppad.readthedocs.io/}{\code{CppAD}} has a special tool for this called \code{CondExp} (short for \verb{conditional expressions}).
The result of taping is an object of class \code{ADFun} in \code{CppAD} and is often called a \emph{tape}.
This \code{ADFun} object can be evaluated, differentiated, used for further taping (via \code{CppAD}'s \code{base2ad()}), solving differential equations and more.
The differentiation is with respect to the independent variables, however the dynamic parameters can be altered which allows for creating a new \code{ADFun} object where the dynamic parameters become independent variables (see \code{\link[=tapeSwap]{tapeSwap()}}).
For the purposes of score matching, there are also \emph{fixed} parameters, which are the elements of the model's parameter vector that are given and not estimated.
}

\section{Warning: multiple CPU}{
Each time a tape is evaluated the corresponding \verb{C++} object is altered. Parallel use of the same \code{ADFun} object thus requires care and is not tested. For now I recommend creating a new \code{ADFun} object for each CPU.
}

\section{Warning}{
 There is no checking of the inputs \code{ytape} and \code{usertheta}.
}

\examples{
p <- 3
u <- rep(1/sqrt(p), p)
ltheta <- p #length of vMF parameter vector
intheta <- rep(NA, length.out = ltheta)
tapes <- buildsmdtape("sph", "identity", "sph", "vMF",
              ytape = u,
              usertheta = intheta,
              "ones", verbose = FALSE
              )
evaltape(tapes$lltape, u, runif(n = ltheta))
evaltape(tapes$smdtape, runif(n = ltheta), u)

u <- rep(1/3, 3)
tapes <- buildsmdtape("sim", "sqrt", "sph", "ppi",
              ytape = u,
              usertheta = ppi_paramvec(p = 3),
              bdryw = "minsq", acut = 0.01,
              verbose = FALSE
              )
evaltape(tapes$lltape, u, rppi_egmodel(1)$theta)
evaltape(tapes$smdtape, rppi_egmodel(1)$theta, u)
}
\references{
\insertAllCited{}
}
\seealso{
Other tape builders: 
\code{\link{moretapebuilders}}
}
\concept{tape builders}
