#' Count Overlap of ATAC-seq Fragments
#'
#' @param file Filename of the file for ATAC-seq fragments.
#' The file must be block gzipped (using the \code{bgzip} command)
#' and accompanied with the index file (made using the \code{tabix} command).
#' The uncompressed file must be a tab delimited file,
#' where each row represents one fragment.
#' The first four columns are chromosome name, start position, end position,
#' and barcode (i.e., name) of the cell including the fragment.
#' The remaining columns are ignored.
#' See vignette for details.
#' @param targetregions GRanges object for the regions where overlaps are counted.
#' Usually all of the autosomes.
#' If there is memory problem, split a chromosome into smaller chunks,
#' for example by 10 Mb.
#' The function loads each element of \code{targetregions} sequentially,
#' and smaller elements require less memory.
#' @param excluderegions GRanges object for the regions to be excluded.
#' Simple repeats in the genome should be listed here,
#' because repeats can cause false overlaps.
#' A fragment is discarded if its 5' or 3' end is located in \code{excluderegions}.
#' If \code{NULL}, fragments are not excluded by this criterion.
#' @param targetbarcodes Character vector for the barcodes of cells to be analyzed,
#' such as those passing quality control.
#' If \code{NULL}, all barcodes in the input file are analyzed.
#' @param Tn5offset Numeric vector of length two.
#' The enzyme for ATAC-seq is a homodimer of Tn5.
#' The transposition sites of two Tn5 proteins are 9 bp apart,
#' and the (representative) site of accessibility is in between.
#' If the start and end position of your input file is taken from BAM file,
#' set the paramater to \code{c(4, -5)} to adjust the offset.
#' Alternatively, values such as \code{c(0, -9)} could generate similar results;
#' what matters the most is the difference between the two numbers.
#' The fragments.tsv.gz file generated by 10x Cell Ranger already adjusts the shift
#' but is recorded as a BED file. In this case, use \code{c(1, 0)} (default value).
#' If unsure, set to \code{"guess"},
#' in which case the program returns a guess.
#' @return A tibble with each row corresponding to a cell.
#' For each cell, its barcode, the total count of the fragments \code{nfrag},
#' and the count distinguished by overlap depth are given.
#'
#' @importFrom dplyr arrange desc distinct group_by lag mutate n rename summarize ungroup
#' @importFrom GenomicRanges findOverlaps makeGRangesFromDataFrame
#' @importFrom rlang .data
#' @importFrom Rsamtools scanTabix TabixFile
#' @importFrom utils read.csv setTxtProgressBar txtProgressBar
#' @export
fragmentoverlapcount = function (file,
                                 targetregions,
                                 excluderegions = NULL,
                                 targetbarcodes = NULL,
                                 Tn5offset = c(1, 0)) {
  sumoverlaplist = list()
  tbx = TabixFile(file = file)
  pb = txtProgressBar(max = length(targetregions), style = 3)
  for (i in 1:length(targetregions)) {

    # Load fragments.
    res = scanTabix(tbx, param = targetregions[i])
    frags = read.csv(textConnection(res[[1]]),
                     sep = "\t",
                     header = FALSE)
    frags = frags[, 1:4]
    colnames(frags) = c("chr", "start", "end", "BC")

    if (! is.null(targetbarcodes)) {
      frags = frags[frags$BC %in% targetbarcodes, ]
    }
    if (nrow(frags) == 0) { next() }

    # Discard "semi-duplicate" fragments;
    # Hypothesized Tn5 transposition only at one strand.
    frags = frags %>%
      group_by(.data$BC) %>%
      # Between chrX:100-200 and chrX:100-300, only retain first
      arrange(.data$start, .data$end) %>%
      distinct(.data$start, .keep_all = TRUE) %>%
      # Between chrX:100-300 and chrX:200-300, only retain last
      arrange(desc(.data$end), desc(.data$start)) %>%
      distinct(.data$end, .keep_all = TRUE) %>%
      ungroup() %>%
      arrange(.data$start, .data$end, .data$BC)

    # Discard fragment if 5' or 3' is located in excluderegions.
    if (! is.null(excluderegions)) {
      query = makeGRangesFromDataFrame(
        data.frame(
          seqnames = frags$chr,
          start    = frags$start,
          end      = frags$start))
      x = is.na(findOverlaps(query, excluderegions, select = "first"))
      query = makeGRangesFromDataFrame(
        data.frame(
          seqnames = frags$chr,
          start    = frags$end,
          end      = frags$end))
      x = x & is.na(findOverlaps(query, excluderegions, select = "first"))
      frags = frags[x, ]
      if (nrow(frags) == 0) { next() }
    }

    # Adjust Tn5 site offset
    if (identical(Tn5offset, "guess")) {
      x = frags %>%
        group_by(.data$BC) %>%
        mutate(overlap = (lag(.data$end) - .data$start + 1)) %>%
        ungroup()
      x = x$overlap
      x = x[abs(x) <= 18]
      if (length(x) < 20) {
        stop('Error: datasize is too small for guessing Tn5offset')
      }
      x = table(x)
      x = as.numeric(names(x)[which.max(x)])
      x = c(0, -x)
      x = x - round(mean(x))
      setTxtProgressBar(pb, length(targetregions))
      close(pb)
      return(x)
    } else {
      frags$start = frags$start + Tn5offset[1]
      frags$end   = frags$end   + Tn5offset[2]
    }

    # Count overlap at 5' end of each fragment.
    frags = frags %>%
      group_by(.data$BC) %>%
      mutate(overlapcount = .overlapwithprecedingcount(.data$start, .data$end, TRUE)) %>%
      ungroup()

    # Summarize per BC.
    fragsbyBC = frags %>%
      group_by(.data$BC) %>%
      summarize(nfrags = n(),
                depth1 = sum(.data$overlapcount == 0),
                depth2 = sum(.data$overlapcount == 1),
                depth3 = sum(.data$overlapcount == 2),
                depth4 = sum(.data$overlapcount == 3),
                depth5 = sum(.data$overlapcount == 4),
                depth6 = sum(.data$overlapcount == 5))

    sumoverlaplist = c(sumoverlaplist, list(fragsbyBC))
    setTxtProgressBar(pb, i)
  }
  close(pb)
  if (identical(sumoverlaplist, list())) {
    stop('Error: no fragments remained after filtering')
  }

  sumoverlap =
    do.call(rbind, sumoverlaplist) %>%
    group_by(.data$BC) %>%
    summarize(nfrags = sum(.data$nfrags),
              depth1 = sum(.data$depth1),
              depth2 = sum(.data$depth2),
              depth3 = sum(.data$depth3),
              depth4 = sum(.data$depth4),
              depth5 = sum(.data$depth5),
              depth6 = sum(.data$depth6))
  sumoverlap = sumoverlap %>%
    rename(barcode = .data$BC)
  return(sumoverlap)
}

# A utility function.
# The fragments must be sorted by start, end.
.overlapwithprecedingcount =
  function (start, end, include) {
    ct = rep(NA, length(start))
    if (length(include) == 1) {
      include = rep(include, length(start))
    }
    unfinishedends = c()
    for (i in 1:length(start)) {
      if (!is.na(include[i]) & include[i]) {
        unfinishedends = unfinishedends[unfinishedends >= start[i]]
        ct[i] = length(unfinishedends)
        unfinishedends = c(unfinishedends, end[i])
      }
    }
    return(ct)
  }

#' Infer Ploidy from ATAC-seq Fragment Overlap
#'
#' @param fragmentoverlap Frequency of fragment overlap in each cell
#' computed by the function \code{fragmentoverlapcount}.
#' @param levels Possible values of ploidy. For example,
#' \code{c(2, 4)} if the cells can be diploids or tetraploids.
#' The values must be larger than one.
#' @param s Seed for random numbers used in EM algorithm.
#' @return A data.frame with each row corresponding to a cell.
#' For each cell, its barcode, ploidy inferred by moment method,
#' the same with additional K-means clustering,
#' and ploidy inferred by EM algorithm of mixture are given.
#' I recommend using \code{ploidy.moment}.
#'
#' @importFrom matrixStats rowMins
#' @importFrom mixtools multmixEM
#' @importFrom stats kmeans
#' @export
ploidy = function (fragmentoverlap,
                   levels,
                   s = 100) {
  if (min(levels) <= 1) {
    stop('Error: elements of levels must be larger than one')
  }

  ### MOMENT BASED METHOD
  # We model the overlapping of fragments by binomial distribution:
  # ------------------------------------------------------------
  # binomial distribution   | overlap of fragments   | parameter
  # ------------------------------------------------------------
  # one observation         | 5' end of a fragment   |
  # number of trials (size) | ploidy                 | p
  # number of success       | depth of overlap       |
  # probability of success  | probability of overlap | s
  # ------------------------------------------------------------
  # Under a predetermined p, for each cell, we estimate s based on
  # the overlap depth observed in the fragments belonging to the cell.
  # Since we cannot properly count observations with zero success,
  # we model as truncated binomial distribution.
  # We use the moment method in Paul R. Rider (1955).
  smat =
    do.call(
      cbind,
      lapply(
        levels,
        function (p) {
          x = as.matrix(fragmentoverlap[, -(1:2)])
          if (p < ncol(x)) {
            x = cbind(x[, 1:(p-1)], rowSums(x[, p:ncol(x)]))
          }
          T0 = x %*% rep(1, ncol(x))
          T1 = x %*% seq(1, ncol(x))
          T2 = x %*% (seq(1, ncol(x))^2)
          s = (T2 - T1) / ((p - 1) * T1)
          return(s)
        }))
  # For a cell with all fragments having depth 1,
  # T1 == T2 and s == 0.
  # We replace these with smallest non-zero s computed from all cells.
  for (i in 1:ncol(smat)) {
    smat[smat[, i] == 0, i] = min(smat[smat[, i] > 0, i])
  }
  # We seek s that is concordant across all cells.
  # In each cell, p is allowed to take any value from levels.
  smat = log10(smat)
  scandidates = sort(as.numeric(smat))
  scandidatespenalty =
    as.numeric(
      lapply(
        scandidates,
        function (s) {
          sum(matrixStats::rowMins(abs(smat - s)))
        }))
  soptimal = scandidates[which.min(scandidatespenalty)]
  # Adopting the probability soptimal,
  # infer ploidy of each cell.
  p.moment = apply(
    abs(smat - soptimal),
    1,
    which.min)
  p.moment = levels[p.moment]

  ### EM ALGORITHM FOR MIXTURES
  # We superficially (and possibly robustly) model
  # as mixtures of multinomial distributions
  # We first try with (depth2, depth3, depth4),
  # but if the clusters don't separate,
  # next try (depth3, depth4, depth5), and so on.
  for (j in 4:6) {
    sumoverlapsubmatrix =
      as.matrix(fragmentoverlap[, 0:2 + j])
    set.seed(s)
    em.out = multmixEM(
      y = sumoverlapsubmatrix,
      k = length(levels))
    if (max(em.out$lambda) < 0.99) { break }
  }
  p.em = apply(em.out$posterior, 1, which.max)
  # EM is simple clustering and unaware of the labeling in levels.
  # We infer the labeling from the last element of theta,
  # which represents the overlaps of largest depth used for clustering.
  p.em = (levels[order(em.out$theta[, 3])])[p.em]

  ### K-MEANS POST-PROCESSING OF MOMENT
  x = log10(
    (fragmentoverlap[, 4:6] + 1) / fragmentoverlap$nfrags)
  kmclust =
    kmeans(
      x,
      do.call(
        rbind,
        tapply(
          as.list(as.data.frame(t(x))),
          p.moment,
          function (x) {rowMeans(do.call(cbind, x))})))
  p.kmeans = levels[kmclust$cluster]

  return(data.frame(
    barcode = fragmentoverlap$barcode,
    ploidy.moment = p.moment,
    ploidy.kmeans = p.kmeans,
    ploidy.em = p.em))
}
