% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/make_test.R
\name{make_test}
\alias{make_test}
\title{Creating object representing a test}
\usage{
make_test(
  scoringMatrix,
  slopes,
  intercepts,
  mode = c("sequential", "simultaneous"),
  scoringOnPreviousResponses = NULL,
  editResponse = NULL,
  names = paste0("i", 1:nrow(slopes))
)
}
\arguments{
\item{scoringMatrix}{\emph{scoring matrix} that should be used for items,
especially one generated with \code{\link{make_scoring_matrix_aem}}}

\item{slopes}{matrix of slope parameters (items in rows, traits in cols),
especially generated with \code{\link{generate_slopes}}}

\item{intercepts}{matrix of intercept parameters (items in rows,
intercepts/thresholds in cols), especially generated with
\code{\link{generate_intercepts}}}

\item{mode}{a way the item should be answered - see
\code{\link{generate_item_responses_sqn}},
\code{\link{generate_item_responses_sml}}}

\item{scoringOnPreviousResponses}{optional function returning a column vector
that will be put before first column of the \code{scoringMatrix}}

\item{editResponse}{only if \code{mode='sequential'}: optional function
returning scoring matrix that should replace that provided by
\code{scoringMatrix} after \emph{response is made} at the first \emph{node};
this should be function accepting two arguments: \code{response} - generated
response (by the model described with the first column of the
\code{scoringMatrix}) that is supposed to be \emph{edited} and
\code{scoringMatrix} - current scoring matrix (to be replaced)}

\item{names}{optional character vector providing names of the items (by
default names will be created as concatenation of the letter "i" - like
"item" - and consecutive integers)}
}
\value{
List of objects of the \emph{rstylesItem} class.
}
\description{
Function makes a list object representing test items given
matrices of slopes and intercepts/thresholds parameters. Result may be used
within a call to \code{\link{generate_test_responses}}.
}
\details{
Function is actually a simple wrapper around \code{\link{make_item}}
- see documentation of this function for further details.

\strong{Column names of the \code{intercepts} matrix:}

\itemize{
  \item{If \code{mode = "simultaneous"} names should be of the form:
        \emph{dN} where \emph{d} stands for itself and \emph{N} are
        consecutive integers from 1 to one less than the number of rows of
        the the \emph{scoring matrix}.}
  \item{If \code{mode = "sequential"} names should be of the form:
        \emph{traitN} where \emph{trait} are names of traits - the same as
        those in columns of the \emph{scoring matrix} - and \emph{N} are
        integers describing consecutive thresholds of a \emph{pseudo-item}
        representing the given trait. In the most typical case with binary
        \emph{pseudo-items} this should be simply "1" (with in one column for
        each trait) but polytomous \emph{pseudo-items} need to be represented
        by the number of columns less by one than the number of possible
        responses, with \emph{N} being consecutive integers.}
}
}
\examples{
################################################################################
# responses to 10 items using 5-point Likert scale
# with respect to the Bockenholt's IRTree (i.e. "sequential") "MAE" model
# 1) make scoring matrix
sM <- make_scoring_matrix_aem(5, "mae")
# 2) generate items' slopes:
# slopes on the 'middle" and "extreme" latent traits set to 1 for all items
# and slopes on the "acquiescence" latent trait generated from a log-normal
# distribution with expected value of about 1.02 and standard deviation of
# about 0.21
slopes <-  cbind(generate_slopes(10, sM[, 1, drop = FALSE], 1),
                 generate_slopes(10, sM[, 2, drop = FALSE], FUN = rlnorm,
                                 meanlog = 0,
                                 sdlog = 0.2),
                 generate_slopes(10, sM[, 3, drop = FALSE], 1))
# 3) generate items' intercepts:
# intercepts generated from the uniform distribution with limits set to
# -1.5 and 1.5
intercepts <- generate_intercepts(10, sM, runif,
                                  list(min = -1.5,
                                       max = 1.5))
# 4) call `make_test()`
# (for IRTree mode must be set to "sequential")
test <- make_test(sM, slopes, intercepts, "sequential")

################################################################################
# responses to 20 items using 5-point Likert scale
# with respect to the Plieninger's "simultaneous" (partialy-compensatory) model
# 1) make scoring matrix
sM <- make_scoring_matrix_aem(5, "simultaneous")
# 2) generate items' slopes:
# slopes on the 'middle", "extreme" and "acquiescence" latent traits
# set to 1 for all items and slopes on the "intensity" latent trait generated
# from a normal distribution with expected value of 1 and standard deviation
# of 0.3
slopes <-  cbind(generate_slopes(20, sM[, 1, drop = FALSE], FUN = rnorm,
                                 mean = 1, sd = 0.3),
                 generate_slopes(20, sM[, -1], 1))
# 3) generate items' thresholds:
# thresholds generated from the normal distributon of items' (general)
# diificulty wit expected value of 0 and standardo deviation of 1.5 and the
# uniform distribution with limits -1 and 1 of categories' thresholds
# relative to item's difficulty
intercepts <- generate_intercepts(20, sM, FUNd = rnorm, FUNt = runif,
                                  argsd = list(mean = 0, sd = 1.5),
                                  argst = list(min = -1, max = 1))
# 4) call `make_test()`
test <- make_test(sM, slopes, intercepts, "simultaneous")
}
