## creation.r
##   - Functions for creating new GP individuals (individual initialization)
##
## RGP - a GP system for R
## 2010 Oliver Flasch (oliver.flasch@fh-koeln.de)
## with contributions of Thomas Bartz-Beielstein, Olaf Mersmann and Joerg Stork
## released under the GPL v2
##

##' @include stypes.r
NA

##' Creates an R expression by random growth
##'
##' Creates a random R expression by randomly growing its tree. In each step of growth,
##' with probability \code{subtreeprob}, an operator is chosen from the function set \code{funcset}.
##' The operands are then generated by recursive calls. If no subtree is generated, a constant will
##' be generated with probability \code{constprob}. If no constant is generated, an input variable
##' will be chosen randomly. The depth of the resulting expression trees can be bounded by the
##' \code{maxdepth} parameter.
##' \code{randexprFull} creates a random full expression tree of depth \code{maxdepth}. The algorithm
##' is the same as \code{randexprGrow}, with the exception that the probability of generating
##' a subtree is fixed to 1  until the desired tree depth \code{maxdepth} is reached.
##'
##' @param funcset The function set.
##' @param inset The set of input variables.
##' @param conset The set of constant factories.
##' @param maxdepth The maximum expression tree depth.
##' @param constprob The probability of generating a constant in a step of growth, if no subtree
##'   is generated. If neither a subtree nor a constant is generated, a randomly chosen input variable
##'   will be generated.
##' @param subtreeprob The probability of generating a subtree in a step of growth.
##' @param curdepth (internal) The depth of the random expression currently generated, used internally
##'   in recursive calls.
##' @return A new R expression generated by random growth.
##' @rdname randomExpressionCreation
##' @export
randexprGrow <- function(funcset, inset, conset,
                         maxdepth = 16,
                         constprob = 0.5, subtreeprob = 0.5,
                         curdepth = 1) {
  if (curdepth >= maxdepth) { # maximum depth reached, create terminal
    if (runif(1) <= constprob) { # create constant
      constfactory <- randelt(conset$all)
      constfactory()
    } else { # create input variable
      randelt(inset$all)
    }
  } else { # maximum depth not reached, create subtree or terminal
  	if (runif(1) <= subtreeprob) { # create subtree
      funcname <- randelt(funcset$all)
      funcarity <- arity(funcname)
      as.call(append(funcname,
                     lapply(1:funcarity, function(i) randexprGrow(funcset, inset, conset, maxdepth,
                                                                  constprob, subtreeprob, curdepth + 1))))
    } else { # create terminal
  	  if (runif(1) <= constprob) { # create constant
        constfactory <- randelt(conset$all)
        constfactory()
      } else { # create input variable
        randelt(inset$all)
      }
    }
  }
}

##' @rdname randomExpressionCreation
##' @export
randexprFull <- function(funcset, inset, conset,
                         maxdepth = 16,
                         constprob = 0.5) {
  randexprGrow(funcset, inset, conset, maxdepth, constprob, 1.0)
}

##' Creates an R function with a random expression as its body
##'
##' @param funcset The function set.
##' @param inset The set of input variables.
##' @param conset The set of constant factories.
##' @param maxdepth The maximum expression tree depth.
##' @param exprfactory The function to use for randomly creating the function's body.
##' @return A randomly generated R function.
##' @export
randfunc <- function(funcset, inset, conset, maxdepth = 16, exprfactory = randexprGrow) {
  newf <- new.function()
  formals(newf) <- new.alist(inset$all)
  body(newf) <- exprfactory(funcset, inset, conset, maxdepth)
  newf
}

##' Creates an R expression by random growth respecting type constraints
##'
##' Creates a random R expression by randomly growing its tree. In each step of growth,
##' with probability \code{subtreeprob}, an operator is chosen from the function set \code{funcset}.
##' The operands are then generated by recursive calls. If no subtree is generated, a constant will
##' be generated with probability \code{constprob}. If no constant is generated, an input variable
##' will be chosen randomly. The depth of the resulting expression trees can be bounded by the
##' \code{maxdepth} parameter.
##' In contrast to \code{randexprGrow}, this function respects sType tags of functions, input
##' variables, and constant factories. Only well-typed expressions are created.
##' \code{randexprTypedFull} creates a random full expression tree of depth \code{maxdepth},
##' respecting type constraints.
##' All nodes in the created expressions will be tagged with their sTypes.
##'
##' @param type The (range) type the created expression should have.
##' @param funcset The function set.
##' @param inset The set of input variables.
##' @param conset The set of constant factories.
##' @param maxdepth The maximum expression tree depth.
##' @param constprob The probability of generating a constant in a step of growth, if no subtree
##'   is generated. If neither a subtree nor a constant is generated, a randomly chosen input variable
##'   will be generated.
##' @param subtreeprob The probability of generating a subtree in a step of growth.
##' @param curdepth (internal) The depth of the random expression currently generated, used internally
##'   in recursive calls.
##' @return A new R expression generated by random growth.
##' @rdname randomExpressionCreationTyped
##' @export
randexprTypedGrow <- function(type, funcset, inset, conset,
                              maxdepth = 16,
                              constprob = 0.5, subtreeprob = 0.5,
                              curdepth = 1) {
  typeString <- type$string
  insetTypes <- Map(sType, inset$all)
  if (curdepth >= maxdepth) { # maximum depth reached, create terminal of correct type
    randterminalTyped(typeString, inset, conset, constprob) %::% type
  } else { # maximum depth not reached, create subtree or terminal
  	if (runif(1) <= subtreeprob) { # create subtree of correct type
      funcname <- randelt(funcset$byRange[[typeString]])
      if (is.null(funcname)) stop("Could not find a function of range type ", typeString, ".")
      functype <- sType(funcname)
      funcdomaintypes <- functype$domain
      newSubtree <-
        as.call(append(funcname,
                       Map(function(domaintype) randexprTypedGrow(domaintype, funcset, inset, conset, maxdepth,
                                                                  constprob, subtreeprob, curdepth + 1),
                           funcdomaintypes)))
      ## the type of the generated subtree is a function type with the input variable types as domain types...
      newSubtreeType <- insetTypes %->% type
      newSubtree %::% newSubtreeType
    } else { # create terminal of correct type
  	  randterminalTyped(typeString, inset, conset, constprob) %::% type
    }
  }
}

##' @rdname randomExpressionCreationTyped
##' @export
randexprTypedFull <- function(type, funcset, inset, conset,
                              maxdepth = 16,
                              constprob = 0.5) {
  randexprTypedGrow(type, funcset, inset, conset, maxdepth, constprob, 1.0)
}

##' Creates a well-typed R function with a random expression as its body
##'
##' @param type The range type of the random function to create.
##' @param funcset The function set.
##' @param inset The set of input variables.
##' @param conset The set of constant factories.
##' @param maxdepth The maximum expression tree depth.
##' @param exprfactory The function to use for randomly creating the function's body.
##' @return A randomly generated well-typed R function.
##' @export
randfuncTyped <- function(type, funcset, inset, conset, maxdepth = 16, exprfactory = randexprTypedGrow) {
  newf <- new.function()
  formals(newf) <- new.alist(inset$all)
  body(newf) <- exprfactory(type, funcset, inset, conset, maxdepth)
  newf
}

##' Create a random terminal node
##'
##' @param typeString The string label of the type of the random terminal node to create.
##' @param inset The set of input variables.
##' @param conset The set of constant factories.
##' @param constprob The probability of creating a constant versus an input variable.
##' @return A random terminal node, i.e. an input variable or a constant.
randterminalTyped <- function(typeString, inset, conset, constprob) {
  if (runif(1) <= constprob) { # create constant of correct type
    constfactory <- randelt(conset$byRange[[typeString]])
    if (is.null(constfactory)) stop("Could not find a constant factory for type ", typeString, ".")
    constfactory()
  } else { # create input variable of correct type
    invar <- randelt(inset$byRange[[typeString]])
    if (is.null(invar)) { # there are no input variables of the requested type, try to create a contant instead
      constfactory <- randelt(conset$byRange[[typeString]])
      if (is.null(constfactory)) stop("Could not find a constant factory for type ", typeString, ".")
      constfactory()
    } else {
      invar
    }
  }
}
