% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/R6Map.R
\name{R6Map}
\alias{R6Map}
\title{R6 class to display Earth Engine (EE) spatial objects}
\value{
Object of class \code{leaflet} and \code{EarthEngineMap}, with the
following extra parameters: tokens, name, opacity, shown, min, max, palette,
position, and legend. Use the $ method to retrieve the data (e.g., m$rgee$min).
}
\description{
Create interactive visualizations of spatial EE objects
(ee$Geometry, ee$Image, ee$Feature, and ee$FeatureCollection)
using \code{leaflet}.
}
\details{
\code{R6Map} uses the Earth Engine method
\href{https://developers.google.com/earth-engine/api_docs#ee.data.getmapid/}{
getMapId} to fetch and return an ID dictionary used to create
layers in a \code{leaflet} object. Users can specify visualization
parameters to Map$addLayer by using the visParams argument. Each Earth
Engine spatial object has a specific format. For
\code{ee$Image}, the
\href{https://developers.google.com/earth-engine/guides/image_visualization}{
parameters} available are:

\tabular{lll}{
\strong{Parameter}\tab \strong{Description}  \tab \strong{Type}\cr
\strong{bands}    \tab  Comma-delimited list of three band (RGB) \tab  list \cr
\strong{min}      \tab  Value(s) to map to 0 \tab  number or list of three
numbers, one for each band \cr
\strong{max}      \tab  Value(s) to map to 1 \tab  number or list of three
numbers, one for each band \cr
\strong{gain}     \tab  Value(s) by which to multiply each pixel value \tab
number or list of three numbers, one for each band \cr
\strong{bias}     \tab  Value(s) to add to each Digital Number
value \tab number or list of three numbers, one for each band \cr
\strong{gamma}    \tab  Gamma correction factor(s) \tab  number or list of
three numbers, one for each band \cr
\strong{palette}  \tab  List of CSS-style color strings
(single-band only) \tab  comma-separated list of hex strings \cr
\strong{opacity}   \tab  The opacity of the layer (from 0 to 1)  \tab  number \cr
}

If you add an \code{ee$Image} to Map$addLayer without any additional
parameters. By default it assigns the first three bands to red,
green, and blue bands, respectively. The default stretch is based on the
min-max range. On the other hand, the available parameters for
\code{ee$Geometry}, \code{ee$Feature}, and \code{ee$FeatureCollection}
are:

\itemize{
\item \strong{color}: A hex string in the format RRGGBB specifying the
color to use for drawing the features. By default #000000.
\item \strong{pointRadius}: The radius of the point markers. By default 3.
\item \strong{strokeWidth}: The width of lines and polygon borders. By
default 3.
}
}
\examples{

## ------------------------------------------------
## Method `R6Map$reset`
## ------------------------------------------------

\dontrun{
library(rgee)
ee_Initialize()

# Load an Image
image <- ee$Image("LANDSAT/LC08/C01/T1/LC08_044034_20140318")

# Create
Map <- R6Map$new()
Map$centerObject(image)

# Simple display: Map just will
Map$addLayer(
  eeObject = image,
  visParams = list(min=0, max = 10000, bands = c("B4", "B3", "B2")),
  name = "l8_01"
)
Map # display map

Map$reset() # Reset arguments
Map
}

## ------------------------------------------------
## Method `R6Map$setCenter`
## ------------------------------------------------

\dontrun{
library(rgee)

ee_Initialize()

Map <- R6Map$new()
Map$setCenter(lon = -76, lat = 0, zoom = 5)
Map

# Map$lat
# Map$lon
# Map$zoom
}

## ------------------------------------------------
## Method `R6Map$setZoom`
## ------------------------------------------------

\dontrun{
library(rgee)

ee_Initialize()

Map <- R6Map$new()
Map$setZoom(zoom = 4)
Map

# Map$lat
# Map$lon
# Map$zoom
}

## ------------------------------------------------
## Method `R6Map$centerObject`
## ------------------------------------------------

\dontrun{
library(rgee)

ee_Initialize()

Map <- R6Map$new()
image <- ee$Image("LANDSAT/LC08/C01/T1/LC08_044034_20140318")
Map$centerObject(image)
Map
}

## ------------------------------------------------
## Method `R6Map$addLayer`
## ------------------------------------------------

\dontrun{
library(rgee)
ee_Initialize()

# Load an Image
image <- ee$Image("LANDSAT/LC08/C01/T1/LC08_044034_20140318")

# Create
Map <- R6Map$new()
Map$centerObject(image)

# Simple display: Map just will
Map$addLayer(
  eeObject = image,
  visParams = list(min=0, max = 10000, bands = c("B4", "B3", "B2")),
  name = "l8_01"
)

Map$addLayer(
  eeObject = image,
  visParams = list(min=0, max = 20000, bands = c("B4", "B3", "B2")),
  name = "l8_02"
)

# Simple display: Map just will (if the position is not specified it will
# be saved on the right side)
Map$reset() # Reset Map to the initial arguments.
Map$centerObject(image)
Map$addLayer(
  eeObject = image,
  visParams = list(min=0, max=10000, bands = c("B4", "B3", "B2")),
  name = "l8_left",
  position = "left"
)

Map$addLayer(
  eeObject = image,
  visParams = list(min=0, max=20000, bands = c("B4", "B3", "B2")),
  name = "l8_right"
)

Map$reset()
}

## ------------------------------------------------
## Method `R6Map$addLayers`
## ------------------------------------------------

\dontrun{
library(sf)
library(rgee)
library(rgeeExtra)

ee_Initialize()

Map <- R6Map$new()

nc <- st_read(system.file("shape/nc.shp", package = "sf")) \%>\%
  st_transform(4326) \%>\%
  sf_as_ee()

ee_s2 <- ee$ImageCollection("COPERNICUS/S2")$
  filterDate("2016-01-01", "2016-01-31")$
  filterBounds(nc) \%>\%
  ee_get(0:2)

Map$centerObject(nc$geometry())
Map$addLayers(eeObject = ee_s2,position = "right")

# digging up the metadata
Map$previous_map_right$rgee$tokens

Map$reset()
}

## ------------------------------------------------
## Method `R6Map$addLegend`
## ------------------------------------------------

\dontrun{
library(leaflet)
library(rgee)
ee_Initialize()

Map$reset()

# Load MODIS ImageCollection
imgcol <- ee$ImageCollection$Dataset$MODIS_006_MOD13Q1

# Parameters for visualization
labels <- c("good", "marginal", "snow", "cloud")
cols   <- c("#999999", "#00BFC4", "#F8766D", "#C77CFF")
vis_qc <- list(min = 0, max = 3, palette = cols, bands = "SummaryQA", values = labels)

# Create interactive map
m_qc <- Map$addLayer(imgcol$median(), vis_qc, "QC")

# continous palette
Map$addLegend(vis_qc)

# categorical palette
Map$addLegend(vis_qc, name = "Legend1", color_mapping = "discrete")

# character palette
Map$addLegend(vis_qc, name = "Legend2", color_mapping = "character")
}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{lon}}{The longitude of the center, in degrees.}

\item{\code{lat}}{The latitude of the center, in degrees.}

\item{\code{zoom}}{The zoom level, from 1 to 24.}

\item{\code{save_maps}}{Should \code{R6Map} save the previous maps?. If TRUE, Map
will work in an OOP style. Otherwise it will be a functional programming
style.}

\item{\code{previous_map_left}}{Container on maps in the left side.}

\item{\code{previous_map_right}}{Container on maps in the right side.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{R6Map$new()}}
\item \href{#method-reset}{\code{R6Map$reset()}}
\item \href{#method-print}{\code{R6Map$print()}}
\item \href{#method-setCenter}{\code{R6Map$setCenter()}}
\item \href{#method-setZoom}{\code{R6Map$setZoom()}}
\item \href{#method-centerObject}{\code{R6Map$centerObject()}}
\item \href{#method-addLayer}{\code{R6Map$addLayer()}}
\item \href{#method-addLayers}{\code{R6Map$addLayers()}}
\item \href{#method-addLegend}{\code{R6Map$addLegend()}}
\item \href{#method-clone}{\code{R6Map$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Constructor of R6Map.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{R6Map$new(lon = 0, lat = 0, zoom = 1, save_maps = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{lon}}{The longitude of the center, in degrees. By default -76.942478.}

\item{\code{lat}}{The latitude of the center, in degrees. By default -12.172116.}

\item{\code{zoom}}{The zoom level, from 1 to 24. By default 18.}

\item{\code{save_maps}}{Should \code{R6Map} save previous maps?.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new \code{EarthEngineMap} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-reset"></a>}}
\if{latex}{\out{\hypertarget{method-reset}{}}}
\subsection{Method \code{reset()}}{
Reset to initial arguments.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{R6Map$reset(lon = 0, lat = 0, zoom = 1, save_maps = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{lon}}{The longitude of the center, in degrees. By default -76.942478.}

\item{\code{lat}}{The latitude of the center, in degrees. By default -12.172116.}

\item{\code{zoom}}{The zoom level, from 1 to 24. By default 18.}

\item{\code{save_maps}}{Should \code{R6Map} save previous maps?.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new \code{EarthEngineMap} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
library(rgee)
ee_Initialize()

# Load an Image
image <- ee$Image("LANDSAT/LC08/C01/T1/LC08_044034_20140318")

# Create
Map <- R6Map$new()
Map$centerObject(image)

# Simple display: Map just will
Map$addLayer(
  eeObject = image,
  visParams = list(min=0, max = 10000, bands = c("B4", "B3", "B2")),
  name = "l8_01"
)
Map # display map

Map$reset() # Reset arguments
Map
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-print"></a>}}
\if{latex}{\out{\hypertarget{method-print}{}}}
\subsection{Method \code{print()}}{
Display a \code{EarthEngineMap} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{R6Map$print()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
An \code{EarthEngineMap} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-setCenter"></a>}}
\if{latex}{\out{\hypertarget{method-setCenter}{}}}
\subsection{Method \code{setCenter()}}{
Centers the map view at the given coordinates with the given zoom level. If
no zoom level is provided, it uses 10 by default.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{R6Map$setCenter(lon = 0, lat = 0, zoom = 10)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{lon}}{The longitude of the center, in degrees. By default -76.942478.}

\item{\code{lat}}{The latitude of the center, in degrees. By default -12.172116.}

\item{\code{zoom}}{The zoom level, from 1 to 24. By default 18.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
No return value, called to set initial coordinates and zoom.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
library(rgee)

ee_Initialize()

Map <- R6Map$new()
Map$setCenter(lon = -76, lat = 0, zoom = 5)
Map

# Map$lat
# Map$lon
# Map$zoom
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-setZoom"></a>}}
\if{latex}{\out{\hypertarget{method-setZoom}{}}}
\subsection{Method \code{setZoom()}}{
Sets the zoom level of the map.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{R6Map$setZoom(zoom = 10)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{zoom}}{The zoom level, from 1 to 24. By default 10.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
No return value, called to set zoom.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
library(rgee)

ee_Initialize()

Map <- R6Map$new()
Map$setZoom(zoom = 4)
Map

# Map$lat
# Map$lon
# Map$zoom
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-centerObject"></a>}}
\if{latex}{\out{\hypertarget{method-centerObject}{}}}
\subsection{Method \code{centerObject()}}{
Centers the map view on a given object. If no zoom level is provided, it
will be predicted according to the bounds of the Earth Engine object
specified.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{R6Map$centerObject(
  eeObject,
  zoom = NULL,
  maxError = ee$ErrorMargin(1),
  titiler_server = "https://api.cogeo.xyz/"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{eeObject}}{Earth Engine spatial object.}

\item{\code{zoom}}{The zoom level, from 1 to 24. By default NULL.}

\item{\code{maxError}}{Max error when input image must be reprojected to an
explicitly requested result projection or geodesic state.}

\item{\code{titiler_server}}{TiTiler endpoint. Defaults to "https://api.cogeo.xyz/".}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
No return value, called to set zoom.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
library(rgee)

ee_Initialize()

Map <- R6Map$new()
image <- ee$Image("LANDSAT/LC08/C01/T1/LC08_044034_20140318")
Map$centerObject(image)
Map
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-addLayer"></a>}}
\if{latex}{\out{\hypertarget{method-addLayer}{}}}
\subsection{Method \code{addLayer()}}{
Adds a given Earth Engine spatial object to the map as a layer
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{R6Map$addLayer(
  eeObject,
  visParams = NULL,
  name = NULL,
  shown = TRUE,
  opacity = 1,
  position = NULL,
  titiler_viz_convert = TRUE,
  titiler_server = "https://api.cogeo.xyz/"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{eeObject}}{The Earth Engine spatial object to display in the interactive map.}

\item{\code{visParams}}{List of parameters for visualization. See details.}

\item{\code{name}}{The name of layers.}

\item{\code{shown}}{A flag indicating whether layers should be on by default.}

\item{\code{opacity}}{The layer's opacity is represented as a number between 0 and 1. Defaults to 1.}

\item{\code{position}}{Character. Activate panel creation. If "left" the map will be displayed in
the left panel. Otherwise, if it is "right" the map will be displayed in the right panel.
By default NULL (No panel will be created).}

\item{\code{titiler_viz_convert}}{Logical. If it is TRUE, Map$addLayer will transform the
visParams to titiler style. Ignored if eeObject is not a COG file.}

\item{\code{titiler_server}}{TiTiler endpoint. Defaults to "https://api.cogeo.xyz/".}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An \code{EarthEngineMap} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
library(rgee)
ee_Initialize()

# Load an Image
image <- ee$Image("LANDSAT/LC08/C01/T1/LC08_044034_20140318")

# Create
Map <- R6Map$new()
Map$centerObject(image)

# Simple display: Map just will
Map$addLayer(
  eeObject = image,
  visParams = list(min=0, max = 10000, bands = c("B4", "B3", "B2")),
  name = "l8_01"
)

Map$addLayer(
  eeObject = image,
  visParams = list(min=0, max = 20000, bands = c("B4", "B3", "B2")),
  name = "l8_02"
)

# Simple display: Map just will (if the position is not specified it will
# be saved on the right side)
Map$reset() # Reset Map to the initial arguments.
Map$centerObject(image)
Map$addLayer(
  eeObject = image,
  visParams = list(min=0, max=10000, bands = c("B4", "B3", "B2")),
  name = "l8_left",
  position = "left"
)

Map$addLayer(
  eeObject = image,
  visParams = list(min=0, max=20000, bands = c("B4", "B3", "B2")),
  name = "l8_right"
)

Map$reset()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-addLayers"></a>}}
\if{latex}{\out{\hypertarget{method-addLayers}{}}}
\subsection{Method \code{addLayers()}}{
Adds a given ee$ImageCollection to the map as multiple layers.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{R6Map$addLayers(
  eeObject,
  visParams = NULL,
  nmax = 5,
  name = NULL,
  shown = TRUE,
  position = NULL,
  opacity = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{eeObject}}{ee$ImageCollection to display in the interactive map.}

\item{\code{visParams}}{List of parameters for visualization. See details.}

\item{\code{nmax}}{Numeric. The maximum number of images to display. By default 5.}

\item{\code{name}}{The name of layers.}

\item{\code{shown}}{A flag indicating whether layers should be on by default.}

\item{\code{position}}{Character. Activate panel creation. If "left" the map will be displayed in
the left panel. Otherwise, if it is "right" the map will be displayed in the right panel.
By default NULL (No panel will be created).}

\item{\code{opacity}}{The layer's opacity is represented as a number between 0 and 1. Defaults to 1.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{EarthEngineMap} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
library(sf)
library(rgee)
library(rgeeExtra)

ee_Initialize()

Map <- R6Map$new()

nc <- st_read(system.file("shape/nc.shp", package = "sf")) \%>\%
  st_transform(4326) \%>\%
  sf_as_ee()

ee_s2 <- ee$ImageCollection("COPERNICUS/S2")$
  filterDate("2016-01-01", "2016-01-31")$
  filterBounds(nc) \%>\%
  ee_get(0:2)

Map$centerObject(nc$geometry())
Map$addLayers(eeObject = ee_s2,position = "right")

# digging up the metadata
Map$previous_map_right$rgee$tokens

Map$reset()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-addLegend"></a>}}
\if{latex}{\out{\hypertarget{method-addLegend}{}}}
\subsection{Method \code{addLegend()}}{
Adds a color legend to an EarthEngineMap.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{R6Map$addLegend(
  visParams,
  name = "Legend",
  position = c("bottomright", "topright", "bottomleft", "topleft"),
  color_mapping = "numeric",
  opacity = 1,
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{visParams}}{List of parameters for visualization.}

\item{\code{name}}{The title of the legend.}

\item{\code{position}}{Character. The position of the legend. By default bottomright.}

\item{\code{color_mapping}}{Map data values (numeric or factor/character) to
colors according to a given palette. Use "numeric" ("discrete") for continuous
(categorical) data. For display characters use "character" and add to visParams
the element "values" containing the desired character names.}

\item{\code{opacity}}{The legend's opacity is represented as a number between 0
and 1. Defaults to 1.}

\item{\code{...}}{Extra legend creator arguments. See \link[leaflet]{addLegend}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{EarthEngineMap} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
library(leaflet)
library(rgee)
ee_Initialize()

Map$reset()

# Load MODIS ImageCollection
imgcol <- ee$ImageCollection$Dataset$MODIS_006_MOD13Q1

# Parameters for visualization
labels <- c("good", "marginal", "snow", "cloud")
cols   <- c("#999999", "#00BFC4", "#F8766D", "#C77CFF")
vis_qc <- list(min = 0, max = 3, palette = cols, bands = "SummaryQA", values = labels)

# Create interactive map
m_qc <- Map$addLayer(imgcol$median(), vis_qc, "QC")

# continous palette
Map$addLegend(vis_qc)

# categorical palette
Map$addLegend(vis_qc, name = "Legend1", color_mapping = "discrete")

# character palette
Map$addLegend(vis_qc, name = "Legend2", color_mapping = "character")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{R6Map$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
