% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kalseries.r
\name{kalseries}
\alias{deviance.kalseries}
\alias{fitted.kalseries}
\alias{kalseries}
\alias{print.kalseries}
\alias{residuals.kalseries}
\title{Repeated Measurements Models for Continuous Variables with Frailty or
Serial Dependence}
\usage{
kalseries(response = NULL, times = NULL, intensity = "exponential",
  depend = "independence", mu = NULL, shape = NULL, density = FALSE,
  ccov = NULL, tvcov = NULL, torder = 0, interaction = NULL,
  preg = NULL, ptvc = NULL, pintercept = NULL, pshape = NULL,
  pinitial = 1, pdepend = NULL, pfamily = NULL, delta = NULL,
  transform = "identity", link = "identity", envir = parent.frame(),
  print.level = 0, ndigit = 10, gradtol = 1e-05, steptol = 1e-05,
  fscale = 1, iterlim = 100, typsize = abs(p), stepmax = 10 * sqrt(p
  \%*\% p))
}
\arguments{
\item{response}{A list of two column matrices with responses and
corresponding times for each individual, one matrix or dataframe of
response values, or an object of class, \code{response} (created by
\code{\link[rmutil]{restovec}}) or \code{repeated} (created by
\code{\link[rmutil]{rmna}} or \code{\link[rmutil]{lvna}}). If the
\code{repeated} data object contains more than one response variable, give
that object in \code{envir} and give the name of the response variable to
be used here.}

\item{times}{When response is a matrix, a vector of possibly unequally
spaced times when they are the same for all individuals or a matrix of
times. Not necessary if equally spaced. Ignored if response has class,
\code{response} or \code{repeated}.}

\item{intensity}{The form of function to be put in the Pareto distribution.
Choices are exponential, Weibull, gamma, normal, logistic, Cauchy, log
normal, log logistic, log Cauchy, log Student, inverse Gauss, and
gen(eralized) logistic. (For definitions of distributions, see the
corresponding [dpqr]distribution help.)}

\item{depend}{Type of dependence. Choices are \code{independence},
\code{Markov}, \code{serial}, and \code{frailty}.}

\item{mu}{A regression function for the location parameter or a formula
beginning with ~, specifying either a linear regression function in the
Wilkinson and Rogers notation or a general function with named unknown
parameters. Give the initial estimates in \code{preg} if there are no
time-varying covariates and in \code{ptvc} if there are.}

\item{shape}{A regression function for the shape parameter or a formula
beginning with ~, specifying either a linear regression function in the
Wilkinson and Rogers notation or a general function with named unknown
parameters. It must yield one value per observation.}

\item{density}{If TRUE, the density of the function specified in
\code{intensity} is used instead of the intensity.}

\item{ccov}{A vector or matrix containing time-constant baseline covariates
with one row per individual, a model formula using vectors of the same
size, or an object of class, \code{tccov} (created by
\code{\link[rmutil]{tcctomat}}). If response has class, \code{repeated},
the covariates must be supplied as a Wilkinson and Rogers formula unless
none are to be used or \code{mu} is given.}

\item{tvcov}{A list of matrices with time-varying covariate values,
observed at the event times in \code{response}, for each individual (one
column per variable), one matrix or dataframe of such covariate values, or
an object of class, \code{tvcov} (created by
\code{\link[rmutil]{tvctomat}}). If a time-varying covariate is observed at
arbitrary time, \code{\link[rmutil]{gettvc}} can be used to find the most
recent values for each response and create a suitable list. If response has
class, \code{repeated}, the covariates must be supplied as a Wilkinson and
Rogers formula unless none are to be used or \code{mu} is given.}

\item{torder}{The order of the polynomial in time to be fitted.}

\item{interaction}{Vector of length equal to the number of time-constant
covariates, giving the levels of interactions between them and the
polynomial in time in the \code{linear model}.}

\item{preg}{Initial parameter estimates for the regression model:
intercept, one for each covariate in \code{ccov}, and \code{torder} plus
sum(\code{interaction}). If \code{mu} is a formula or function, the
parameter estimates must be given here only if there are no time-varying
covariates. If \code{mu} is a formula with unknown parameters, their
estimates must be supplied either in their order of appearance in the
expression or in a named list.}

\item{ptvc}{Initial parameter estimates for the coefficients of the
time-varying covariates, as many as in \code{tvcov}. If \code{mu} is a
formula or function, the parameter estimates must be given here if there
are time-varying covariates present.}

\item{pintercept}{The initial estimate of the intercept for the generalized
logistic intensity.}

\item{pshape}{An initial estimate for the shape parameter of the intensity
function (except exponential intensity). If \code{shape} is a function or
formula, the corresponding initial estimates. If \code{shape} is a formula
with unknown parameters, their estimates must be supplied either in their
order of appearance in the expression or in a named list.}

\item{pinitial}{An initial estimate for the initial parameter. With
\code{frailty} dependence, this is the frailty parameter.}

\item{pdepend}{An initial estimate for the serial dependence parameter. For
\code{frailty} dependence, if a value is given here, an autoregression is
fitted as well as the frailty.}

\item{pfamily}{An optional initial estimate for the second parameter of a
two-parameter power variance family mixture instead of the default gamma
mixture. This yields a gamma mixture as \code{family -> 0}, an inverse
Gauss mixture for \code{family = 0.5}, and a compound distribution of a
Poisson-distributed number of gamma distributions for \code{-1 < family <
0}.}

\item{delta}{Scalar or vector giving the unit of measurement for each
response value, set to unity by default. For example, if a response is
measured to two decimals, delta=0.01. If the response has been
pretransformed, this must be multiplied by the Jacobian. This
transformation cannot contain unknown parameters. For example, with a log
transformation, \code{delta=1/y}. The jacobian is calculated automatically
for the transform option. Ignored if response has class, \code{response} or
\code{repeated}.}

\item{transform}{Transformation of the response variable: \code{identity},
\code{exp}, \code{square}, \code{sqrt}, or \code{log}.}

\item{link}{Link function for the mean: \code{identity}, \code{exp},
\code{square}, \code{sqrt}, or \code{log}.}

\item{envir}{Environment in which model formulae are to be interpreted or a
data object of class, \code{repeated}, \code{tccov}, or \code{tvcov}; the
name of the response variable should be given in \code{response}. If
\code{response} has class \code{repeated}, it is used as the environment.}

\item{print.level}{Arguments for nlm.}

\item{ndigit}{Arguments for nlm.}

\item{gradtol}{Arguments for nlm.}

\item{steptol}{Arguments for nlm.}

\item{fscale}{Arguments for nlm.}

\item{iterlim}{Arguments for nlm.}

\item{typsize}{Arguments for nlm.}

\item{stepmax}{Arguments for nlm.}
}
\value{
A list of classes \code{kalseries} and \code{recursive} is
returned.
}
\description{
\code{kalseries} is designed to handle repeated measurements models with
time-varying covariates. The distributions have two extra parameters as
compared to the functions specified by \code{intensity} and are generally
longer tailed than those distributions. Dependence among observations on a
unit can be through gamma or power variance family frailties (a type of
random effect), with or without autoregression, or one of two types of
serial dependence over time.
}
\details{
By default, a gamma mixture of the distribution specified in
\code{intensity} is used, as the conditional distribution in the
\code{Markov} and \code{serial} dependence models, and as a symmetric
multivariate (random effect) model for \code{frailty} dependence. For
example, with a Weibull \code{intensity} and \code{frailty} dependence,
this yields a multivariate Burr distribution and with \code{Markov} or
\code{serial} dependence, univariate Burr conditional distributions.

If a value for \code{pfamily} is used, the gamma mixture is replaced by a
power variance family mixture.

Nonlinear regression models can be supplied as formulae where parameters
are unknowns in which case factor variables cannot be used and parameters
must be scalars. (See \code{\link[rmutil]{finterp}}.)

Marginal and individual profiles can be plotted using
\code{\link[rmutil]{mprofile}} and \code{\link[rmutil]{iprofile}} and
residuals with \code{\link[rmutil]{plot.residuals}}.
}
\examples{

treat <- c(0,0,1,1)
tr <- tcctomat(treat)
dose <- matrix(rpois(20,10), ncol=5)
dd <- tvctomat(dose)
y <- restovec(matrix(rnorm(20), ncol=5), name="y")
reps <- rmna(y, ccov=tr, tvcov=dd)
#
# normal intensity, independence model
kalseries(y, intensity="normal", dep="independence", preg=1, pshape=5)
\dontrun{
# random effect
kalseries(y, intensity="normal", dep="frailty", preg=1, pinitial=1, psh=5)
# serial dependence
kalseries(y, intensity="normal", dep="serial", preg=1, pinitial=1,
	pdep=0.1, psh=5)
# random effect and autoregression
kalseries(y, intensity="normal", dep="frailty", preg=1, pinitial=1,
	pdep=0.1, psh=5)
#
# add time-constant variable
kalseries(y, intensity="normal", dep="serial", pinitial=1,
	pdep=0.1, psh=5, preg=c(1,0), ccov=treat)
# or equivalently
kalseries(y, intensity="normal", mu=~treat, dep="serial", pinitial=1,
	pdep=0.1, psh=5, preg=c(1,0))
# or
kalseries(y, intensity="normal", mu=~b0+b1*treat, dep="serial",
	pinitial=1, pdep=0.1, psh=5, preg=c(1,0), envir=reps)
#
# add time-varying variable
kalseries(y, intensity="normal", dep="serial", pinitial=1, pdep=0.1,
	psh=5, preg=c(1,0), ccov=treat, ptvc=0, tvc=dose)
# or equivalently, from the environment
dosev <- as.vector(t(dose))
kalseries(y, intensity="normal",
	mu=~b0+b1*rep(treat,rep(5,4))+b2*dosev,
	dep="serial", pinitial=1, pdep=0.1, psh=5, ptvc=c(1,0,0))
# or from the reps data object
kalseries(y, intensity="normal", mu=~b0+b1*treat+b2*dose,
	dep="serial", pinitial=1, pdep=0.1, psh=5, ptvc=c(1,0,0),
	envir=reps)
# try power variance family instead of gamma distribution for mixture
kalseries(y, intensity="normal", mu=~b0+b1*treat+b2*dose,
	dep="serial", pinitial=1, pdep=0.1, psh=5, ptvc=c(1,0,0),
	pfamily=0.1, envir=reps)
# first-order one-compartment model
# data objects for formulae
dose <- c(2,5)
dd <- tcctomat(dose)
times <- matrix(rep(1:20,2), nrow=2, byrow=TRUE)
tt <- tvctomat(times)
# vector covariates for functions
dose <- c(rep(2,20),rep(5,20))
times <- rep(1:20,2)
# functions
mu <- function(p) exp(p[1]-p[3])*(dose/(exp(p[1])-exp(p[2]))*
	(exp(-exp(p[2])*times)-exp(-exp(p[1])*times)))
shape <- function(p) exp(p[1]-p[2])*times*dose*exp(-exp(p[1])*times)
# response
conc <- matrix(rgamma(40,shape(log(c(0.01,1))),
	scale=mu(log(c(1,0.3,0.2))))/shape(log(c(0.1,0.4))),ncol=20,byrow=TRUE)
conc[,2:20] <- conc[,2:20]+0.5*(conc[,1:19]-matrix(mu(log(c(1,0.3,0.2))),
	ncol=20,byrow=TRUE)[,1:19])
conc <- restovec(ifelse(conc>0,conc,0.01))
reps <- rmna(conc, ccov=dd, tvcov=tt)
#
# constant shape parameter
kalseries(reps, intensity="gamma", dep="independence", mu=mu,
	ptvc=c(-1,-1.1,-1), pshape=1.5)
# or
kalseries(reps, intensity="gamma", dep="independence",
	mu=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*times)-exp(-exp(absorption)*times)),
	ptvc=list(absorption=-1,elimination=-1.1,volume=-1),
	pshape=1.2)
# add serial dependence
kalseries(reps, intensity="gamma", dep="serial", pdep=0.9,
	mu=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*times)-exp(-exp(absorption)*times)),
	ptvc=list(absorption=-1,elimination=-1.1,volume=-1),
	pshape=0.2)
# time dependent shape parameter
kalseries(reps, intensity="gamma", dep="independence", mu=mu,
	shape=shape, ptvc=c(-1,-1.1,-1), pshape=c(-3,0))
# or
kalseries(reps, intensity="gamma", dep="independence",
	mu=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*times)-exp(-exp(absorption)*times)),
	ptvc=list(absorption=-1,elimination=-1.1,volume=-1),
	shape=~exp(b1-b2)*times*dose*exp(-exp(b1)*times),
	pshape=list(b1=-3,b2=0))
# add serial dependence
kalseries(reps, intensity="gamma", dep="serial", pdep=0.5,
	mu=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*times)-exp(-exp(absorption)*times)),
	ptvc=list(absorption=-1,elimination=-1.1,volume=-1),
	shape=~exp(b1-b2)*times*dose*exp(-exp(b1)*times),
	pshape=list(b1=-3,b2=0))
}
}
\author{
J.K. Lindsey
}
\keyword{models}

