/*
 * Decompiled with CFR 0.152.
 */
package org.openscience.cdk.smiles;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import org.openscience.cdk.annotations.TestClass;
import org.openscience.cdk.annotations.TestMethod;
import org.openscience.cdk.config.IsotopeFactory;
import org.openscience.cdk.config.Isotopes;
import org.openscience.cdk.exception.CDKException;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.interfaces.IBond;
import org.openscience.cdk.interfaces.IDoubleBondStereochemistry;
import org.openscience.cdk.interfaces.IIsotope;
import org.openscience.cdk.interfaces.IStereoElement;
import org.openscience.cdk.interfaces.ITetrahedralChirality;
import org.openscience.cdk.stereo.ExtendedTetrahedral;
import uk.ac.ebi.beam.Atom;
import uk.ac.ebi.beam.AtomBuilder;
import uk.ac.ebi.beam.Bond;
import uk.ac.ebi.beam.Configuration;
import uk.ac.ebi.beam.Edge;
import uk.ac.ebi.beam.Element;
import uk.ac.ebi.beam.Graph;
import uk.ac.ebi.beam.GraphBuilder;

@TestClass(value="org.openscience.cdk.smiles.CDKToBeamTest")
final class CDKToBeam {
    private final boolean isomeric;
    private final boolean aromatic;
    private final boolean atomClasses;

    CDKToBeam() {
        this(true, true);
    }

    CDKToBeam(boolean isomeric) {
        this(isomeric, true);
    }

    CDKToBeam(boolean isomeric, boolean aromatic) {
        this(isomeric, aromatic, true);
    }

    CDKToBeam(boolean isomeric, boolean aromatic, boolean atomClasses) {
        this.isomeric = isomeric;
        this.aromatic = aromatic;
        this.atomClasses = atomClasses;
    }

    @TestMethod(value="adenine,benzene,imidazole")
    Graph toBeamGraph(IAtomContainer ac) throws CDKException {
        int order = ac.getAtomCount();
        GraphBuilder gb = GraphBuilder.create(order);
        HashMap<IAtom, Integer> indices = Maps.newHashMapWithExpectedSize(order);
        for (IAtom a : ac.atoms()) {
            indices.put(a, indices.size());
            gb.add(this.toBeamAtom(a));
        }
        for (IBond b : ac.bonds()) {
            gb.add(this.toBeamEdge(b, indices));
        }
        if (this.isomeric) {
            for (IStereoElement se : ac.stereoElements()) {
                if (se instanceof ITetrahedralChirality) {
                    this.addTetrahedralConfiguration((ITetrahedralChirality)se, gb, indices);
                    continue;
                }
                if (se instanceof IDoubleBondStereochemistry) {
                    this.addGeometricConfiguration((IDoubleBondStereochemistry)se, gb, indices);
                    continue;
                }
                if (!(se instanceof ExtendedTetrahedral)) continue;
                this.addExtendedTetrahedralConfiguration((ExtendedTetrahedral)se, gb, indices);
            }
        }
        return gb.build();
    }

    @TestMethod(value="aliphaticAtom,aromaticAtom")
    Atom toBeamAtom(IAtom a) {
        Integer massNumber;
        boolean aromatic = this.aromatic && a.getFlag(32);
        Integer charge = a.getFormalCharge();
        String symbol = Preconditions.checkNotNull(a.getSymbol(), "An atom had an undefined symbol");
        Element element = Element.ofSymbol(symbol);
        if (element == null) {
            element = Element.Unknown;
        }
        AtomBuilder ab = aromatic ? AtomBuilder.aromatic(element) : AtomBuilder.aliphatic(element);
        Integer hCount = a.getImplicitHydrogenCount();
        if (element == Element.Unknown) {
            ab.hydrogens(hCount != null ? hCount : 0);
        } else {
            ab.hydrogens(Preconditions.checkNotNull(hCount, "One or more atoms had an undefined number of implicit hydrogens"));
        }
        if (charge != null) {
            ab.charge(charge);
        }
        if (this.isomeric && (massNumber = a.getMassNumber()) != null) {
            try {
                Isotopes isotopes = Isotopes.getInstance();
                IIsotope isotope = ((IsotopeFactory)isotopes).getMajorIsotope(a.getSymbol());
                if (isotope == null || !isotope.getMassNumber().equals(massNumber)) {
                    ab.isotope(massNumber);
                }
            }
            catch (IOException e) {
                throw new InternalError("Isotope factory wouldn't load: " + e.getMessage());
            }
        }
        Integer atomClass = (Integer)a.getProperty("cdk:AtomAtomMapping");
        if (this.atomClasses && atomClass != null) {
            ab.atomClass(atomClass);
        }
        return ab.build();
    }

    @TestMethod(value="singleBond,doubleBond,tripleBond")
    Edge toBeamEdge(IBond b, Map<IAtom, Integer> indices) throws CDKException {
        Preconditions.checkArgument(b.getAtomCount() == 2, "Invalid number of atoms on bond");
        int u = indices.get(b.getAtom(0));
        int v = indices.get(b.getAtom(1));
        return this.toBeamEdgeLabel(b).edge(u, v);
    }

    private Bond toBeamEdgeLabel(IBond b) throws CDKException {
        if (this.aromatic && b.getFlag(32)) {
            return Bond.AROMATIC;
        }
        if (b.getOrder() == null) {
            throw new CDKException("A bond had undefined order, possible query bond?");
        }
        IBond.Order order = b.getOrder();
        switch (order) {
            case SINGLE: {
                return Bond.SINGLE;
            }
            case DOUBLE: {
                return Bond.DOUBLE;
            }
            case TRIPLE: {
                return Bond.TRIPLE;
            }
            case QUADRUPLE: {
                return Bond.QUADRUPLE;
            }
        }
        throw new CDKException("Unsupported bond order: " + (Object)((Object)order));
    }

    private void addGeometricConfiguration(IDoubleBondStereochemistry dbs, GraphBuilder gb, Map<IAtom, Integer> indices) {
        IBond db = dbs.getStereoBond();
        IBond[] bs = dbs.getBonds();
        if (this.aromatic && db.getFlag(32)) {
            return;
        }
        int u = indices.get(db.getAtom(0));
        int v = indices.get(db.getAtom(1));
        int x = indices.get(bs[0].getConnectedAtom(db.getAtom(0)));
        int y = indices.get(bs[1].getConnectedAtom(db.getAtom(1)));
        if (dbs.getStereo() == IDoubleBondStereochemistry.Conformation.TOGETHER) {
            gb.geometric(u, v).together(x, y);
        } else {
            gb.geometric(u, v).opposite(x, y);
        }
    }

    private void addTetrahedralConfiguration(ITetrahedralChirality tc, GraphBuilder gb, Map<IAtom, Integer> indices) {
        IAtom[] ligands = tc.getLigands();
        int u = indices.get(tc.getChiralAtom());
        int[] vs = new int[]{indices.get(ligands[0]), indices.get(ligands[1]), indices.get(ligands[2]), indices.get(ligands[3])};
        gb.tetrahedral(u).lookingFrom(vs[0]).neighbors(vs[1], vs[2], vs[3]).winding(tc.getStereo() == ITetrahedralChirality.Stereo.CLOCKWISE ? Configuration.CLOCKWISE : Configuration.ANTI_CLOCKWISE).build();
    }

    private void addExtendedTetrahedralConfiguration(ExtendedTetrahedral et, GraphBuilder gb, Map<IAtom, Integer> indices) {
        IAtom[] ligands = et.peripherals();
        int u = indices.get(et.focus());
        int[] vs = new int[]{indices.get(ligands[0]), indices.get(ligands[1]), indices.get(ligands[2]), indices.get(ligands[3])};
        gb.extendedTetrahedral(u).lookingFrom(vs[0]).neighbors(vs[1], vs[2], vs[3]).winding(et.winding() == ITetrahedralChirality.Stereo.CLOCKWISE ? Configuration.CLOCKWISE : Configuration.ANTI_CLOCKWISE).build();
    }
}

