/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.ebi.beam;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import uk.ac.ebi.beam.Atom;
import uk.ac.ebi.beam.AtomImpl;
import uk.ac.ebi.beam.Bond;
import uk.ac.ebi.beam.CharBuffer;
import uk.ac.ebi.beam.Configuration;
import uk.ac.ebi.beam.Edge;
import uk.ac.ebi.beam.Element;
import uk.ac.ebi.beam.Graph;
import uk.ac.ebi.beam.IntStack;
import uk.ac.ebi.beam.InvalidSmilesException;
import uk.ac.ebi.beam.Topology;

final class Parser {
    private final IntStack stack = new IntStack(10);
    private final Graph g;
    private RingBond[] rings = new RingBond[10];
    private Map<Integer, LocalArrangement> arrangement = new HashMap<Integer, LocalArrangement>(5);
    private Map<Integer, Configuration> configurations = new HashMap<Integer, Configuration>(5);
    private Bond bond = Bond.IMPLICIT;
    private Configuration configuration = Configuration.UNKNOWN;
    private Set<Integer> start = new TreeSet<Integer>();
    private int openRings = 0;
    private final boolean strict;
    private BitSet checkDirectionalBonds = new BitSet();

    Parser(CharBuffer buffer, boolean strict) throws InvalidSmilesException {
        this.strict = strict;
        this.g = new Graph(1 + 2 * (buffer.length() / 3));
        this.readSmiles(buffer);
        if (this.openRings > 0) {
            throw new InvalidSmilesException("Unclosed ring detected:", buffer);
        }
        if (this.stack.size() > 1) {
            throw new InvalidSmilesException("Unclosed branch detected:", buffer);
        }
        this.start.add(0);
        if (this.g.getFlags(14) != 0) {
            this.createTopologies(buffer);
        }
    }

    Parser(String str) throws InvalidSmilesException {
        this(CharBuffer.fromString(str), false);
    }

    static Graph strict(String str) throws InvalidSmilesException {
        return new Parser(CharBuffer.fromString(str), true).molecule();
    }

    static Graph losse(String str) throws InvalidSmilesException {
        return new Parser(CharBuffer.fromString(str), false).molecule();
    }

    Graph molecule() {
        return this.g;
    }

    private void createTopologies(CharBuffer buffer) throws InvalidSmilesException {
        for (Map.Entry<Integer, Configuration> e : this.configurations.entrySet()) {
            this.addTopology(e.getKey(), Topology.toExplicit(this.g, e.getKey(), e.getValue()));
        }
        int v = this.checkDirectionalBonds.nextSetBit(0);
        while (v >= 0) {
            Bond bond;
            Edge e;
            int j;
            int nUpV = 0;
            int nDownV = 0;
            int nUpW = 0;
            int nDownW = 0;
            int w = -1;
            int d = this.g.degree(v);
            for (j = 0; j < d; ++j) {
                e = this.g.edgeAt(v, j);
                bond = e.bond(v);
                if (bond == Bond.UP) {
                    ++nUpV;
                    continue;
                }
                if (bond == Bond.DOWN) {
                    ++nDownV;
                    continue;
                }
                if (bond != Bond.DOUBLE) continue;
                w = e.other(v);
            }
            if (w >= 0) {
                this.checkDirectionalBonds.clear(w);
                d = this.g.degree(w);
                for (j = 0; j < d; ++j) {
                    e = this.g.edgeAt(w, j);
                    bond = e.bond(w);
                    if (bond == Bond.UP) {
                        ++nUpW;
                        continue;
                    }
                    if (bond != Bond.DOWN) continue;
                    ++nDownW;
                }
                if (nUpV + nDownV != 0 && nUpW + nDownW != 0) {
                    if (nUpV > 1) {
                        throw new InvalidSmilesException("Multiple directional bonds on atom " + v, buffer);
                    }
                    if (nDownV > 1) {
                        throw new InvalidSmilesException("Multiple directional bonds on atom " + v, buffer);
                    }
                    if (nUpW > 1) {
                        throw new InvalidSmilesException("Multiple directional bonds on atom " + w, buffer);
                    }
                    if (nDownW > 1) {
                        throw new InvalidSmilesException("Multiple directional bonds on atom " + w, buffer);
                    }
                }
            }
            v = this.checkDirectionalBonds.nextSetBit(v + 1);
        }
    }

    private void addTopology(int u, Configuration c) throws InvalidSmilesException {
        if (this.arrangement.containsKey(u)) {
            int[] vs = this.arrangement.get(u).toArray();
            ArrayList<Edge> es = new ArrayList<Edge>(vs.length);
            for (int v : vs) {
                es.add(this.g.edge(u, v));
            }
            if (c.type() == Configuration.Type.Tetrahedral) {
                vs = this.insertThImplicitRef(u, vs);
            } else if (c.type() == Configuration.Type.DoubleBond) {
                vs = this.insertDbImplicitRef(u, vs);
            }
            this.g.addTopology(Topology.create(u, vs, es, c));
        } else {
            int[] us = new int[this.g.degree(u)];
            List<Edge> es = this.g.edges(u);
            for (int i = 0; i < us.length; ++i) {
                us[i] = es.get(i).other(u);
            }
            if (c.type() == Configuration.Type.Tetrahedral) {
                us = this.insertThImplicitRef(u, us);
            } else if (c.type() == Configuration.Type.DoubleBond) {
                us = this.insertDbImplicitRef(u, us);
            } else if (c.type() == Configuration.Type.ExtendedTetrahedral) {
                int x;
                this.g.addFlags(4);
                int v = es.get(0).other(u);
                int w = es.get(1).other(u);
                List<Edge> vs = this.g.edges(v);
                List<Edge> ws = this.g.edges(w);
                us = new int[]{-1, v, -1, w};
                int i = 0;
                for (Edge e : vs) {
                    x = e.other(v);
                    if (e.bond().order() != 1) continue;
                    us[i++] = x;
                }
                i = 2;
                for (Edge e : ws) {
                    x = e.other(w);
                    if (e.bond().order() != 1) continue;
                    us[i++] = x;
                }
                if (us[0] < 0 || us[2] < 0) {
                    return;
                }
                Arrays.sort(us);
            }
            this.g.addTopology(Topology.create(u, us, es, c));
        }
    }

    private int[] insertThImplicitRef(int u, int[] vs) throws InvalidSmilesException {
        if (vs.length == 4) {
            return vs;
        }
        if (vs.length != 3) {
            throw new InvalidSmilesException("Invaid number of verticies for TH1/TH2 stereo chemistry");
        }
        if (this.start.contains(u)) {
            return new int[]{u, vs[0], vs[1], vs[2]};
        }
        return new int[]{vs[0], u, vs[1], vs[2]};
    }

    private int[] insertDbImplicitRef(int u, int[] vs) throws InvalidSmilesException {
        if (vs.length == 3) {
            return vs;
        }
        if (vs.length != 2) {
            throw new InvalidSmilesException("Invaid number of verticies for DB1/DB2 stereo chemistry");
        }
        if (this.start.contains(u)) {
            return new int[]{u, vs[0], vs[1]};
        }
        return new int[]{vs[0], u, vs[1]};
    }

    private void addAtom(Atom a, CharBuffer buffer) throws InvalidSmilesException {
        int v = this.g.addAtom(a);
        if (!this.stack.empty()) {
            int u = this.stack.pop();
            if (this.bond != Bond.DOT) {
                if (this.bond.directional()) {
                    this.checkDirectionalBonds.set(u);
                    this.checkDirectionalBonds.set(v);
                }
                this.g.addEdge(new Edge(u, v, this.bond));
            } else {
                this.start.add(v);
            }
            if (this.arrangement.containsKey(u)) {
                this.arrangement.get(u).add(v);
            }
        }
        this.stack.push(v);
        this.bond = Bond.IMPLICIT;
        if (this.configuration != Configuration.UNKNOWN) {
            this.g.addFlags(2);
            this.configurations.put(v, this.configuration);
            this.configuration = Configuration.UNKNOWN;
        }
    }

    private void readSmiles(CharBuffer buffer) throws InvalidSmilesException {
        block35: while (buffer.hasRemaining()) {
            char c = buffer.get();
            switch (c) {
                case '*': {
                    this.addAtom(AtomImpl.AliphaticSubset.Unknown, buffer);
                    continue block35;
                }
                case 'B': {
                    if (buffer.getIf('r')) {
                        this.addAtom(AtomImpl.AliphaticSubset.Bromine, buffer);
                        continue block35;
                    }
                    this.addAtom(AtomImpl.AliphaticSubset.Boron, buffer);
                    continue block35;
                }
                case 'C': {
                    if (buffer.getIf('l')) {
                        this.addAtom(AtomImpl.AliphaticSubset.Chlorine, buffer);
                        continue block35;
                    }
                    this.addAtom(AtomImpl.AliphaticSubset.Carbon, buffer);
                    continue block35;
                }
                case 'N': {
                    this.addAtom(AtomImpl.AliphaticSubset.Nitrogen, buffer);
                    continue block35;
                }
                case 'O': {
                    this.addAtom(AtomImpl.AliphaticSubset.Oxygen, buffer);
                    continue block35;
                }
                case 'P': {
                    this.addAtom(AtomImpl.AliphaticSubset.Phosphorus, buffer);
                    continue block35;
                }
                case 'S': {
                    this.addAtom(AtomImpl.AliphaticSubset.Sulfur, buffer);
                    continue block35;
                }
                case 'F': {
                    this.addAtom(AtomImpl.AliphaticSubset.Fluorine, buffer);
                    continue block35;
                }
                case 'I': {
                    this.addAtom(AtomImpl.AliphaticSubset.Iodine, buffer);
                    continue block35;
                }
                case 'b': {
                    this.addAtom(AtomImpl.AromaticSubset.Boron, buffer);
                    this.g.addFlags(1);
                    continue block35;
                }
                case 'c': {
                    this.addAtom(AtomImpl.AromaticSubset.Carbon, buffer);
                    this.g.addFlags(1);
                    continue block35;
                }
                case 'n': {
                    this.addAtom(AtomImpl.AromaticSubset.Nitrogen, buffer);
                    this.g.addFlags(1);
                    continue block35;
                }
                case 'o': {
                    this.addAtom(AtomImpl.AromaticSubset.Oxygen, buffer);
                    this.g.addFlags(1);
                    continue block35;
                }
                case 'p': {
                    this.addAtom(AtomImpl.AromaticSubset.Phosphorus, buffer);
                    this.g.addFlags(1);
                    continue block35;
                }
                case 's': {
                    this.addAtom(AtomImpl.AromaticSubset.Sulfur, buffer);
                    this.g.addFlags(1);
                    continue block35;
                }
                case 'H': {
                    if (this.strict) {
                        throw new InvalidSmilesException("hydrogens should be specified in square brackets - '[H]'", buffer);
                    }
                    this.addAtom(AtomImpl.EXPLICIT_HYDROGEN, buffer);
                    continue block35;
                }
                case 'D': {
                    if (this.strict) {
                        throw new InvalidSmilesException("deuterium should be specified as a hydrogen isotope - '[2H]'", buffer);
                    }
                    this.addAtom(AtomImpl.DEUTERIUM, buffer);
                    continue block35;
                }
                case 'T': {
                    if (this.strict) {
                        throw new InvalidSmilesException("tritium should be specified as a hydrogen isotope - '[3H]'", buffer);
                    }
                    this.addAtom(AtomImpl.TRITIUM, buffer);
                    continue block35;
                }
                case '[': {
                    this.addAtom(this.readBracketAtom(buffer), buffer);
                    continue block35;
                }
                case '0': 
                case '1': 
                case '2': 
                case '3': 
                case '4': 
                case '5': 
                case '6': 
                case '7': 
                case '8': 
                case '9': {
                    this.ring(c - 48, buffer);
                    continue block35;
                }
                case '%': {
                    int num = buffer.getNumber(2);
                    if (num < 0) {
                        throw new InvalidSmilesException("a number (<digit>+) must follow '%':", buffer);
                    }
                    this.ring(num, buffer);
                    continue block35;
                }
                case '-': {
                    if (this.bond != Bond.IMPLICIT) {
                        throw new InvalidSmilesException("Multiple bonds specified:", buffer);
                    }
                    this.bond = Bond.SINGLE;
                    continue block35;
                }
                case '=': {
                    if (this.bond != Bond.IMPLICIT) {
                        throw new InvalidSmilesException("Multiple bonds specified:", buffer);
                    }
                    this.bond = Bond.DOUBLE;
                    continue block35;
                }
                case '#': {
                    if (this.bond != Bond.IMPLICIT) {
                        throw new InvalidSmilesException("Multiple bonds specified:", buffer);
                    }
                    this.bond = Bond.TRIPLE;
                    continue block35;
                }
                case '$': {
                    if (this.bond != Bond.IMPLICIT) {
                        throw new InvalidSmilesException("Multiple bonds specified:", buffer);
                    }
                    this.bond = Bond.QUADRUPLE;
                    continue block35;
                }
                case ':': {
                    if (this.bond != Bond.IMPLICIT) {
                        throw new InvalidSmilesException("Multiple bonds specified:", buffer);
                    }
                    this.g.addFlags(1);
                    this.bond = Bond.AROMATIC;
                    continue block35;
                }
                case '/': {
                    if (this.bond != Bond.IMPLICIT) {
                        throw new InvalidSmilesException("Multiple bonds specified:", buffer);
                    }
                    this.bond = Bond.UP;
                    this.g.addFlags(8);
                    continue block35;
                }
                case '\\': {
                    if (this.bond != Bond.IMPLICIT && this.bond != Bond.DOWN) {
                        throw new InvalidSmilesException("Multiple bonds specified:", buffer);
                    }
                    this.bond = Bond.DOWN;
                    this.g.addFlags(8);
                    continue block35;
                }
                case '.': {
                    if (this.bond != Bond.IMPLICIT) {
                        throw new InvalidSmilesException("Bond specified before disconnection:", buffer);
                    }
                    this.bond = Bond.DOT;
                    continue block35;
                }
                case '(': {
                    if (this.stack.empty()) {
                        throw new InvalidSmilesException("cannot open branch - there were no previous atoms:", buffer);
                    }
                    this.stack.push(this.stack.peek());
                    continue block35;
                }
                case ')': {
                    if (this.stack.size() < 2) {
                        throw new InvalidSmilesException("closing of an unopened branch:", buffer);
                    }
                    this.stack.pop();
                    continue block35;
                }
                case '\t': 
                case ' ': {
                    StringBuilder sb = new StringBuilder();
                    while (buffer.hasRemaining() && (c = buffer.get()) != '\n' && c != '\r') {
                        sb.append(c);
                    }
                    this.g.setTitle(sb.toString());
                }
                case '\n': 
                case '\r': {
                    return;
                }
            }
            throw new InvalidSmilesException("unexpected character:", buffer);
        }
    }

    Atom readBracketAtom(CharBuffer buffer) throws InvalidSmilesException {
        int start = buffer.position;
        boolean arbitraryLabel = false;
        if (!buffer.hasRemaining()) {
            throw new InvalidSmilesException("Unclosed bracket atom", buffer);
        }
        int isotope = buffer.getNumber();
        boolean aromatic = buffer.next() >= 'a' && buffer.next() <= 'z';
        Element element = Element.read(buffer);
        if (this.strict && element == null) {
            throw new InvalidSmilesException("unrecognised element symbol: ", buffer);
        }
        if (element != null && aromatic) {
            this.g.addFlags(1);
        }
        if (this.strict && !element.aromatic(Element.AromaticSpecification.OpenSmiles)) {
            throw new InvalidSmilesException("abnormal aromatic element", buffer);
        }
        if (element == null) {
            arbitraryLabel = true;
        }
        this.configuration = Configuration.read(buffer);
        int hCount = Parser.readHydrogens(buffer);
        int charge = Parser.readCharge(buffer);
        int atomClass = Parser.readClass(buffer);
        if (!arbitraryLabel && !buffer.getIf(']')) {
            if (this.strict) {
                throw InvalidSmilesException.invalidBracketAtom(buffer);
            }
            arbitraryLabel = true;
        }
        if (arbitraryLabel) {
            int end = buffer.position;
            int depth = 1;
            while (buffer.hasRemaining()) {
                char c = buffer.get();
                if (c == '[') {
                    ++depth;
                } else if (c == ']' && --depth == 0) break;
                ++end;
            }
            if (depth != 0) {
                throw new InvalidSmilesException("unparsable label in bracket atom", buffer, buffer.position - 1);
            }
            String label = buffer.substr(start, end);
            return new AtomImpl.BracketAtom(label);
        }
        return new AtomImpl.BracketAtom(isotope, element, hCount, charge, atomClass, aromatic);
    }

    static int readHydrogens(CharBuffer buffer) {
        if (buffer.getIf('H')) {
            int count = buffer.getNumber();
            return count < 0 ? 1 : count;
        }
        return 0;
    }

    static int readCharge(CharBuffer buffer) {
        return Parser.readCharge(0, buffer);
    }

    private static int readCharge(int acc, CharBuffer buffer) {
        if (buffer.getIf('+')) {
            return buffer.nextIsDigit() ? acc + buffer.getNumber() : Parser.readCharge(acc + 1, buffer);
        }
        if (buffer.getIf('-')) {
            return buffer.nextIsDigit() ? acc - buffer.getNumber() : Parser.readCharge(acc - 1, buffer);
        }
        return acc;
    }

    static int readClass(CharBuffer buffer) throws InvalidSmilesException {
        if (buffer.getIf(':')) {
            if (buffer.nextIsDigit()) {
                return buffer.getNumber();
            }
            throw new InvalidSmilesException("invalid atom class, <digit>+ must follow ':'", buffer);
        }
        return 0;
    }

    private void ring(int rnum, CharBuffer buffer) throws InvalidSmilesException {
        if (this.bond == Bond.DOT) {
            throw new InvalidSmilesException("a ring bond can not be a 'dot':", buffer, -1);
        }
        if (this.rings.length <= rnum || this.rings[rnum] == null) {
            this.openRing(rnum);
        } else {
            this.closeRing(rnum, buffer);
        }
    }

    private void openRing(int rnum) {
        if (rnum >= this.rings.length) {
            this.rings = Arrays.copyOf(this.rings, Math.min(100, rnum * 2));
        }
        int u = this.stack.peek();
        this.rings[rnum] = new RingBond(u, this.bond);
        this.createArrangement(u).add(-rnum);
        ++this.openRings;
        this.bond = Bond.IMPLICIT;
    }

    private LocalArrangement createArrangement(int u) {
        LocalArrangement la = this.arrangement.get(u);
        if (la == null) {
            la = new LocalArrangement();
            int d = this.g.degree(u);
            for (int j = 0; j < d; ++j) {
                Edge e = this.g.edgeAt(u, j);
                la.add(e.other(u));
            }
            this.arrangement.put(u, la);
        }
        return la;
    }

    private void closeRing(int rnum, CharBuffer buffer) throws InvalidSmilesException {
        RingBond rbond = this.rings[rnum];
        this.rings[rnum] = null;
        int u = rbond.u;
        int v = this.stack.peek();
        if (u == v) {
            throw new InvalidSmilesException("Endpoints of ringbond are the same - loops are not allowed", buffer);
        }
        if (this.g.adjacent(u, v)) {
            throw new InvalidSmilesException("Endpoints of ringbond are already connected - multi-edges are not allowed", buffer);
        }
        this.bond = Parser.decideBond(rbond.bond, this.bond.inverse(), buffer);
        if (this.bond.directional()) {
            this.checkDirectionalBonds.set(u);
            this.checkDirectionalBonds.set(v);
        }
        this.g.addEdge(new Edge(u, v, this.bond));
        this.bond = Bond.IMPLICIT;
        this.arrangement.get(rbond.u).replace(-rnum, this.stack.peek());
        if (this.arrangement.containsKey(v)) {
            this.arrangement.get(v).add(rbond.u);
        }
        --this.openRings;
    }

    static Bond decideBond(Bond a, Bond b, CharBuffer buffer) throws InvalidSmilesException {
        if (a == b) {
            return a;
        }
        if (a == Bond.IMPLICIT) {
            return b;
        }
        if (b == Bond.IMPLICIT) {
            return a;
        }
        throw new InvalidSmilesException("Ring closure bonds did not match. Ring was opened with '" + (Object)((Object)a) + "' and closed with '" + (Object)((Object)b) + "'." + " Note - directional bonds ('/','\\') are relative.", buffer, -1);
    }

    static Graph parse(String str) throws InvalidSmilesException {
        return new Parser(str).molecule();
    }

    private static final class LocalArrangement {
        int[] vs = new int[4];
        int n;

        private LocalArrangement() {
        }

        void add(int v) {
            if (this.n == this.vs.length) {
                this.vs = Arrays.copyOf(this.vs, this.n * 2);
            }
            this.vs[this.n++] = v;
        }

        void replace(int u, int v) {
            for (int i = 0; i < this.n; ++i) {
                if (this.vs[i] != u) continue;
                this.vs[i] = v;
                return;
            }
        }

        int[] toArray() {
            return Arrays.copyOf(this.vs, this.n);
        }
    }

    private static final class RingBond {
        int u;
        Bond bond;

        private RingBond(int u, Bond bond) {
            this.u = u;
            this.bond = bond;
        }
    }
}

