% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pretty_coefficients.R
\name{pretty_coefficients}
\alias{pretty_coefficients}
\title{pretty_coefficients}
\usage{
pretty_coefficients(
  model_object,
  relativity_transform = NULL,
  relativity_label = "relativity",
  type_iii = NULL,
  conf.int = FALSE,
  vimethod = "model",
  spline_seperator = NULL,
  significance_level = 0.05,
  return_data = FALSE,
  ...
)
}
\arguments{
\item{model_object}{Model object to create coefficient table for. Must be of type: \code{\link[stats]{glm}}, \code{\link[stats]{lm}}.}

\item{relativity_transform}{String of the function to be applied to the model estimate to calculate the relativity, for example: 'exp(estimate)-1'. Default is for relativity to be excluded from output.}

\item{relativity_label}{String of label to give to relativity column if you want to change the title to your use case.}

\item{type_iii}{Type III statistical test to perform. Default is none. Options are 'Wald' or 'LR'. Warning 'LR' can be computationally expensive. Test performed via \code{\link[car]{Anova}}}

\item{conf.int}{Set to TRUE to include confidence intervals in summary table. Warning, can be computationally expensive.}

\item{vimethod}{Variable importance method to pass to method of \code{\link[vip]{vi}}. Defaults to "model". Currently supports "permute" and "firm", pass any additional arguments to \code{\link[vip]{vi}} in ...}

\item{spline_seperator}{Separator to look for to identity a spline. If this input is not null, it is assumed any features with this separator are spline columns. For example an age spline from 0 to 25 you could use: AGE_0_25 and "_".}

\item{significance_level}{Significance level to P-values by in kable. Defaults to 0.05.}

\item{return_data}{Set to TRUE to return \code{\link[base]{data.frame}} instead of creating \code{\link[knitr]{kable}}.}

\item{...}{Any additional parameters to be past to  \code{\link[vip]{vi}}}
}
\value{
\code{\link[knitr]{kable}} if return_data = FALSE. \code{\link[base]{data.frame}} if return_data = TRUE.
}
\description{
Creates a pretty kable of model coefficients including coefficient base levels, type III P.values, and variable importance.
}
\examples{

library(dplyr)
library(prettyglm)
data('titanic')
columns_to_factor <- c('Pclass',
                       'Sex',
                       'Cabin',
                       'Embarked',
                       'Cabintype',
                       'Survived')
meanage <- base::mean(titanic$Age, na.rm=TRUE)

titanic  <- titanic  \%>\%
 dplyr::mutate_at(columns_to_factor, list(~factor(.))) \%>\%
 dplyr::mutate(Age =base::ifelse(is.na(Age)==TRUE,meanage,Age)) \%>\%
 dplyr::mutate(Age_0_25 = prettyglm::splineit(Age,0,25),
               Age_25_50 = prettyglm::splineit(Age,25,50),
               Age_50_120 = prettyglm::splineit(Age,50,120)) \%>\%
 dplyr::mutate(Fare_0_250 = prettyglm::splineit(Fare,0,250),
               Fare_250_600 = prettyglm::splineit(Fare,250,600))

# A simple example
survival_model <- stats::glm(Survived ~
                              Pclass +
                              Sex +
                              Age +
                              Fare +
                              Embarked +
                              SibSp +
                              Parch +
                              Cabintype,
                             data = titanic,
                             family = binomial(link = 'logit'))
pretty_coefficients(survival_model)

# A more complicated example with a spline and different importance method
survival_model3 <- stats::glm(Survived ~
                                        Pclass +
                                        Age_0_25 +
                                        Age_25_50 +
                                        Age_50_120 +
                                        Sex:Fare_0_250 +
                                        Sex:Fare_250_600 +
                                        Embarked +
                                        SibSp +
                                        Parch +
                                        Cabintype,
                              data = titanic,
                              family = binomial(link = 'logit'))
pretty_coefficients(survival_model3,
                    relativity_transform = 'exp(estimate)-1',
                    spline_seperator = '_',
                    vimethod = 'permute',
                    target = 'Survived',
                    metric = "roc_auc",
                    event_level = 'second',
                    pred_wrapper = predict.glm,
                    smaller_is_better = FALSE,
                    train = survival_model3$data, # need to supply training data for vip importance
                    reference_class = 0)


}
