% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/m325traceline.R
\name{m325traceline}
\alias{m325traceline}
\title{Minenergo-325. Trace thermal-hydraulic regime for linear segment}
\usage{
m325traceline(
  temperature = 130,
  pressure = mpa_kgf(6),
  consumption = 250,
  g = 0,
  d = 700,
  len = c(600, 530, 300, 350),
  year = 1986,
  insulation = 0,
  laying = "underground",
  beta = FALSE,
  exp5k = TRUE,
  roughness = 0.006,
  inlet = 0,
  outlet = 0,
  elev_tol = 0.1,
  method = "romeo",
  forward = TRUE,
  absg = TRUE
)
}
\arguments{
\item{temperature}{temperature of heat carrier (water) inside the pipe sensor-measured at the inlet
(forward tracing) or at the outlet (backward tracing) of path, [\emph{°C}].
Type: \code{\link{assert_number}}.}

\item{pressure}{\href{https://en.wikipedia.org/wiki/Pressure_measurement#Absolute}{absolute pressure}
of heat carrier (water) sensor-measured at the inlet
(forward tracing) or at the outlet (backward tracing) of path, [\emph{MPa}].
Type: \code{\link{assert_number}}.}

\item{consumption}{amount of heat carrier (water) sensor-measured at the inlet (forward tracing) or at
the outlet (backward tracing) of path, [\emph{ton/hour}].
Type: \code{\link{assert_number}}.}

\item{g}{amount of heat carrier discharge to network for each pipe segment in the
tracing path enumerated along the direction of flow. If flag \code{absg}
is \code{TRUE} then they treat argument \code{g} as absolute value in
[\emph{ton/hour}], otherwise they do as percentage of consumption in the
pipe segment.
Type: \code{\link{assert_double}}.}

\item{d}{internal diameters of subsequent pipes in tracing path that are enumerated
along the direction of flow, [\emph{mm}].
Type: \code{\link{assert_double}}.}

\item{len}{length of subsequent pipes in tracing path that are enumerated
along the direction of flow, [\emph{m}].
Type: \code{\link{assert_double}}.}

\item{year}{year when pipe is put in operation after laying or total overhaul for
each pipe in tracing path enumerated along the direction of flow.
Type: \code{\link{assert_integerish}}.}

\item{insulation}{insulation that covers the exterior of pipe:
\describe{
  \item{\code{0}}{no insulation}
  \item{\code{1}}{foamed polyurethane or analogue}
  \item{\code{2}}{polymer concrete}
}
for each pipe in tracing path enumerated along the direction of flow.
Type: \code{\link{assert_numeric}} and \code{\link{assert_subset}}.}

\item{laying}{type of pipe laying depicting the position of pipe in space:
\itemize{
  \item \code{air}
  \item \code{channel}
  \item \code{room}
  \item \code{tunnel}
  \item \code{underground}
}
for each pipe in tracing path enumerated along the direction of flow.
Type: \code{\link{assert_character}} and \code{\link{assert_subset}}.}

\item{beta}{should they consider additional heat losses of fittings? Logical value
for each pipe in tracing path enumerated along the direction of flow.
Type: \code{\link{assert_logical}}.}

\item{exp5k}{pipe regime flag: is pipe operated more that \code{5000} hours per year? Logical
value for each pipe in tracing path enumerated along the direction of flow.
Type: \code{\link{assert_logical}}.}

\item{roughness}{roughness of internal wall for each pipe in tracing path enumerated along
the direction of flow, [\emph{m}]. Type: \code{\link{assert_double}}.}

\item{inlet}{elevation of pipe inlet for each pipe in tracing path enumerated along
the direction of flow, [\emph{m}]. Type: \code{\link{assert_double}}.}

\item{outlet}{elevation of pipe outlet for each pipe in tracing path enumerated along
the direction of flow, [\emph{m}]. Type: \code{\link{assert_double}}.}

\item{elev_tol}{maximum allowed discrepancy between adjacent outlet and inlet elevations of
two subsequent pipes in the traced path, [\emph{m}].
Type: \code{\link{assert_number}}.}

\item{method}{method of determining \emph{Darcy friction factor}
\itemize{
  \item \code{romeo}
  \item \code{vatankhan}
  \item \code{buzelli}
}
Type: \code{\link{assert_choice}}. For more details see \code{\link{dropp}}.}

\item{forward}{tracing direction flag: is it a forward direction of tracing?
If \code{FALSE} the backward tracing is performed.
Type: \code{\link{assert_flag}}.}

\item{absg}{Whether argument \code{g} (amount of heat carrier discharge to network) is an
absolute value in [\emph{ton/hour}] (\code{TRUE}) or is it a percentage of
consumption in the pipe segment (\code{FALSE})?
Type: \code{\link{assert_flag}}.}
}
\value{
named list of regime parameters for the traced path with the next elements:
 \describe{
   \item{\code{temperature}}{calculated temperatures of heat carrier for all pipeline segments, [\emph{°C}].
   Type: \code{\link{assert_double}}.}
   \item{\code{pressure}}{calculated pressures of heat carrier for all pipeline segments, [\emph{MPa}].
   Type: \code{\link{assert_double}}.}
   \item{\code{consumption}}{calculated consumption(s) of heat carrier for all pipeline segments, [\emph{ton/hour}].
   Type: \code{\link{assert_double}}.}
 }
}
\description{
Trace values of thermal-hydraulic regime (temperature, pressure,
 consumption) along the adjacent linear segments of pipeline using norms of
 heat flux values prescribed by
 \href{http://docs.cntd.ru/document/902148459}{Minenergo Order 325}.
}
\details{
The calculated (values of) regime may be considered as representation of
 district heating process in conditions of hypothetically perfect
 technical state of pipe walls and insulation.

 They consider only simple tracing paths which do not contain rings and any
 kind of parallelization. At the same time bidirectional (forward and
 backward) tracing is possible in accordance with sensor position. They also
 may consider discharges to network at the inlet of each pipeline segment
 as an approximation of actual forks of flows. Relevant illustration of
 adopted assumptions for 4-segment tracing path is depicted on the next
 figure.

 \figure{m325regtrace.png}

 They make additional check for consistency of \code{inlet} and \code{outlet}
 values for subsequent pipe segments. Discrepancy of appropriate elevations
 cannot be more than \code{elev_tol}.
}
\examples{
# Consider 4-segment tracing path depicted in ?m325regtrace help page.
# First, let sensor readings for forward tracing:
t_fw <- 130  # [°C]
p_fw <- .588399*all.equal(.588399, mpa_kgf(6))  # [MPa]
g_fw <- 250  # [ton/hour]

# Let discharges to network for each pipeline segment are somehow determined as
discharges <- seq(0, 30, 10)  # [ton/hour]

\donttest{
# Then the calculated regime (red squares) for forward tracing is
regime_fw <- m325traceline(t_fw, p_fw, g_fw, discharges, forward = TRUE)
print(regime_fw)

# $temperature
# [1] 129.1799 128.4269 127.9628 127.3367
#
# $pressure
# [1] 0.5878607 0.5874226 0.5872143 0.5870330
#
# $consumption
# [1] 250 240 220 190
}
# Next consider values of traced regime as sensor readings for backward tracing:
t_bw <- 127.3367  # [°C]
p_bw <- .5870330  # [MPa]
g_bw <- 190  # [ton/hour]

# Then the calculated regime (red squares) for backward tracing is
\donttest{
regime_bw <- m325traceline(t_bw, p_bw, g_bw, discharges, forward = FALSE)
print(regime_bw)

# $temperature
# [1] 129.9953 129.1769 128.4254 127.9619
#
# $pressure
# [1] 0.5883998 0.5878611 0.5874228 0.5872144
#
# $consumption
# [1] 250 250 240 220

# Let compare sensor readings with backward tracing results:
tracing <- with(regime_bw, {
  lambda <- function(val, constraint)
    c(val, constraint, constraint - val,
      abs(constraint - val)*100/constraint)
  first <- 1
  structure(
    rbind(
      lambda(temperature[first], t_fw),
      lambda(pressure[first],    p_fw),
      lambda(consumption[first], g_fw)
    ),
    dimnames = list(
      c("temperature", "pressure", "consumption"),
      c("sensor.value", "traced.value", "abs.discr", "rel.discr")
    )
  )
})
print(tracing)

# sensor.value traced.value     abs.discr    rel.discr
# temperature   130.000000  129.9952943  4.705723e-03 0.0036197868
# pressure        0.588399    0.5883998 -8.290938e-07 0.0001409067
# consumption   250.000000  250.0000000  0.000000e+00 0.0000000000
}
}
\seealso{
\code{\link{m325dropt}} for calculating normative temperature drop in
 single pipeline segment

Other Regime tracing: 
\code{\link{m325tracebw}()},
\code{\link{m325tracefw}()}
}
\concept{Regime tracing}
