% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/board_azure.R
\name{board_azure}
\alias{board_azure}
\title{Use an Azure storage container as a board}
\usage{
board_azure(
  container,
  path = "",
  n_processes = 10,
  versioned = TRUE,
  cache = NULL
)
}
\arguments{
\item{container}{An azure storage container created by
\code{\link[AzureStor:blob_container]{AzureStor::blob_container()}} or similar.}

\item{path}{Path to the directory in the container to store pins. Will be
created if it doesn't already exist. The equivalent of a \code{prefix} for AWS
S3 storage.}

\item{n_processes}{Maximum number of processes used for parallel
uploads/downloads.}

\item{versioned}{Should this board be registered with support for versions?}

\item{cache}{Cache path. Every board requires a local cache to avoid
downloading files multiple times. The default stores in a standard
cache location for your operating system, but you can override if needed.}
}
\description{
Pin data to a container in Azure storage using the AzureStor package.
}
\details{
You can create a board in any of the services that AzureStor supports: blob
storage, file storage and Azure Data Lake Storage Gen2 (ADLSgen2).

Blob storage is the classic storage service that is most familiar to people,
but is relatively old and inefficient. ADLSgen2 is a modern replacement API
for working with blobs that is much faster when working with directories.
You should consider using this rather than the classic blob API where
possible; see the examples below.

\code{board_azure()} is powered by the AzureStor package, which is a suggested
dependency of pins (not required for pins in general). If you run into errors
when deploying content to a server like \url{https://www.shinyapps.io} or
\href{https://posit.co/products/enterprise/connect/}{Connect}, add
\code{requireNamespace("AzureStor")} to your app or document for \href{https://docs.posit.co/connect/user/troubleshooting/#render-missing-r-package}{automatic dependency discovery}.
}
\examples{
if (requireNamespace("AzureStor")) {
  # Public access board
  url <- "https://pins.blob.core.windows.net/public-data"
  container <- AzureStor::blob_container(url)
  board <- board_azure(container)
  board \%>\% pin_read("mtcars")
}

\dontrun{
# To create a board that you can write to, you'll need to supply one
# of `key`, `token`, or `sas` to AzureStor::blob_container()
# First, we create a board using the classic Azure blob API
url <- "https://myaccount.blob.core.windows.net/mycontainer"
container <- AzureStor::blob_container(url, sas = "my-sas")
board <- board_azure(container, "path/to/board")
board \%>\% pin_write(iris)

# ADLSgen2 is a modern, efficient way to access blobs
# - Use 'dfs' instead of 'blob' in the account URL to use the ADLSgen2 API
# - Use the 'storage_container' generic instead of the service-specific
#   'blob_container'
# - We reuse the board created via the blob API above
adls_url <- "https://myaccount.dfs.core.windows.net/mycontainer"
container <- AzureStor::storage_container(adls_url, sas = "my-sas")
board <- board_azure(container, "path/to/board")
board \%>\% pin_list()
board \%>\% pin_read("iris")
}
}
