#' Calculate weighted centroids
#'
#' Calculate the weighted centroids of clouds of multivariate points.
#'
#' @param ordination.results Data frame of ordination results, e.g. the $x element from a
#' prcomp object, or the $points element from a metaMDS object in vegan. 
#' @param road.map Identical to the input for the 'a' argument in the dbFD function of
#' the FD package, and to the picante.cdm argument used elsewhere in this package.
#' Thus, this is a matrix containing the abundance of each 'species' in
#' the ordination results. Rows are "sites" and columns are "species". Rather than
#' abundances, the values can simply be presence/absences. Moreover, sites could be
#' species and species could be individuals. See details.
#' 
#' @details The definition of FDis provided by Laliberte and Legendre (2010) and
#' implemented in the dbFD function of the FD package is geared towards calculating the
#' functional diversity of a community, given a set of species with an array of traits.
#' Another useful way in which FDis might be implemented is as a measure of a species'
#' niche breadth. In this case, ordination.results would be measures (e.g.
#' foraging observations) of multiple individuals of multiple species', and road.map would
#' be a matrix describing which observations belong to which species. The abundances in
#' the matrix in this case would describe how much weight to assign to each individual
#' observation.
#'
#' @return Matrix specifying the position of the centroids along each axis.
#'
#' @export
#'
#' @references Miller, E. T. 2016. Random thoughts.
#'
#' Laliberte, E. & P. Legendre. 2010. A distance-based framework for measuring functional
#' diversity from multiple traits. Ecology 91:299-305.
#'
#' @examples
#' #example of how to calculate the weighted centroids of a series of plots based on the
#' #traits of a set of species. begin by simulating a phylogeny with a birth-death process
#' tree <- geiger::sim.bdtree(b=0.1, d=0, stop="taxa", n=50)
#'
#' #create a log-normal abundance distribution
#' sim.abundances <- round(rlnorm(5000, meanlog=2, sdlog=1)) + 1
#'
#' #simulate a community data matrix, with species as columns and sites as rows
#' cdm <- simulateComm(tree, richness.vector=10:25, abundances=sim.abundances)
#'
#' #simulate two traits, combine into a matrix, then ordinate with a PCA
#' traits <- evolveTraits(tree)[[2]]
#' ord <- prcomp(traits)
#' 
#' #the weighted centroids ("average traits") of the species in each plot
#' exCenter <- centers(ordination.results=ord$x, road.map=cdm)
#'
#' #visual demonstration of how the function works. plot all of the species in the first
#' #three sites in the CDM according to their ordinated trait values, color the points
#' #according to the site (red=site1, purple=site2, blue=site3), and scale the size of the
#' #point to the abundance of that species in that site. centroids are plotted in the
#' #same color but given a thick black border.
#' plot(ord$x, type="n")
#' for(i in 1:3)
#' {
#' 	#find the species and their abundances that occur in the relevant plot
#' 	focal <- as.matrix(cdm)[i,][as.matrix(cdm)[i,]!=0]
#' 	pointColors <- c(rgb(1, 0, 0, 0.6), rgb(0.7, 0, 1, 0.6), rgb(0, 0, 1, 0.6))
#' 	for(j in 1:length(focal))
#' 	{
#' 		location <- ord$x[names(focal)[j],]
#' 		points(x=location[1], y=location[2], col=pointColors[i], pch=20, cex=focal[j]/5)
#' 	}
#' 	points(x=exCenter[i,1], y=exCenter[i,2], pch=21, col="black", lwd=2, cex=2,
#'		bg=pointColors[i])
#' }

centers <- function(ordination.results, road.map)
{
	results <- matrix(ncol=dim(ordination.results)[2], nrow=dim(road.map)[1])
	for(i in 1:dim(road.map)[1])
	{
		results[i,] <- apply(ordination.results, 2, weighted.mean, w=road.map[i,])
	}
	
	row.names(results) <- row.names(road.map)
	
	return(results)
}
