% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/viz_forest.R
\name{viz_forest}
\alias{viz_forest}
\title{Forest plot variants for meta-analyses}
\usage{
viz_forest(x, group = NULL, type = "standard", variant = "classic",
  method = "FE", study_labels = NULL, summary_label = NULL,
  confidence_level = 0.95, col = "Blues", text_size = 3,
  xlab = "Effect", x_limit = NULL, x_trans_function = NULL,
  x_breaks = NULL, annotate_CI = FALSE, study_table = NULL,
  summary_table = NULL, table_headers = NULL, table_layout = NULL, ...)
}
\arguments{
\item{x}{data.frame or matrix with the effect sizes of all studies (e.g.,
correlations, log odds ratios, or Cohen \emph{d}) in the first column and their
respective standard errors in the second column. Alternatively, x can be the
output object of function \code{\link[metafor]{rma.uni}} from package
\pkg{metafor}; then effect sizes and standard errors are extracted from \code{x}.}

\item{group}{factor indicating the subgroup of each study to plot a subgroup forest plot. Has to be in the same order than \code{x}.}

\item{type}{character string indicating the type of forest plot to be plotted. Can be "standard" (default), "study_only",
"summary_only", "cumulative", or "sensitivity". See 'Details'.}

\item{variant}{character string indicating the forest plot variant that should be plotted. Can be "rain" for rainforest plot,
"thick" for a thick forest plot, or "classic" (default) for a traditional forest plot.}

\item{method}{character string indicating which method should be used to compute the study weights and summary effect(s).
Can be any method argument from \code{\link[metafor]{rma.uni}}
(e.g., "FE" for the fixed effect model, or "DL" for the random effects model using the
DerSimonian-Laird method to estimate \eqn{\tau^2}{tau squared}).}

\item{study_labels}{a character vector with names/identifiers to annotate each study in the forest plot.
Has to be in the same order than \code{x}. Ignored if \code{study_table} and/or \code{summary_table} is supplied.}

\item{summary_label}{a character string specifying the name to annotate the summary effect. If a subgroup
analysis is plotted, \code{summary_label} should be a character vector with a name for each
subgroup summary effect, arranged in the order of the levels of \code{group}. Ignored if \code{study_table} and/or
\code{summary_table} is supplied.}

\item{confidence_level}{numeric value. The confidence level for the plotted confidence intervals.}

\item{col}{character string specifying the main color for plotting. For \code{variant = "rain"} must be one of the following palettes from package
\pkg{RColorBrewer}: "Blues", "Greys", "Oranges", "Greens", "Reds", or "Purples".}

\item{text_size}{numeric value. Size of text in the forest plot. Default is 3.}

\item{xlab}{character string specifying the label of the x axis. By default also used for the header of the aligned table if \code{annotate_CI} is \code{TRUE}.}

\item{x_limit}{numeric vector of length 2 with the limits (minimum, maximum) of the x axis.}

\item{x_trans_function}{function to transform the labels of the x axis. Common uses are to transform
log-odds-ratios or log-risk-ratios with \code{exp} to their original scale (odds ratios and risk ratios), or Fisher's z values
back to correlation coefficients using \code{tanh}. See vignette('metaviz').}

\item{x_breaks}{numeric vector of values for the breaks on the x-axis. When used in tandem with \code{x_trans_function}
the supplied values should be not yet transformed.}

\item{annotate_CI}{logical scalar. Should the effect size and confidence interval values be shown as text in an aligned table on the right-hand side of the forest plot?}

\item{study_table}{a data.frame with additional study-level variables which should be shown in an aligned table.
Has to be in the same order than \code{x}. See vignette('metaviz').}

\item{summary_table}{a data.frame with additional summary-level information shown in an aligned table.
If \code{group} is supplied, \code{summary_table} must have a row for each subgroup
summary effect, arranged in the order of the levels of \code{group}. See vignette('metaviz').}

\item{table_headers}{character vector. Headers for each column of aligned tables via \code{study_table}, \code{summary_table}, or \code{annotate_CI}.}

\item{table_layout}{numeric layout matrix passed to \code{layout_matrx} of \code{\link[gridExtra]{arrangeGrob}}. Can be used to overwrite the default spacing
of the forest plot and aligned tables via \code{study_table}, \code{summary_table}, and \code{annotate_CI}.}

\item{...}{further arguments passed to \code{\link[metaviz]{viz_rainforest}} for \code{variant = "rain"}, or
\code{\link[metaviz]{viz_thickforest}} for \code{variant = "thick"}.}
}
\value{
A forest plot is created using ggplot2.
}
\description{
Creates a rainforest, thick forest, or classic forest plot.
}
\details{
The forest plot is the most widely used display to visualize meta-analytic results.
The function \code{viz_forest} creates visually appealing and informative-rich forest plots using ggplot2. Many options
to flexibly customize the visual appearance and statistical information displayed are provided. In addition,
rainforest plots as well as the thick forest plots can be created, two variants and enhancements of the
classical forest plot recently proposed by Schild and Voracek (2015). For further details see the documentation of
\code{\link[metaviz]{viz_rainforest}}, and \code{\link[metaviz]{viz_thickforest}}.

\bold{Available forest plot types}

Different aspects of meta-analytic data can be shown in forest plots. Five different types are available in \code{viz_forest} via the \code{type} parameter.
Argument \code{"standard"} (default) shows study results as well as summary results in the forest plot. \code{"study_only"} allows to only show study results without the meta-analytic summary estimate.
\code{"summary_only"} can be used to only show meta-analytic summary estimate(s), which is primarily useful to visualize several subgroup results (using \code{group}).
\code{"cumulative"} shows a cumulative meta-analysis, that is, meta-analytic summary effects are computed sequentially by adding each study one-by-one.
Studies are added in the same order than they were supplied in \code{x}. Finally, \code{"sensitivity"} shows for each study the meta-analytic summary
effect if that particular study is not considered in the computation of the summary effect (leave-one-out analysis).
}
\examples{
library(metaviz)
# Plotting the mozart data using a classic forest plot
viz_forest(x = mozart[, c("d", "se")],
study_labels = mozart[, "study_name"], xlab = "Cohen d")

# Subgroup analysis of published and unpublished studies shown in a rainforest plot
viz_forest(x = mozart[, c("d", "se")], study_labels = mozart[, "study_name"], method = "REML",
variant = "rain", summary_label = c("Summary (rr_lab = no)", "Summary (rr_lab = yes)"),
group = mozart[, "rr_lab"], xlab = "Cohen d")

# Thick forest plot with additional information in aligned tables. Log risk
# ratios are labeled in their original metric (risk ratios) on the x axis.
viz_forest(x = exrehab[, c("logrr", "logrr_se")], variant = "thick",
xlab = "RR", x_trans_function = exp, annotate_CI = TRUE,
study_table = data.frame(
Name = exrehab[, "study_name"],
eventsT = paste(exrehab$ai, "/", exrehab$ai + exrehab$bi, sep = ""),
eventsC = paste(exrehab$ci, "/", exrehab$ci + exrehab$di, sep = "")),
summary_table = data.frame(
Name = "Summary",
eventsT = paste(sum(exrehab$ai), "/", sum(exrehab$ai + exrehab$bi), sep = ""),
eventsC = paste(sum(exrehab$ci), "/", sum(exrehab$ci + exrehab$di), sep = "")),
table_layout = matrix(c(1, 1, 2, 2, 3), nrow = 1))
}
\references{
Schild, A. H., & Voracek, M. (2015). Finding your way out of the
 forest without a trail of bread crumbs: Development and evaluation of two
 novel displays of forest plots. \emph{Research Synthesis Methods}, \emph{6},
 74-86.
}
\author{
Michael Kossmeier* <michael.kossmeier@univie.ac.at>

Ulrich S. Tran* <ulrich.tran@univie.ac.at>

Martin Voracek* <martin.voracek@univie.ac.at>

*Department of Basic Psychological Research and Research Methods, School of Psychology, University of Vienna
}
