% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mdgc.R
\name{mdgc_log_ml}
\alias{mdgc_log_ml}
\title{Evaluate the Log Marginal Likelihood and Its Derivatives}
\usage{
mdgc_log_ml(
  ptr,
  vcov,
  mea,
  rel_eps = 0.01,
  n_threads = 1L,
  comp_derivs = FALSE,
  indices = NULL,
  do_reorder = TRUE,
  maxpts = 100000L,
  abs_eps = -1,
  minvls = 100L,
  use_aprx = FALSE
)
}
\arguments{
\item{ptr}{object returned by \code{\link{get_mdgc_log_ml}}.}

\item{vcov}{covariance matrix.}

\item{mea}{vector with non-zero mean entries.}

\item{rel_eps}{relative error for each marginal likelihood factor.}

\item{n_threads}{number of threads to use.}

\item{comp_derivs}{logical for whether to approximate the gradient.}

\item{indices}{integer vector with which terms (observations) to include.
Must be zero-based. \code{NULL} yields all observations.}

\item{do_reorder}{logical for whether to use a heuristic variable
reordering. \code{TRUE} is likely the best option.}

\item{maxpts}{maximum number of samples to draw for each marginal likelihood
term.}

\item{abs_eps}{absolute convergence threshold for each marginal likelihood
factor.}

\item{minvls}{minimum number of samples.}

\item{use_aprx}{logical for whether to use an approximation of
\code{\link{pnorm}} and \code{\link{qnorm}}. This may yield a
noticeable reduction in the computation time.}
}
\value{
A numeric vector with a single element with the log marginal likelihood
approximation. Two attributes are added if \code{comp_derivs} is
\code{TRUE}: \code{"grad_vcov"} for the derivative approximation with
respect to \code{vcov} and \code{"grad_mea"} for the derivative
approximation with respect to \code{mea}.
}
\description{
Approximates the log marginal likelihood and the derivatives using
randomized quasi-Monte Carlo. The method uses a generalization of the Fortran
code by Genz and Bretz (2002).

Mean terms for observed continuous variables are always assumed to be
zero.

The returned log marginal likelihood is not a proper log marginal likelihood
if the \code{ptr} object is constructed from a mdgc object from
\code{\link{get_mdgc}} as it does not include the log of the determinants of
the Jacobians for the transformation of the continuous variables.
}
\examples{
# there is a bug on CRAN's check on Solaris which I have failed to reproduce.
# See https://github.com/r-hub/solarischeck/issues/8#issuecomment-796735501.
# Thus, this example is not run on Solaris
is_solaris <- tolower(Sys.info()[["sysname"]]) == "sunos"

if(!is_solaris){
  # randomly mask data
  set.seed(11)
  masked_data <- iris
  masked_data[matrix(runif(prod(dim(iris))) < .10, NROW(iris))] <- NA

  # use the functions in the package
  library(mdgc)
  obj <- get_mdgc(masked_data)
  ptr <- get_mdgc_log_ml(obj)
  start_vals <- mdgc_start_value(obj)
  print(mdgc_log_ml(ptr, start_vals, obj$means))
  print(mdgc_log_ml(ptr, start_vals, obj$means, use_aprx = TRUE))
  print(mdgc_log_ml(ptr, start_vals, obj$means, use_aprx = TRUE,
                    comp_derivs = TRUE))
}

}
\references{
Genz, A., & Bretz, F. (2002). \emph{Comparison of Methods for the
Computation of Multivariate t Probabilities}.
Journal of Computational and Graphical Statistics.

Genz, A., & Bretz, F. (2008).
\emph{Computation of Multivariate Normal and t Probabilities}.
Springer-Verlag, Heidelberg.
}
\seealso{
\code{\link{mdgc_fit}}
}
