% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/popup.R
\name{popupTable}
\alias{popupTable}
\alias{popupImage}
\alias{popupGraph}
\title{Create HTML strings for popups}
\usage{
popupTable(x, zcol, row.numbers = TRUE, feature.id = TRUE)

popupImage(img, src = c("local", "remote"), embed = FALSE, ...)

popupGraph(graphs, type = c("png", "svg", "html"), width = 300,
  height = 300, ...)
}
\arguments{
\item{x}{A \code{Spatial*} object.}

\item{zcol}{\code{numeric} or \code{character} vector indicating the columns
included in the output popup table. If missing, all columns are displayed.}

\item{row.numbers}{\code{logical} whether to include row numbers in the popup table.}

\item{feature.id}{\code{logical} whether to add 'Feature ID' entry to popup table.}

\item{img}{A character \code{vector} of file path(s) or
web-URL(s) to any sort of image file(s).}

\item{src}{Whether the source is "local" (i.e. valid file path(s)) or
"remote" (i.e. valid URL(s)).}

\item{embed}{whether to embed the (local) images in the popup html as
base64 ecoded. Set this to TRUE if you want to save and share your map, unless
you want render many images, then set to FALSE and make sure to copy ../graphs
when copying the map to a different location.}

\item{...}{further arguments passed on to underlying methods such as
height and width.}

\item{graphs}{A \code{list} of figures associated with \code{x}.}

\item{type}{Output filetype, one of "png" (default), "svg" or "html".}

\item{width}{popup width in pixels.}

\item{height}{popup height in pixels.}
}
\value{
A \code{list} of HTML strings required to create feature popup tables.

A \code{list} of HTML strings required to create popup graphs.

A \code{list} of HTML strings required to create popup graphs.
}
\description{
Create HTML strings for \code{popup} tables used as input for
\code{\link{mapview}} or \code{\link{leaflet}}.
This optionally allows the user to include only a subset of feature attributes.

Create HTML strings for \code{popup} images used as input for
\code{\link{mapview}} or \code{\link{leaflet}}.

Create HTML strings for \code{popup} graphs used as input for
\code{\link{mapview}} or \code{\link{leaflet}}.
}
\details{
Type \code{svg} uses native \code{svg} encoding via \code{\link{readLines}}.
\code{height} and \code{width} are set via \code{...} and passed on to
\code{\link{svg}} \cr
Type \code{png} embeds via \code{"<img src = ..."}.
\code{height} and \code{width} are set via \code{...} and passed on to
\code{\link{png}} \cr
Type \code{html} embeds via \code{"<iframe src = ..."}.
\code{height} and \code{width} are set directly in pixels. \cr
}
\examples{
library(leaflet)

## include columns 1 and 2 only
mapview(franconia, popup = popupTable(franconia, zcol = 1:2))
mapview(breweries, zcol = "founded", legend = TRUE,
        popup = popupTable(breweries, zcol = c("founded", "village")))
leaflet() \%>\% addCircleMarkers(data = breweries)
leaflet() \%>\% addCircleMarkers(data = breweries,
                               popup = popupTable(breweries))

\dontrun{
## remote images -----
### one image
library(sf)

pnt = st_as_sf(data.frame(x = 174.764474, y = -36.877245),
                coords = c("x", "y"),
                crs = 4326)

img = "http://bit.ly/1TVwRiR"

mapview(pnt, popup = popupImage(img, src = "remote"))

### multiple file (types)
library(sp)
images = c(img,
            "https://upload.wikimedia.org/wikipedia/commons/9/91/Octicons-mark-github.svg",
            "https://www.r-project.org/logo/Rlogo.png",
            "https://upload.wikimedia.org/wikipedia/commons/d/d6/MeanMonthlyP.gif")

pt4 = data.frame(x = jitter(rep(174.764474, 4), factor = 0.01),
                  y = jitter(rep(-36.877245, 4), factor = 0.01))
coordinates(pt4) = ~ x + y
proj4string(pt4) = "+init=epsg:4326"

mapview(pt4, popup = popupImage(images)) # NOTE the gif animation

## local images -----
pnt = st_as_sf(data.frame(x = 174.764474, y = -36.877245),
                coords = c("x", "y"), crs = 4326)
img = system.file("img","Rlogo.png",package="png")
mapview(pnt, popup = popupImage(img))
}

\dontrun{
### example: svg -----

library(sp)

data(meuse)
coordinates(meuse) = ~ x + y
proj4string(meuse) = CRS("+init=epsg:28992")

## create plots with points colored according to feature id
library(lattice)
p = xyplot(copper ~ cadmium, data = meuse@data, col = "grey")
p = mget(rep("p", length(meuse)))

clr = rep("grey", length(meuse))
p = lapply(1:length(p), function(i) {
  clr[i] = "red"
  update(p[[i]], col = clr)
})

mapview(meuse, popup = popupGraph(p, type = "svg"))

### example: png -----
pt = data.frame(x = 174.764474, y = -36.877245)

coordinates(pt) = ~ x + y
proj4string(pt) = "+init=epsg:4326"

p2 = levelplot(t(volcano), col.regions = terrain.colors(100))

mapview(pt, popup = popupGraph(p2, width = 300, height = 400))

### example: html -----
mapview(breweries[1, ], map.types = "Esri.WorldImagery",
        popup = popupGraph(mapview(breweries[1, ])@map,
                           type = "html",
                           width = 500,
                           height = 400))
}

}
