% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitted.lmvar.R
\name{fitted.lmvar}
\alias{fitted.lmvar}
\title{Fitted values for an 'lmvar' object}
\usage{
\method{fitted}{lmvar}(object, mu = TRUE, sigma = TRUE, log = FALSE,
  interval = c("none", "confidence", "prediction"), level = 0.95, ...)
}
\arguments{
\item{object}{An 'lmvar' object}

\item{mu}{Boolean, specifies whether or not to return the expected values}

\item{sigma}{Boolean, specifies whether or not to return the standard deviations}

\item{log}{Boolean, specifies whether expected values, standard deviations (as well as their confidence intervals) and
prediction intervals should be for \eqn{Y} (\code{log = FALSE}) or for \eqn{e^Y} (\code{log = TRUE}).}

\item{interval}{Character string, specifying the type of interval. Possible values are
\itemize{
\item "none" No interval, this is the default
\item "confidence" Confidence intervals for the expected values (if \code{mu = TRUE}) and the standard deviation
(if \code{sigma = TRUE})
\item "prediction" Prediction intervals for the response vector \eqn{Y} (\code{log = FALSE}) or for \eqn{e^Y} (\code{log = TRUE})
}}

\item{level}{Numeric value between 0 and 1, specifying the confidence level}

\item{...}{For compatibility with \code{\link[stats]{fitted}} generic.}
}
\value{
In the case \code{mu = FALSE} and \code{interval = "none"}: a numeric vector containing the estimators for
the standard deviation.

In the case \code{sigma = FALSE} and \code{interval = "none"}: a numeric vector containing the estimators for
the expected values.

In all other cases: a matrix with one column for each requested feature and one row for each observation. The column names are
\itemize{
\item \code{mu} Estimators for the expected value \eqn{\mu}
\item \code{sigma} Estimators for the standard deviation \eqn{\sigma}
\item \code{mu_lwr} Lower bound of the confidence interval for \eqn{\mu}
\item \code{mu_upr} Upper bound of the confidence interval for \eqn{\mu}
\item \code{sigma_lwr} Lower bound of the confidence interval for \eqn{\sigma}
\item \code{sigma_upr} Upper bound of the confidence interval for \eqn{\sigma}
\item \code{lwr} Lower bound of the prediction interval
\item \code{upr} Upper bound of the prediction interval
}
}
\description{
Estimators and confidence intervals for the expected values and standard deviations of the response vector \eqn{Y} of
an 'lmvar' model. Prediction intervals for \eqn{Y} as well. Alternatively, estimators and intervals can be for \eqn{e^Y}.
}
\details{
If \code{log = FALSE} and \eqn{Y} the vector of observations stored in \code{object}, \code{fitted.lmvar} returns
expected values and standard deviations for the observations \eqn{Y}.

If \code{log = TRUE} and \eqn{Y} the vector of observations stored in \code{object}, \code{fitted.lmvar} returns expected
values and standard deviations for \eqn{e^Y}.

Confidence intervals are calculated with an approximation that is valid when the number of observations is large.
Intervals must be treated cautiously in case of a small number of observations.

This function is identical to the function \code{\link{predict.lmvar}} in which the parameters \code{X_mu} and
\code{X_sigma} are left unspecified.
}
\examples{
# As example we use the dataset 'attenu' from the library 'datasets'. The dataset contains
# the response variable 'accel' and two explanatory variables 'mag'  and 'dist'.
library(datasets)

# Create the model matrix for the expected values
X = cbind(attenu$mag, attenu$dist)
colnames(X) = c("mag", "dist")

# Create the model matrix for the standard deviations.
X_s = cbind(attenu$mag, 1 / attenu$dist)
colnames(X_s) = c("mag", "dist_inv")

# Carry out the fit
y = attenu$accel
fit = lmvar(y, X, X_s)

# Calculate the expected value of each observation
fitted(fit, sigma = FALSE)

# Calculate the standard deviation of each observation
fitted(fit, mu = FALSE)

# Calculate the expected values and their 95\% confidence intervals
fitted(fit, sigma = FALSE, interval = "confidence")

# Calculate the standard deviations and their 80\% confidence intervals
fitted(fit, mu = FALSE, interval = "confidence", level = 0.8)

# Calculate both the expected values and the standard deviations
fitted(fit)

# Calculate the expected values, the standard deviations and their 95\% confidence intervals
fitted(fit, interval = "confidence")

# Calculate the expected values and the 90\% prediction intervals
fitted(fit, interval = "prediction", level = 0.9)

# Fit the log of 'accel'
y = log(attenu$accel)
fit_log = lmvar(y, X, X_s)

# Calculate both the expected values and the standard deviations of the log of 'accel'
fitted(fit_log)

# Calculate the expected values and the standard deviations of 'accel'
fitted(fit_log, log = TRUE)

# Calculate the expected values and the standard deviations of 'accel',
# as well as their 90\% confidence intervals
fitted(fit_log, log = TRUE, interval = "confidence", level = 0.9)


}
\seealso{
\code{\link{predict.lmvar}} for expected values, standard deviations and intervals for model matrices different from
the ones present in \code{object}.

\code{\link{coef.lmvar}} and \code{\link[stats]{confint}} for maximum likelihood estimators and confidence intervals for
\eqn{\beta_\mu} and \eqn{\beta_\sigma}.
}
