% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lfmm.R
\name{glm_test}
\alias{glm_test}
\title{GLM tests with latent factor mixed models}
\usage{
glm_test(Y, X, lfmm.obj, calibrate = "gif", family = binomial(link =
  "logit"))
}
\arguments{
\item{Y}{a response variable matrix with n rows and p columns.
Each column is a response variable (numeric).}

\item{X}{an explanatory variable matrix with n rows and d columns.
Each column corresponds to an explanatory variable (numeric).}

\item{lfmm.obj}{an object of class \code{lfmm} returned by the \link{lfmm_lasso}
or \link{lfmm_ridge} function}

\item{calibrate}{a character string, "gif". If the "gif" option is set (default),
significance values are calibrated by using the genomic control method. Genomic control
uses a robust estimate of the variance of z-scores called "genomic inflation factor".}

\item{family}{a description of the error distribution and link function to be used
in the model. For glm this can be a character string naming a family function,
a family function or the result of a call to a family function.}
}
\value{
a list with the following attributes:
\itemize{
\item B the effect size matrix with dimensions p x d.
\item score a p x d matrix which contains z-scores for each explanatory variable (columns of X),
\item pvalue a p x d matrix which contains p-values for each explanatory variable,
\item calibrated.pvalue a p x d matrix which contains calibrated p-values for each explanatory variable,
\item gif a numeric value for the genomic inflation factor.
}
}
\description{
This function returns significance values for the association between each column of the
response matrix, Y, and the explanatory variables, X, including correction for unobserved confounders
(latent factors). The test is based on an LFMM fitted with a ridge or lasso penalty and a generalized linear
model.
}
\details{
The response variable matrix Y and the explanatory variable are centered.
}
\examples{

library(lfmm)


## An EWAS example with Y = methylation data 
## and X = "exposure" 
## Simulate the data

dat <- lfmm_sampler(n = 100, 
                    p = 500,
                    K = 3,
                    outlier.prop = 0.01,
                    cs = 0.1,
                    sigma = 0.2,
                    B.sd = 5,
                    B.mean = 0,
                    U.sd = 1.0,
                    V.sd = 1.0)
                    
Y <- pnorm(dat$Y)
X <- dat$X

## Fit an LFMM with 2 latent factors
mod.lfmm <- lfmm_ridge(Y = Y,
                       X = X, 
                       K = 3)
                       
## Perform association testing using the fitted model:
pv <- glm_test(Y = pnorm(Y), 
                X = X,
                lfmm.obj = mod.lfmm, 
                family = binomial(link = "probit"),
                calibrate = "gif")
                
## Manhattan plot with true associations shown
causal <- dat$outlier
pvalues <- pv$calibrated.pvalue
plot(-log10(pvalues), 
     pch = 19, 
     cex = .3,
     xlab = "Probe",
     col = "grey")
     
points(causal, 
      -log10(pvalues)[causal], 
       col = "blue")
}
\seealso{
\link{lfmm_test}
}
\author{
Kevin Caye, Basile Jumentier, Olivier Francois
}
