% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summarizeCosts.R
\name{summarizeCosts}
\alias{summarizeCosts}
\title{Summarize costs of invasions over periods of time}
\usage{
summarizeCosts(
  costdb,
  cost.column = "Cost_estimate_per_year_2017_USD_exchange_rate",
  year.column = "Impact_year",
  in.millions = TRUE,
  minimum.year = 1960,
  maximum.year = max(costdb[, year.column]),
  year.breaks = seq(minimum.year, maximum.year, by = 10),
  include.last.year = TRUE
)
}
\arguments{
\item{costdb}{The \bold{expanded 'InvaCost' database} output from 
\code{\link{expandYearlyCosts}},
where costs occurring over several years are expanded over each year of impact.}

\item{cost.column}{Name of the cost column to use in \code{costdb} (usually, 
choose between the exchange rate (default) or Purchase Power parity cost per
year)}

\item{year.column}{Name of the year column to use in \code{costdb}( usually, 
"Impact_year" from \code{\link{expandYearlyCosts}}}

\item{in.millions}{If \code{TRUE}, cost values are transformed in 
millions (to make graphs easier to read), otherwise if \code{FALSE}, cost
values are
not transformed.}

\item{minimum.year}{The starting year of the analysis. By default, 
1960 was chosen because it marks the period from which world bank data is 
available for exchange rates and inflation values.}

\item{maximum.year}{The ending year for the analysis. By default, the last
year of \code{costdb} is chosen.}

\item{year.breaks}{A vector of breaks for the year intervals over which
you want to summarize cost values}

\item{include.last.year}{\code{TRUE} or \code{FALSE}. Defines if the last
year of the dataset is included in the last interval (\code{TRUE}) or is
considered as an interval of its own (\code{FALSE}). Generally only useful
if the last year is at the limit of an interval.}
}
\value{
A \code{list} with 6 elements:

\itemize{
\item{\code{cost.data}: the input data}
\item{\code{parameters}: parameters used to run the function. The 
\code{minimum.year} and \code{maximum.year} are based on the input data
(i.e., the user may specify \code{minimum.year = 1960} but the input data may
only have data starting from 1970, hence the \code{minimum.year} will be
 1970)}
\item{\code{year.breaks}: the years used to define year intervals over which costs were calculated.}
\item{\code{cost.per.year}: the costs of invasions expressed per year, 
as sums of all costs for each year}
\item{\code{average.total.cost}: the average annual cost of invasive Alien
species calculated
over the entire time period}
\item{\code{average.cost.per.period}: a data.frame containing the average 
annual cost of invasive alien species calculated over each time interval}
}
The structure of this object can be seen using \code{str()}.
}
\description{
This function summarizes the cumulative costs and average annual costs of
invasive alien species and breaks it down into regular periods of time, on 
the basis
of cost estimates as they appeared in the provided in the source references
collected in the 'InvaCost' database
}
\details{
Missing data are ignored. However, note that the average for each 
interval is always calculated on the basis of the full temporal range.
For example, if there is only data for 1968 for the 1960-1969 interval,
then the total cost for the interval will be equal to the cost of 1968, and the
average annual cost for 1960-1969 will be the cost of 1968 / 10.
}
\examples{
data(invacost)

### Cleaning steps
# Eliminating data with no information on starting and ending years
invacost <- invacost[-which(is.na(invacost$Probable_starting_year_adjusted)), ]
invacost <- invacost[-which(is.na(invacost$Probable_ending_year_adjusted)), ]
# Keeping only observed and reliable costs
invacost <- invacost[invacost$Implementation == "Observed", ]
invacost <- invacost[which(invacost$Method_reliability == "High"), ]
# Eliminating data with no usable cost value
invacost <- invacost[-which(is.na(invacost$Cost_estimate_per_year_2017_USD_exchange_rate)), ]

### Expansion
\donttest{
db.over.time <- expandYearlyCosts(invacost,
                                  startcolumn = "Probable_starting_year_adjusted",
                                  endcolumn = "Probable_ending_year_adjusted")
                                  
### Analysis
res <- summarizeCosts(db.over.time,
                      maximum.year = 2020) # Excluding data after 2020 (e.g. planned budgets)
res}
}
\references{
\url{https://github.com/Farewe/invacost}

Leroy Boris, Kramer Andrew M, Vaissière Anne-Charlotte, Kourantidou Melina,
Courchamp Franck & Diagne Christophe (2022). Analysing economic costs 
of invasive alien species with the invacost R package. Methods in Ecology
and Evolution. \doi{10.1111/2041-210X.13929}
}
\seealso{
\code{\link{expandYearlyCosts}} to get the database in appropriate format.
}
\author{
Boris Leroy \email{leroy.boris@gmail.com}, Andrew Kramer, Anne-Charlotte
Vaissière, Christophe Diagne
}
