% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/armitage.R
\name{armitage_eval}
\alias{armitage_eval}
\title{Evaluate the updated Armitage model}
\usage{
armitage_eval(
  casrn.vector = NA_character_,
  nomconc.vector = 1,
  this.well_number = 384,
  this.FBSf = NA_real_,
  tcdata = NA,
  this.sarea = NA_real_,
  this.v_total = NA_real_,
  this.v_working = NA_real_,
  this.cell_yield = NA_real_,
  this.Tsys = 37,
  this.Tref = 298.15,
  this.option.kbsa2 = FALSE,
  this.option.swat2 = FALSE,
  this.pseudooct = 0.01,
  this.memblip = 0.04,
  this.nlom = 0.2,
  this.P_nlom = 0.035,
  this.P_dom = 0.05,
  this.P_cells = 1,
  this.csalt = 0.15,
  this.celldensity = 1,
  this.cellmass = 3,
  this.f_oc = 1,
  this.conc_ser_alb = 24,
  this.conc_ser_lip = 1.9,
  this.Vdom = 0,
  this.pH = 7,
  restrict.ion.partitioning = FALSE
)
}
\arguments{
\item{casrn.vector}{For vector or single value, CAS number}

\item{nomconc.vector}{For vector or single value, micromolar (uM = mol/L) nominal 
concentration (e.g. AC50 value)}

\item{this.well_number}{For single value, plate format default is 384, used
if is.na(tcdata)==TRUE. This value chooses default surface area settings for
\code{\link{armitage_estimate_sarea}} based on the number of plates per well.}

\item{this.FBSf}{Fraction fetal bovine serum, must be entered by user.}

\item{tcdata}{A data.table with casrn, nomconc, MP, gkow, gkaw, gswat, sarea,
v_total, v_working. Otherwise supply single values to this.params (e.g., this.sarea,
this.v_total, etc.). Chemical parameters are taken from 
\code{\link{chem.physical_and_invitro.data}}.}

\item{this.sarea}{Surface area per well (m^2)}

\item{this.v_total}{Total volume per well (uL)}

\item{this.v_working}{Working volume per well (uL)}

\item{this.cell_yield}{Number of cells per well}

\item{this.Tsys}{System temperature (degrees C)}

\item{this.Tref}{Reference temperature (degrees K)}

\item{this.option.kbsa2}{Use alternative bovine-serum-albumin partitioning
model}

\item{this.option.swat2}{Use alternative water solubility correction}

\item{this.pseudooct}{Pseudo-octanol cell storage lipid content}

\item{this.memblip}{Membrane lipid content of cells}

\item{this.nlom}{Structural protein content of cells}

\item{this.P_nlom}{Proportionality constant to octanol structural protein}

\item{this.P_dom}{Proportionality constant to dissolve organic material}

\item{this.P_cells}{Proportionality constant to octanol storage lipid}

\item{this.csalt}{Ionic strength of buffer (M = mol/L)}

\item{this.celldensity}{Cell density kg/L, g/mL}

\item{this.cellmass}{Mass per cell, ng/cell}

\item{this.f_oc}{Everything assumed to be like proteins}

\item{this.conc_ser_alb}{Mass concentration of albumin in serum (g/L)}

\item{this.conc_ser_lip}{Mass concentration of lipids in serum (g/L)}

\item{this.Vdom}{0 ml, the volume of dissolved organic matter (DOM)}

\item{this.pH}{7.0, pH of cell culture}

\item{restrict.ion.partitioning}{FALSE, Should we restrict the chemical available to partition to only the neutral fraction?}
}
\value{
\tabular{lll}{
\strong{Param} \tab \strong{Description} \tab \strong{Units} \cr
casrn \tab Chemical Abstracts Service Registry Number \tab character \cr
nomconc \tab Nominal Concentration \tab uM=umol/L \cr       
well_number \tab Number of wells in plate (used to set default surface area) \tab unitless \cr   
sarea \tab Surface area of well \tab m^2 \cr         
v_total \tab Total volume of well \tab uL \cr       
v_working \tab Filled volume of well \tab uL \cr     
cell_yield \tab Number of cells \tab cells \cr    
gkow \tab The log10 octanol to water (PC) (logP)\tab log10 unitless ratio \cr          
logHenry \tab The log10 Henry's law constant '\tab log10 unitless ratio \cr      
gswat \tab The log10 water solubility (logWSol) \tab log10 mg/L \cr         
MP \tab The chemical compound melting point \tab degrees Kelvin \cr           
MW \tab The chemical compound molecular weight \tab g/mol \cr            
gkaw \tab The air to water PC \tab unitless ratio \cr          
dsm \tab \tab \cr           
duow \tab \tab \cr          
duaw \tab \tab \cr          
dumw \tab \tab \cr          
gkmw \tab log10 \tab \cr          
gkcw \tab The log10 cell/tissue to water PC \tab log10 unitless ratio\cr          
gkbsa \tab The log10 bovine serum albumin to water partitiion coefficient \tab unitless \cr         
gkpl \tab log10\tab \cr          
ksalt \tab Setschenow constant \tab L/mol \cr        
Tsys \tab System temperature \tab degrees C \cr          
Tref \tab Reference temperature\tab degrees K \cr          
option.kbsa2 \tab Use alternative bovine-serum-albumin partitioning model \tab logical \cr  
option.swat2 \tab Use alternative water solubility correction \tab logical \cr  
FBSf \tab Fraction fetal bovine serum \tab unitless \cr          
pseudooct \tab Pseudo-octanol cell storage lipid content \tab \cr     
memblip \tab Membrane lipid content of cells \tab  \cr       
nlom \tab Structural protein content of cells \tab \cr
P_nlom \tab Proportionality constant to octanol structural protein \tab unitless \cr   
P_dom \tab Proportionality constant to dissolved organic material (DOM) \tab unitless \cr         
P_cells \tab Proportionality constant to octanol storage lipid \tab unitless \cr      
csalt \tab Ionic strength of buffer \tab M=mol/L \cr
celldensity \tab Cell density \tab kg/L, g/mL \cr   
cellmass \tab Mass per cell \tab ng/cell \cr      
f_oc \tab \tab \cr
cellwat \tab \tab \cr       
Tcor \tab \tab \cr          
Vm \tab Volume of media \tab L \cr            
Vwell \tab Volume of medium (aqueous phase only) \tab L \cr         
Vair \tab Volume of head space \tab L \cr          
Vcells \tab Volume of cells/tissue\tab L \cr        
Valb \tab Volume of serum albumin \tab L \cr         
Vslip \tab Volume of serum lipids \tab L \cr         
Vdom \tab Volume of dissolved organic matter\tab L \cr          
F_ratio \tab \tab \cr       
gs1.GSE \tab \tab \cr       
s1.GSE \tab \tab \cr        
gss.GSE \tab \tab \cr       
ss.GSE \tab \tab \cr        
kmw \tab \tab \cr           
kow \tab The octanol to water PC (i.e., 10^gkow) \tab unitless \cr           
kaw \tab The air to water PC (i.e., 10^gkaw) \tab unitless \cr           
swat \tab The water solubility (i.e., 10^gswat) \tab mg/L \cr         
kpl \tab \tab \cr           
kcw \tab The cell/tissue to water PC (i.e., 10^gkcw) \tab unitless \cr           
kbsa \tab The bovine serum albumin to water PC \tab unitless \cr          
swat_L \tab \tab \cr        
soct_L \tab \tab \cr        
scell_L \tab \tab \cr       
cinit \tab Initial concentration \tab uM=umol/L \cr         
mtot \tab Total micromoles \tab umol \cr          
cwat \tab Total concentration in water \tab uM=umol/L \cr          
cwat_s \tab Dissolved concentration in water \tab uM=umol/L \cr        
csat \tab Is the solution saturated (1/0) \tab logical \cr         
activity \tab \tab \cr      
cair \tab Concentration in head space\tab uM=umol/L \cr          
calb \tab Concentration in serum albumin\tab uM=umol/L \cr          
cslip \tab Concentration in serum lipids\tab uM=umol/L \cr         
cdom \tab Concentration in dissolved organic matter\tab uM=umol/L \cr          
ccells \tab Concentration in cells\tab uM=umol/L \cr        
cplastic \tab Concentration in plastic\tab uM=umol/L \cr      
mwat_s \tab Mass dissolved in water \tab umols \cr        
mair \tab Mass in air/head space \tab umols \cr          
mbsa \tab Mass bound to bovine serum albumin \tab umols \cr          
mslip \tab Mass bound to serum lipids \tab umols \cr        
mdom \tab Mass bound to dissolved organic matter \tab umols \cr          
mcells \tab Mass in cells \tab umols \cr        
mplastic \tab Mass bond to plastic \tab umols \cr      
mprecip \tab Mass precipitated out of solution \tab umols\cr       
xwat_s \tab Fraction dissolved in water \tab fraction \cr        
xair \tab Fraction in the air \tab fraction \cr          
xbsa \tab Fraction bound to bovine serum albumin \tab fraction \cr          
xslip \tab Fraction bound to serum lipids \tab fraction \cr         
xdom \tab Fraction bound to dissolved organic matter \tab fraction \cr          
xcells \tab Fraction within cells \tab fraction \cr        
xplastic \tab Fraction bound to plastic \tab fraction \cr     
xprecip \tab Fraction precipitated out of solution \tab fraction \cr       
eta_free \tab Effective availability ratio \tab fraction \cr      
\strong{cfree.invitro} \tab \strong{Free concentration in the in vitro media} (use for Honda1 and Honda2) \tab fraction \cr
}
}
\description{
Evaluate the Armitage model for chemical distributon \emph{in vitro}. Takes input
as data table or vectors of values. Outputs a data table. Updates over
the model published in Armitage et al. (2014) include binding to plastic walls
and lipid and protein compartments in cells.
}
\examples{

library(httk)

# Check to see if we have info on the chemical:
"80-05-7" \%in\% get_cheminfo()

#We do:
temp <- armitage_eval(casrn.vector = c("80-05-7", "81-81-2"), this.FBSf = 0.1,
this.well_number = 384, nomconc = 10)
print(temp$cfree.invitro)

# Check to see if we have info on the chemical:
"793-24-8" \%in\% get_cheminfo()

# Since we don't have any info, let's look up phys-chem from dashboard:
cheminfo <- data.frame(
  Compound="6-PPD",
  CASRN="793-24-8",
  DTXSID="DTXSID9025114",
  logP=4.27, 
  logHenry=log10(7.69e-8),
  logWSol=log10(1.58e-4),
  MP=	99.4,
  MW=268.404
  )
  
# Add the information to HTTK's database:
chem.physical_and_invitro.data <- add_chemtable(
 cheminfo,
 current.table=chem.physical_and_invitro.data,
 data.list=list(
 Compound="Compound",
 CAS="CASRN",
  DTXSID="DTXSID",
  MW="MW",
  logP="logP",
  logHenry="logHenry",
  logWSol="logWSol",
  MP="MP"),
  species="Human",
  reference="CompTox Dashboard 31921")

# Run the Armitage et al. (2014) model:
out <- armitage_eval(
  casrn.vector = "793-24-8", 
  this.FBSf = 0.1,
  this.well_number = 384, 
  nomconc = 10)
  
print(out)

}
\references{
Armitage, J. M.; Wania, F.; Arnot, J. A. Environ. Sci. Technol. 
2014, 48, 9770-9779. https://doi.org/10.1021/es501955g

\insertRef{honda2019using}{httk}
}
\author{
Greg Honda
}
