/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HLmodule.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5Cprivate.h"  
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5HLpkg.h"     
#include "H5MMprivate.h" 

#define H5HL_VERSION 0 

#define H5HL_SPEC_READ_SIZE 512

static herr_t H5HL__cache_prefix_get_initial_load_size(void *udata, size_t *image_len);
static herr_t H5HL__cache_prefix_get_final_load_size(const void *_image, size_t image_len, void *udata,
                                                     size_t *actual_len);
static void  *H5HL__cache_prefix_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5HL__cache_prefix_image_len(const void *thing, size_t *image_len);
static herr_t H5HL__cache_prefix_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5HL__cache_prefix_free_icr(void *thing);

static herr_t H5HL__cache_datablock_get_initial_load_size(void *udata, size_t *image_len);
static void  *H5HL__cache_datablock_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5HL__cache_datablock_image_len(const void *thing, size_t *image_len);
static herr_t H5HL__cache_datablock_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5HL__cache_datablock_notify(H5C_notify_action_t action, void *_thing);
static herr_t H5HL__cache_datablock_free_icr(void *thing);

static herr_t H5HL__hdr_deserialize(H5HL_t *heap, const uint8_t *image, size_t len,
                                    H5HL_cache_prfx_ud_t *udata);

static herr_t H5HL__fl_deserialize(H5HL_t *heap);
static void   H5HL__fl_serialize(const H5HL_t *heap);

const H5AC_class_t H5AC_LHEAP_PRFX[1] = {{
    H5AC_LHEAP_PRFX_ID,                       
    "local heap prefix",                      
    H5FD_MEM_LHEAP,                           
    H5AC__CLASS_SPECULATIVE_LOAD_FLAG,        
    H5HL__cache_prefix_get_initial_load_size, 
    H5HL__cache_prefix_get_final_load_size,   
    NULL,                                     
    H5HL__cache_prefix_deserialize,           
    H5HL__cache_prefix_image_len,             
    NULL,                                     
    H5HL__cache_prefix_serialize,             
    NULL,                                     
    H5HL__cache_prefix_free_icr,              
    NULL,                                     
}};

const H5AC_class_t H5AC_LHEAP_DBLK[1] = {{
    H5AC_LHEAP_DBLK_ID,                          
    "local heap datablock",                      
    H5FD_MEM_LHEAP,                              
    H5AC__CLASS_NO_FLAGS_SET,                    
    H5HL__cache_datablock_get_initial_load_size, 
    NULL,                                        
    NULL,                                        
    H5HL__cache_datablock_deserialize,           
    H5HL__cache_datablock_image_len,             
    NULL,                                        
    H5HL__cache_datablock_serialize,             
    H5HL__cache_datablock_notify,                
    H5HL__cache_datablock_free_icr,              
    NULL,                                        
}};

static herr_t
H5HL__hdr_deserialize(H5HL_t *heap, const uint8_t *image, size_t len, H5HL_cache_prfx_ud_t *udata)
{
    const uint8_t *p_end     = image + len - 1; 
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(heap);
    assert(image);
    assert(udata);

    
    if (H5_IS_BUFFER_OVERFLOW(image, H5_SIZEOF_MAGIC, p_end))
        HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    if (memcmp(image, H5HL_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad local heap signature");
    image += H5_SIZEOF_MAGIC;

    
    if (H5_IS_BUFFER_OVERFLOW(image, 1, p_end))
        HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    if (H5HL_VERSION != *image++)
        HGOTO_ERROR(H5E_HEAP, H5E_VERSION, FAIL, "wrong version number in local heap");

    
    if (H5_IS_BUFFER_OVERFLOW(image, 3, p_end))
        HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    image += 3;

    
    heap->prfx_addr = udata->prfx_addr;
    heap->prfx_size = udata->sizeof_prfx;

    
    if (H5_IS_BUFFER_OVERFLOW(image, udata->sizeof_size, p_end))
        HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    H5_DECODE_LENGTH_LEN(image, heap->dblk_size, udata->sizeof_size);

    
    if (H5_IS_BUFFER_OVERFLOW(image, udata->sizeof_size, p_end))
        HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    H5_DECODE_LENGTH_LEN(image, heap->free_block, udata->sizeof_size);
    if (heap->free_block != H5HL_FREE_NULL && heap->free_block >= heap->dblk_size)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad heap free list");

    
    if (H5_IS_BUFFER_OVERFLOW(image, udata->sizeof_addr, p_end))
        HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    H5F_addr_decode_len(udata->sizeof_addr, &image, &(heap->dblk_addr));

    
    if (!H5_addr_defined(heap->dblk_addr))
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad datablock address");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HL__fl_deserialize(H5HL_t *heap)
{
    H5HL_free_t *fl = NULL, *tail = NULL; 
    hsize_t      free_block;              
    herr_t       ret_value = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    assert(heap);
    assert(!heap->freelist);
    HDcompile_assert(sizeof(hsize_t) == sizeof(uint64_t));

    
    free_block = heap->free_block;
    while (H5HL_FREE_NULL != free_block) {
        const uint8_t *image; 

        

        if (free_block > UINT64_MAX - (2 * heap->sizeof_size))
            HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "decoded heap block address overflow");

        if ((free_block + (2 * heap->sizeof_size)) > heap->dblk_size)
            HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "bad heap free list");

        
        if (NULL == (fl = H5FL_MALLOC(H5HL_free_t)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "memory allocation failed");
        fl->offset = (size_t)free_block;
        fl->prev   = tail;
        fl->next   = NULL;

        
        image = heap->dblk_image + free_block;
        H5_DECODE_LENGTH_LEN(image, free_block, heap->sizeof_size);
        if (0 == free_block)
            HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "free block size is zero?");

        
        H5_DECODE_LENGTH_LEN(image, fl->size, heap->sizeof_size);
        if ((fl->offset + fl->size) > heap->dblk_size)
            HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "bad heap free list");

        
        if (tail)
            tail->next = fl;
        else
            heap->freelist = fl;
        tail = fl;
        fl   = NULL;
    }

done:
    if (ret_value < 0)
        if (fl)
            
            fl = H5FL_FREE(H5HL_free_t, fl);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static void
H5HL__fl_serialize(const H5HL_t *heap)
{
    H5HL_free_t *fl; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(heap);

    
    for (fl = heap->freelist; fl; fl = fl->next) {
        uint8_t *image; 

        assert(fl->offset == H5HL_ALIGN(fl->offset));
        image = heap->dblk_image + fl->offset;

        if (fl->next)
            H5_ENCODE_LENGTH_LEN(image, fl->next->offset, heap->sizeof_size);
        else
            H5_ENCODE_LENGTH_LEN(image, H5HL_FREE_NULL, heap->sizeof_size);

        H5_ENCODE_LENGTH_LEN(image, fl->size, heap->sizeof_size);
    }

    FUNC_LEAVE_NOAPI_VOID

} 

static herr_t
H5HL__cache_prefix_get_initial_load_size(void H5_ATTR_UNUSED *_udata, size_t *image_len)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(image_len);

    
    *image_len = H5HL_SPEC_READ_SIZE;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HL__cache_prefix_get_final_load_size(const void *_image, size_t image_len, void *_udata, size_t *actual_len)
{
    const uint8_t        *image = (const uint8_t *)_image;        
    H5HL_cache_prfx_ud_t *udata = (H5HL_cache_prfx_ud_t *)_udata; 
    H5HL_t                heap;                                   
    herr_t                ret_value = SUCCEED;                    

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(udata);
    assert(actual_len);
    assert(*actual_len == image_len);

    memset(&heap, 0, sizeof(H5HL_t));

    
    if (H5HL__hdr_deserialize(&heap, (const uint8_t *)image, image_len, udata) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDECODE, FAIL, "can't decode local heap header");

    
    *actual_len = heap.prfx_size;

    
    if (heap.dblk_size)
        
        if (H5_addr_eq((heap.prfx_addr + heap.prfx_size), heap.dblk_addr))
            
            *actual_len += heap.dblk_size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5HL__cache_prefix_deserialize(const void *_image, size_t len, void *_udata, bool H5_ATTR_UNUSED *dirty)
{
    H5HL_t               *heap      = NULL;                           
    H5HL_prfx_t          *prfx      = NULL;                           
    const uint8_t        *image     = (const uint8_t *)_image;        
    const uint8_t        *p_end     = image + len - 1;                
    H5HL_cache_prfx_ud_t *udata     = (H5HL_cache_prfx_ud_t *)_udata; 
    void                 *ret_value = NULL;                           

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(len > 0);
    assert(udata);
    assert(udata->sizeof_size > 0);
    assert(udata->sizeof_addr > 0);
    assert(udata->sizeof_prfx > 0);
    assert(H5_addr_defined(udata->prfx_addr));
    assert(dirty);

    
    if (NULL == (heap = H5HL__new(udata->sizeof_size, udata->sizeof_addr, udata->sizeof_prfx)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, NULL, "can't allocate local heap structure");

    
    if (H5HL__hdr_deserialize(heap, (const uint8_t *)image, len, udata) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDECODE, NULL, "can't decode local heap header");

    
    if (NULL == (prfx = H5HL__prfx_new(heap)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, NULL, "can't allocate local heap prefix");

    
    if (heap->dblk_size) {
        
        if (H5_addr_eq((heap->prfx_addr + heap->prfx_size), heap->dblk_addr)) {
            
            heap->single_cache_obj = true;

            
            if (NULL == (heap->dblk_image = H5FL_BLK_MALLOC(lheap_chunk, heap->dblk_size)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, NULL, "memory allocation failed");

            
            image = ((const uint8_t *)_image) + heap->prfx_size;

            
            if (H5_IS_BUFFER_OVERFLOW(image, heap->dblk_size, p_end))
                HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            H5MM_memcpy(heap->dblk_image, image, heap->dblk_size);

            
            if (H5HL__fl_deserialize(heap) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, NULL, "can't initialize free list");
        }
        else
            
            heap->single_cache_obj = false;
    }

    
    ret_value = prfx;

done:
    
    if (!ret_value) {
        if (prfx) {
            if (FAIL == H5HL__prfx_dest(prfx))
                HDONE_ERROR(H5E_HEAP, H5E_CANTRELEASE, NULL, "unable to destroy local heap prefix");
        }
        else {
            if (heap && FAIL == H5HL__dest(heap))
                HDONE_ERROR(H5E_HEAP, H5E_CANTRELEASE, NULL, "unable to destroy local heap");
        }
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HL__cache_prefix_image_len(const void *_thing, size_t *image_len)
{
    const H5HL_prfx_t *prfx = (const H5HL_prfx_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(prfx);
    assert(prfx->cache_info.type == H5AC_LHEAP_PRFX);
    assert(image_len);

    
    *image_len = prfx->heap->prfx_size;

    
    if (prfx->heap->single_cache_obj)
        *image_len += prfx->heap->dblk_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HL__cache_prefix_serialize(const H5_ATTR_NDEBUG_UNUSED H5F_t *f, void *_image,
                             size_t H5_ATTR_NDEBUG_UNUSED len, void *_thing)
{
    H5HL_prfx_t *prfx = (H5HL_prfx_t *)_thing; 
    H5HL_t      *heap;                         
    uint8_t     *image = (uint8_t *)_image;    

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(image);
    assert(prfx);
    assert(prfx->cache_info.type == H5AC_LHEAP_PRFX);
    assert(H5_addr_eq(prfx->cache_info.addr, prfx->heap->prfx_addr));
    assert(prfx->heap);

    
    heap = prfx->heap;
    assert(heap);

#ifndef NDEBUG
    
    size_t buf_size = heap->prfx_size; 
    if (heap->single_cache_obj)
        buf_size += heap->dblk_size;
    assert(len == buf_size);
#endif

    
    heap->free_block = heap->freelist ? heap->freelist->offset : H5HL_FREE_NULL;

    
    H5MM_memcpy(image, H5HL_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;
    *image++ = H5HL_VERSION;
    *image++ = 0; 
    *image++ = 0; 
    *image++ = 0; 
    H5_ENCODE_LENGTH_LEN(image, heap->dblk_size, heap->sizeof_size);
    H5_ENCODE_LENGTH_LEN(image, heap->free_block, heap->sizeof_size);
    H5F_addr_encode_len(heap->sizeof_addr, &image, heap->dblk_addr);

    
    if (heap->single_cache_obj) {
        if ((size_t)(image - (uint8_t *)_image) < heap->prfx_size) {
            size_t gap; 

            
            gap = heap->prfx_size - (size_t)(image - (uint8_t *)_image);
            memset(image, 0, gap);
            image += gap;
        }

        
        H5HL__fl_serialize(heap);

        
        H5MM_memcpy(image, heap->dblk_image, heap->dblk_size);

        
        assert((size_t)(image - (uint8_t *)_image) + heap->dblk_size == len);
    }
    else {
        
        assert((size_t)(image - (uint8_t *)_image) <= len);

        
        memset(image, 0, len - (size_t)(image - (uint8_t *)_image));
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HL__cache_prefix_free_icr(void *_thing)
{
    H5HL_prfx_t *prfx      = (H5HL_prfx_t *)_thing; 
    herr_t       ret_value = SUCCEED;               

    FUNC_ENTER_PACKAGE

    
    assert(prfx);
    assert(prfx->cache_info.type == H5AC_LHEAP_PRFX);
    assert(H5_addr_eq(prfx->cache_info.addr, prfx->heap->prfx_addr));

    
    if (H5HL__prfx_dest(prfx) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't destroy local heap prefix");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HL__cache_datablock_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5HL_t *heap = (H5HL_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(heap);
    assert(heap->dblk_size > 0);
    assert(image_len);

    
    *image_len = heap->dblk_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5HL__cache_datablock_deserialize(const void *image, size_t len, void *_udata, bool H5_ATTR_UNUSED *dirty)
{
    H5HL_dblk_t *dblk      = NULL;             
    H5HL_t      *heap      = (H5HL_t *)_udata; 
    void        *ret_value = NULL;             

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(len > 0);
    assert(heap);
    assert(heap->dblk_size == len);
    assert(!heap->single_cache_obj);
    assert(NULL == heap->dblk);
    assert(dirty);

    
    if (NULL == (dblk = H5HL__dblk_new(heap)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, NULL, "memory allocation failed");

    
    if (NULL == heap->dblk_image) {
        
        if (NULL == (heap->dblk_image = H5FL_BLK_MALLOC(lheap_chunk, heap->dblk_size)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, NULL, "can't allocate data block image buffer");

        
        H5MM_memcpy(heap->dblk_image, image, len);

        
        if (FAIL == H5HL__fl_deserialize(heap))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, NULL, "can't initialize free list");
    }

    
    ret_value = dblk;

done:
    
    if (!ret_value && dblk)
        if (FAIL == H5HL__dblk_dest(dblk))
            HDONE_ERROR(H5E_HEAP, H5E_CANTRELEASE, NULL, "unable to destroy local heap data block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HL__cache_datablock_image_len(const void *_thing, size_t *image_len)
{
    const H5HL_dblk_t *dblk = (const H5HL_dblk_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dblk);
    assert(dblk->cache_info.type == H5AC_LHEAP_DBLK);
    assert(dblk->heap);
    assert(dblk->heap->dblk_size > 0);
    assert(image_len);

    *image_len = dblk->heap->dblk_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HL__cache_datablock_serialize(const H5F_t H5_ATTR_NDEBUG_UNUSED *f, void *image,
                                size_t H5_ATTR_NDEBUG_UNUSED len, void *_thing)
{
    H5HL_t      *heap;                         
    H5HL_dblk_t *dblk = (H5HL_dblk_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(image);
    assert(dblk);
    assert(dblk->cache_info.type == H5AC_LHEAP_DBLK);
    assert(dblk->heap);
    heap = dblk->heap;
    assert(heap->dblk_size == len);
    assert(!heap->single_cache_obj);

    
    heap->free_block = heap->freelist ? heap->freelist->offset : H5HL_FREE_NULL;

    
    H5HL__fl_serialize(heap);

    
    H5MM_memcpy(image, heap->dblk_image, heap->dblk_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HL__cache_datablock_notify(H5C_notify_action_t action, void *_thing)
{
    H5HL_dblk_t *dblk      = (H5HL_dblk_t *)_thing; 
    herr_t       ret_value = SUCCEED;               

    FUNC_ENTER_PACKAGE

    
    assert(dblk);

    switch (action) {
        case H5AC_NOTIFY_ACTION_AFTER_INSERT:
            
            break;

        case H5AC_NOTIFY_ACTION_AFTER_LOAD:
            
            assert(dblk->heap);
            assert(dblk->heap->prfx);

            
            if (FAIL == H5AC_pin_protected_entry(dblk->heap->prfx))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTPIN, FAIL, "unable to pin local heap prefix");
            break;

        case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
        case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
        case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
        case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
        case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
            
            break;

        case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
            
            assert(dblk->heap);
            assert(dblk->heap->prfx);

            
            if (FAIL == H5AC_unpin_entry(dblk->heap->prfx))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPIN, FAIL, "unable to unpin local heap prefix");
            break;

        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
            break;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HL__cache_datablock_free_icr(void *_thing)
{
    H5HL_dblk_t *dblk      = (H5HL_dblk_t *)_thing; 
    herr_t       ret_value = SUCCEED;               

    FUNC_ENTER_PACKAGE

    
    assert(dblk);
    assert(dblk->cache_info.type == H5AC_LHEAP_DBLK);

    
    if (H5HL__dblk_dest(dblk) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to destroy local heap data block");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
