% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/define_linearization_wrapper.R
\name{define_linearization_wrapper}
\alias{define_linearization_wrapper}
\title{Define a linearization wrapper}
\usage{
define_linearization_wrapper(linearization_function, arg_type,
  allow_factor = FALSE, arg_not_affected_by_domain = NULL,
  display_function = standard_display)
}
\arguments{
\item{linearization_function}{An R function with input the quantities 
used in the linearization formula and with output a list with two 
named element: \itemize{\item \code{lin}: a list of numerical vectors (most
of the time, only 1) which correspond to the value of the linearized 
variable \item \code{metadata}: a list of metadata to be used by the display
function (see \code{display_function} argument), including (for the
standard display function) \code{est} for the point-estimate and 
\code{n} for the number of observations used in the estimation.}}

\item{arg_type}{A named list with three character vectors describing 
the type of each argument of \code{linearization_function}: \itemize{
\item \code{data}: data argument(s), numerical vector(s) to be used in the
linearization formula \item \code{weight}: weight argument, numerical vector
to be used as row weights in the linearization formula \item \code{param}: 
parameters, non-data arguments (most of the time boolean) to be used to 
control some aspect of the linearization formula}}

\item{allow_factor}{A logical vector of length 1 (\code{FALSE} by default)
indicating whether factor variable are accepted as-is by the linearization
wrappers. This should be the case when the linearization function only has
one data argument (e.g. \code{total} or \code{mean} linearization formulae).}

\item{arg_not_affected_by_domain}{A character vector indicating the (data) 
arguments which should not be affected by domain-splitting. Such parameters
may appear in some complex linearization formula, for instance when the 
At-Risk of Poverty Rate (ARPR) is estimated by region but with a poverty 
line calculated at the national level.}

\item{display_function}{An R function which produces, for each variance 
estimation, the data.frame row to be displayed by the variance estimation 
wrapper. It uses three arguments: 
\itemize{\item \code{var} the estimated variance \item \code{metadata} the 
metadata associated with the estimation, especially the one outputted by 
\code{linearization_function} (e.g. \code{est}, \code{n}) \item \code{alpha} 
the level for the construction of confidence intervals (at execution time, 
its value is taken from the \code{alpha} argument of the variance wrapper.)}
The default display function (\code{standard_display}) uses standard metadata
to display usual variance indicator (variance, standard deviation, coefficient
of variation, confidence interval) broken down by linearization wrapper, domain 
(if any) and level (if the variable is a factor, see argument \code{allow_factor}).}
}
\value{
A function to be used within a variance estimation wrapper to perform
  a specific linearization (see examples). Its formals are the ones of 
  \code{linearization_function} with the addition of \code{by} and \code{where} 
  (for domain estimation, set to \code{NULL} by default).
}
\description{
Given a linearization \emph{function} (specific to an
  estimator), \code{define_linearization_wrapper} defines a 
  linearization \emph{wrapper} to be used together with
  \code{\link[=define_variance_wrapper]{variance estimation wrappers}}
  in order to make variance estimation easier. 
  This function is intended for \strong{advanced use only} (see Details), 
  standard linearization wrappers are included in the gustave package (see
  \code{\link[=linearization_wrapper_standard]{standard linearization wrappers}})
}
\details{
When the estimator is not the estimator of a total, the application of 
  analytical variance estimation formulae developed for the estimator of a total 
  is not straightforward (Deville, 1999). An asymptotically unbiased variance 
  estimator can nonetheless be obtained if the estimation of variance is performed
  on a variable obtained from the original data through a linearization step. 
  
  \code{define_linearization_wrapper} is the function used to create, given 
  a linearization \emph{function} implementing a given linearization 
  \emph{formula}, a linearization \emph{wrapper} which can be used together 
  with a variance wrapper. 
  
  Linearization wrappers are quite flexible tools
  to apply a variance function to an estimator requiring a linearization step
  (e.g. all estimators except the estimator of a total) with virtually no 
  additional complexity for the end-user. To some extent, linearization wrappers 
  can be seen as ggplot2 \code{geom_} and \code{stat_} functions: they help 
  the end-user in writing down what he or she wants without having to go too 
  deep into the details of the corresponding layers. 
  
  \code{\link[=linearization_wrapper_standard]{standard linearization wrappers}} 
  are included within the gustave package and  automatically added 
  to the variance estimation wrappers. New linearization wrappers can be defined
  using the \code{define_linearization_wrapper} and then explicitly added to the 
  variance estimation wrappers using the \code{objects_to_include} argument.
}
\examples{
### Example from the Information and communication technologies (ICT) survey

# The subset of the (simulated) ICT survey has the following features: 
# - stratified one-stage sampling design of 650 firms;
# - 612 responding firms, non-response correction through reweighting 
# in homogeneous response groups based on economic sub-sector and turnover;
# - calibration on margins (number of firms and turnover broken down
# by economic sub-sector).

# Step 1 : Dummy variance wrapper
# Note : see define_variance_wrapper() for a more 
# realistic variance function and examples.
variance_wrapper <- define_variance_wrapper(
  variance_function = function(y) abs(colSums(y)), 
  reference_id = ict_survey$firm_id, 
  default = list(id = "firm_id", weight = "w_calib")
)
variance_wrapper(ict_survey, total(speed_quanti))

# Step 2 : Redefine the mean linearization wrapper
# The mean() linearization wrapper defined in the gustave 
# package is bulit on top of the ratio() linearization wrapper.
variance_wrapper(ict_survey, mean(speed_quanti))

# Let's redefine it directly from the formula found for instance
# in (Caron, Deville, Sautory, 1998) and without handling NA
# values
mean2 <- define_linearization_wrapper(
  linearization_function = function(y, weight){
    est <- sum(y * weight) / sum(weight)
    lin <- (y - est) / sum(weight)
    list(
      lin = list(lin), 
      metadata = list(est = est, n = length(y))
    )
  },
  arg_type = list(data = "y", weight = "weight"),
  allow_factor = TRUE
)
variance_wrapper(ict_survey, mean(speed_quanti), mean2(speed_quanti))

}
\references{
Deville J.-C. (1999), "Variance estimation for complex statistics and 
  estimators: linearization and residual techniques", \emph{Survey Methodology}, 
  25:193–203
}
\seealso{
\code{\link[=linearization_wrapper_standard]{standard linearization wrappers}} 
  \code{\link{define_variance_wrapper}}
}
\author{
Martin Chevalier
}
