% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/drive_mv.R
\name{drive_mv}
\alias{drive_mv}
\title{Move a Drive file}
\usage{
drive_mv(file, path = NULL, name = NULL, overwrite = NA, verbose = TRUE)
}
\arguments{
\item{file}{Something that identifies the file of interest on your Google
Drive. Can be a name or path, a file id or URL marked with \code{\link[=as_id]{as_id()}}, or a
\code{\link{dribble}}.}

\item{path}{Specifies target destination for the new file on Google
Drive. Can be an actual path (character), a file id marked with \code{\link[=as_id]{as_id()}},
or a \code{\link{dribble}}. If specified as an actual path, it is best to explicitly
indicate if it's a folder by including a trailing slash, since it cannot
always be worked out from the context of the call. Defaults to current name.}

\item{name}{Character, new file name if not specified as part of
\code{path}. This will force \code{path} to be treated as a folder, even if it is
character and lacks a trailing slash. Defaults to current name.}

\item{overwrite}{Logical, indicating whether to check for a pre-existing file
at the targetted "filepath". The quotes around "filepath" refer to the fact
that Drive does not impose a 1-to-1 relationship between filepaths and files,
like a typical file system; read more about that in \code{\link[=drive_get]{drive_get()}}.
\itemize{
\item \code{NA} (default): Just do the operation, even if it results in multiple
files with the same filepath.
\item \code{TRUE}: Check for a pre-existing file at the filepath. If there is
zero or one, move a pre-existing file to the trash, then carry on. Note
that the new file does not inherit any properties from the old one, such
as sharing or publishing settings. It will have a new file ID. An error is
thrown if two or more pre-existing files are found.
\item \code{FALSE}: Error if there is any pre-existing file at the filepath.
}

Note that existence checks, based on filepath, are expensive operations, i.e.
they require additional API calls.}

\item{verbose}{Logical, indicating whether to print informative messages
(default \code{TRUE}).}
}
\value{
An object of class \code{\link{dribble}}, a tibble with one row per item.
}
\description{
Move a Drive file to a different folder, give it a different name, or both.
Note that folders on Google Drive are not like folders on your local
filesystem. They are more like a label, which implies that a Drive file can
have multiple folders as direct parent! However, most people still use and
think of them like "regular" folders. When we say "move a Drive file", it
actually means: "add a new folder to this file's parents and remove the old
one".
}
\examples{
\dontrun{
## create a file to move
file <- drive_upload(drive_example("chicken.txt"), "chicken-mv.txt")

## rename it, but leave in current folder (root folder, in this case)
file <- drive_mv(file, "chicken-mv-renamed.txt")

## create a folder to move the file into
folder <- drive_mkdir("mv-folder")

## move the file and rename it again,
## specify destination as a dribble
file <- drive_mv(file, path = folder, name = "chicken-mv-re-renamed.txt")

## verify renamed file is now in the folder
drive_ls(folder)

## move the file back to root folder
file <- drive_mv(file, "~/")

## move it again
## specify destination as path with trailing slash
## to ensure we get a move vs. renaming it to "mv-folder"
file <- drive_mv(file, "mv-folder/")

## `overwrite = FALSE` errors if something already exists at target filepath
## THIS WILL ERROR!
drive_create("name-squatter", path = "~/")
drive_mv(file, path = "~/", name = "name-squatter", overwrite = FALSE)

## `overwrite = TRUE` moves the existing item to trash, then proceeds
drive_mv(file, path = "~/", name = "name-squatter", overwrite = TRUE)

## Clean up
drive_rm(file, folder)
}
}
