% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smap.R
\name{spmap}
\alias{spmap}
\alias{spmap_vec}
\alias{spmap_lgl}
\alias{spmap_int}
\alias{spmap_dbl}
\alias{spmap_chr}
\alias{spmap_structure}
\title{Map Functions Over Glycan Structure Vectors and Multiple Arguments}
\usage{
spmap(.l, .f, ..., .parallel = FALSE)

spmap_vec(.l, .f, ..., .ptype = NULL, .parallel = FALSE)

spmap_lgl(.l, .f, ..., .parallel = FALSE)

spmap_int(.l, .f, ..., .parallel = FALSE)

spmap_dbl(.l, .f, ..., .parallel = FALSE)

spmap_chr(.l, .f, ..., .parallel = FALSE)

spmap_structure(.l, .f, ..., .parallel = FALSE)
}
\arguments{
\item{.l}{A list where the first element is a glycan structure vector (glyrepr_structure)
and the remaining elements are vectors of the same length or length 1 (will be recycled).}

\item{.f}{A function that takes an igraph object (from first element of \code{.l}) and
values from other elements, returning a result.
Can be a function, purrr-style lambda (\code{~ .x + .y + .z}), or a character string naming a function.}

\item{...}{Additional arguments passed to \code{.f}.}

\item{.parallel}{Logical; whether to use parallel processing. If \code{FALSE} (default),
parallel processing is disabled. Set to \code{TRUE} to enable parallel processing.
See examples in \code{\link{smap}} for how to set up and use parallel processing.}

\item{.ptype}{A prototype for the return type (for \code{spmap_vec}).}
}
\value{
\itemize{
\item \code{spmap()}: A list
\item \code{spmap_vec()}: An atomic vector of type specified by \code{.ptype}
\item \code{spmap_lgl/int/dbl/chr()}: Atomic vectors of the corresponding type
\item \code{spmap_structure()}: A new glyrepr_structure object
}
}
\description{
These functions apply a function to each unique structure in a glycan structure vector
along with corresponding elements from multiple other vectors,
taking advantage of hash-based deduplication to avoid redundant computation.
Similar to purrr pmap functions, but optimized for glycan structure vectors.
}
\details{
These functions only compute \code{.f} once for each unique combination of structure and corresponding
values from other vectors, then map the results back to the original vector positions. This is much more efficient
than applying \code{.f} to each element combination individually when there are duplicate combinations.

\strong{Time Complexity Performance:}

Performance scales with unique combinations of all arguments rather than total vector length.
When argument vectors are highly redundant, performance approaches O(unique_structures).
Scaling factor shows time increase when vector size increases 20x.

\strong{Return Types:}
\itemize{
\item \code{spmap()}: Returns a list with the same length as the input vectors
\item \code{spmap_vec()}: Returns an atomic vector with the same length as the input vectors
\item \code{spmap_lgl()}: Returns a logical vector
\item \code{spmap_int()}: Returns an integer vector
\item \code{spmap_dbl()}: Returns a double vector
\item \code{spmap_chr()}: Returns a character vector
\item \code{spmap_structure()}: Returns a new glycan structure vector (\code{.f} must return igraph objects)
}
}
\examples{
# Create structure vectors with duplicates
core1 <- o_glycan_core_1()
core2 <- n_glycan_core()
structures <- glycan_structure(core1, core2, core1)  # core1 appears twice
weights <- c(1.0, 2.0, 1.0)  # corresponding weights
factors <- c(2, 3, 2)  # corresponding factors

# Map a function that uses structure, weight, and factor
spmap_dbl(list(structures, weights, factors), 
          function(g, w, f) igraph::vcount(g) * w * f)

# Use purrr-style lambda functions  
spmap_dbl(list(structures, weights, factors), ~ igraph::vcount(..1) * ..2 * ..3)

# Test with recycling
spmap_dbl(list(structures, 2.0, 3), ~ igraph::vcount(..1) * ..2 * ..3)

}
