C----------------------------------------------------------------------
C
C
C                             A L M I N I
C                             -----------
C
C
C         EXTENSION OF GEMINI TO AUGMENTED LAGRANGIAN FUNCTIONS
C
C                              VERSION 2
C                         DEC.79, J.M.LALOUEL
C                          MAR.85, S.RUSSELL
C
C -- OPTIMIZATION OF NONLINEAR FUNCTIONS SUBJECT TO NONLINEAR CONSTRAINTS --
C
C     REVISION HISTORY:
C      3/85 -- ORIGINAL VERSION 2, HARRIS FORTRAN 77 (SR)
C     10/85 -- IBM PC VERSION (SR)
C      9/87 -- SPLIT IDIM TO MAXPAR AND MAXITP IN ORDER TO ACCOMMODATE
C              LARGER LISTS OF NON-ITERATED PARAMETERS; ALSO CHANGED
C              PROLIX OUTPUT TO FIT 80 CHARACTER DISPLAYS; ADDED LEVEL
C              5 TO IVERB TO INDICATE REASON FOR EACH FUNCTION CALL (SR)
C     11/90 -- FLUSH OUT PROLIX AFTER EACH ITERATION (MAP)
C
C **************************************************************************
C
C     THIS PROGRAM IS AN EXTENSION OF GEMINI TO OPTIMIZE A NONLINEAR
C FUNCTION SUBJECT TO NONLINEAR INEQUALITY OR EQUALITY CONSTRAINTS.
C IT IS INTENDED FOR THAT CLASS OF PROBLEMS WHERE THE FUNCTION IS STILL
C DEFINED WHEN CONSTRAINTS ARE VIOLATED AND SOME CONDITIONS ON
C CONTINUITY ARE SATISFIED.
C
C     THE METHOD USED CONSISTS IN OPTIMIZING AN AUGMENTED LAGRANGIAN
C FUNCTION, FOLLOWING HESTENES(1969), POWELL(1969) AND ROCKAFELLAR
C (1973).
C
C     INEQUALITY CONSTRAINTS MUST BE WRITTEN IN THE FORM C(I).GE.0.0
C
C ---------------------------------------------------------------------
C
C                          G E M I N I
C                          -----------
C AN OPTIMIZATION ROUTINE FOR GENERAL NONLINEAR FUNCTIONS OF N VARIABLES
C
C                 VERSION 1, FEB. 79, J.M.LALOUEL
C
C     A VARIABLE METRIC ( OR QUASI-NEWTON ) ALGORITHM EVALUATING
C GRADIENT BY FINITE DIFFERENCE APPROXIMATION, WHILE AN APPROXIMATION
C TO THE HESSIAN IS OBTAINED BY RECURRENCE, IMPROVED AT EACH ITERATION
C BY AN UPDATING PROCEDURE.
C
C     THE PRESENT ALGORITHM USE RESULTS DUE TO DAVIDON(1959), FLETCHER
C AND POWELL(1963),STEWART(1967), GILL AND MURRAY(1972), DAVIDON(1975)
C AND GOLDFARB(1976).
C
C     THE MOST ESSENTIAL FEATURE OF THIS ALGORITHM CONCERNS THE
C UPDATING.  THE APPROXIMATION TO THE HESSIAN IS RECURRED IN FACTORIZED
C FORM, AND A MODIFICATION OF AT MOST RANK TWO IS PERFORMED, AT EACH
C ITERATION, IN REAL PRODUCT FORM ACCORDING TO THE METHOD OF GOLDFARB
C (1976).  IN ADDITION THE UPDATE IS OPTIMALLY CONDITIONED IN THE SENSE
C OF DAVIDON.
C
C ---------------------------------------------------------------------
C
C
C           SUMMARY OF INPUT SPECIFICATIONS FOR ALMINI
C           ------------------------------------------
C
C FILE ASSIGNEMENTS :
C     IP      = UNIT NUMBER FOR THE "PROLIX" FILE, WHICH WILL
C               CONTAIN A DETAILED ACCOUNT OF THE MINIMIZATION
C
C PARAMETERS CONTROLLING PRECISION AND TERMINATION :
C     H       = DIFFERENTIATION INTERVAL
C     TOL     = TOLERANCE ON MAGNITUDE OF NORMALIZED GRADIENT
C     TRUPB   = TRUNCATION UPPER BOUND
C     MAXIT   = MAXIMUM NUMBER OF ITERATIONS ( FOR SAFETY )
C
C INPUT DATA :
C     NALL    = TOTAL NUMBER OF ITERABLE PARAMETERS
C     XALL(I) = INITIAL PARAMETER VALUES (I=1,NALL)
C     ITP(I)  = VALUE OF 1 INDICATES ITERATED PARAMETERS (I=1,NALL)
C     BND(I,1)= LOWER BOUNDS ON PARAMETERS (I=1,NALL)
C     BND(I,2)= UPPER BOUNDS ON PARAMETERS (I=1,NALL)
C     TL(I,J) = INITIAL APPROXIMATION TO HESSIAN (OPTIONAL, SEE IHESS)
C
C OPTION INDICATORS :
C     IBND    = 0 : NO BOUNDARY CONDITIONS
C             = 1 : UPPER AND LOWER BOUNDS ARE SPECIFIED IN BND
C     IHESS   = 0 : NO SPECIFICATION OF INITIAL HESSIAN
C             = 1 : INITIAL HESSIAN OBTAINED BY FINITE DIFFERENCE
C             = 2 : INITIAL HESSIAN IS SUPPLIED IN TL(I,J) I.LE.J
C     IVAR    = 0 : NO CALCULATION OF STANDARD ERRORS
C             = 1 : COVARIANCE MATRIX AND STANDARD ERRORS RETURNED
C     IHX     = 0 : CONSTANT DIFFERENTIATION INTERVAL
C             = 1 : DIFFERENTIATION INTERVAL FUNCTION OF PARAMETERS
C     IQOB    = 0 : ALLOW BOUNDARY SOLUTIONS
C             = 1 : QUIT IF A PARAMETER IS SET TO A BOUND (IQOB WILL
C                   BE SET TO INDICATE WHICH PARAMETER IN XALL)
C     IVERB   = 0 : NO OUTPUT TO PROLIX FILE
C             =1-3: RANGES FROM SUMMARY OUTPUT THROUGH DETAILED OUTPUT
C             = 4 : INDICATE FUNCTION VALUE AT EACH FUNCTION EVAL
C             = 5 : MOST VERBOSE OUTPUT TO PROLIX FILE, FOR DEBUGGING
C     INLC    = 0 : NO NON-LINEAR CONSTRAINTS
C             = 1 : INEQUALITY CONSTRAINTS
C             = 2 : EQUALITY CONSTRAINTS
C
C ARRAY DIMENSIONS (INPUT) :
C     MAXPAR : MAXIMUM NUMBER OF PARAMETERS, WHETHER ITERATED OR NOT
C              (MUST MATCH THE VALUE USED IN THE CALLING PROGRAM TO
C              DIMENSION THE ARRAY "BND")
C     MAXITP : MAXIMUM NUMBER OF ITERATED PARAMETERS (MUST MATCH THE
C              VALUE USED IN THE CALLING PROGRAM TO DIMENSION THE
C              ARRAYS "TL" AND "WORK")
C
C INPUT PARAMETERS INVOLVING CONSTRAINTS :
C     RATE    = MINIMUM ACCEPTABLE RATE OF CONVERGENCE FOR REDUCTION
C               OF CONSTRAINT VIOLATION
C     SCL     = SCALING FACTOR MODIFYING SIGMA'S AND THETA'S WHEN
C               ABOVE RATE IS NOT MET.
C     EPSCC   = A BOUND ON ACCEPTABLE CONSTRAINT VIOLATION
C               DETERMINING CONVERGENCE
C     ISCAL   = DISABLES SCALING OF CONSTRAINTS IF ISCAL=1
C     NC      = TOTAL NUMBER OF CONSTRAINTS, WHETHER ACTIVE OR NOT
C     CC(I)   = THE CONSTRAINTS
C
C AUXILIARY PARAMETERS INVOLVING CONSTRAINTS :
C     NK      = NUMBER OF OUTER ITERATIONS
C     ISWC    = A SWITCH CONTROLLING CHOICE OF INITIAL B-MATRIX
C               AT BEGINNING OF EACH INNER ITERATION
C     SC(I)   = SCALING FOR EACH CONSTRAINT
C     TH(I) AND
C     SG(I)   = PARAMETERS OF THE AUGMENTED LAGRANGIAN
C
C     FXC     = AUGMENTED LAGRANGIAN FUNCTION
C     FX      = THE FUNCTION TO MINIMIZE SUBJECT TO CONSTRAINTS
C     FC      = THE TERM INVOLVING THE CONSTRAINTS IN F.
C
C     IAC(I)  = 0 : I-TH CONSTRAINT IS AN EQUALITY
C             = 1 : I-TH CONSTRAINT IS AN INEQUALITY
C
C THE PROGRAM WILL DETERMINE WHETHER EACH INEQUALITY IS PASSIVE OR
C ACTIVE, IAC(I) WILL BE SET TO TWO IF I-TH INEQUALITY IS INACTIVE.
C
C AUXILIARY OUTPUT DATA :
C     NIT     = NUMBER OF ITERATIONS
C     NFE     = NUMBER OF FUNCTION EVALUATIONS
C     IDG     = TERMINATION CODE ( SEE BELOW )
C     PTG     = MAGNITUDE OF NORMALIZED GRADIENT ( - UK-1U )
C     SE(I)   = STANDARD ERRORS OF ITERATED PARAMETERS (ASSUMING
C                  - LOG LIKELIHOOD FUNCTION)
C
C----------------------------------------------------------------------
C
C TERMINATION CODES :
C
C IDG=0  MAXIMUM POSSIBLE ACCURACY REACHED (AS PER TRUPB)
C
C IDG=1  SEARCH DIRECTION NOT DOWNHILL ANYMORE (PTG POSITIVE)
C
C IDG=2  ACCUMULATION OF ROUNDING ERRORS PREVENTS FURTHER PROGRESS
C        (LIMIT ON POSSIBLE ACCURACY REACHED, AS PER TRUPB)
C
C IDG=3  ALL SIGNIFICANT DIGITS LOST THROUGH CANCELLATION IN OPTIMAL
C        CONDITIONING (DIVISION BY ZERO WOULD HAVE OCCURED)
C
C IDG=4  SPECIFIED TOLERANCE ON NORMALIZED GRADIENT WAS MET (AS PER
C        TOL, INDICATES SUCCESSFUL CONVERGENCE)
C
C IDG=5  EXCESSIVE CANCELLATION IN GRADIENT (USUALLY G(I)=ZERO)
C
C IDG=6  MAXIMUM NUMBER OF ITERATIONS REACHED (AS PER MAXIT)
C
C IDG=7  EXCESSIVE CANCELLATION IN NORMALIZED GRADIENT
C        CALCULATION (P(I)=ZERO)
C
C IDG=8  NO ITERATED PARAMETERS (OR ALL PARAMETERS SET TO BOUNDS)
C
C IDG=9  PARAMETER WAS SET TO A BOUND (AS PER IQOB)
C
C IDG=10 UNABLE TO MEET TOLERANCE ON CONSTRAINT VIOLATION (NK=20)
C
C----------------------------------------------------------------------
      SUBROUTINE ALMINI(FX,NALL,XALL,ITP,SE,BND,
     +   IBND,IHESS,IVAR,IHX,IQOB,IVERB,MAXIT,IP,H,TOL,TRUPB,
     +   NIT,NFE,PTG,IDG,TL,MAXPAR,MAXITP,WORK,
     +   INLC,NC,ISCAL,RATE,SCL,EPSCC,FC,FXC,NK,CC,IAC,WRKNLC,iop)
C---
C--- MAIN ROUTINE FOR ALMINI
C---
C--- 1. MAPS XALL INTO X
C--- 2. PARTITIONS WORKSPACE
C--- 3. CALLS OUTER ITERATIVE PROCEDURE
C--- 4. GETS STANDARD ERRORS
C--- 5. MAPS BACK TO XALL
C---
C--- INTERACTION BETWEEN ALMINI AND THE CALLING PROGRAM IS DONE IN
C--- TERMS OF XALL, WHILE ALL OF THE SUBROUTINES HEREAFTER ARE CODED
C--- IN TERMS OF X.  THIS ROUTINE SERVES AS THE INTERFACE, EFFECTING
C--- THE TRANSLATION.  SEE ALSO SUBROUTINE ALFUN, WHERE X IS
C--- TRANSLATED BACK TO XALL PRIOR TO CALLING THE USER FUNCTION.
C---
C---
C---     INPUT ARGUMENTS:
C---     ----------------
C---
C--- ALWAYS REQUIRED:           NALL,XALL,ITP,IBND,IHESS,IVAR,IHX,IQOB,
C---                            IVERB,MAXIT,IP,H,TOL,TRUPB,INLC,NC
C---
C--- REQUIRED IF INLC NON-ZERO: ISCAL,RATE,SCL,EPSCC,IAC
C---
C--- REQUIRED IF IHESS = 2:     TL
C---
C---
C---     OUTPUT ARGUMENTS:
C---     -----------------
C---
C--- ALWAYS RETURNED:           FX,XALL,NIT,NFE,PTG,IDG
C---
C--- RETURNED AS PER INPUT:     SE,IQOB,TL,FC,FXC,NK,CC
C---
C---
C---     ARGUMENTS DEFINING ARRAY DIMENSIONS AND WORKSPACE:
C---     --------------------------------------------------
C---
C--- MAXPAR -- FIRST DIMENSION OF "BND" ARRAY AS SPECIFIED IN THE
C---           CALLING PROGRAM (LIMIT ON NUMBER OF ITERATED PLUS NON-
C---           ITERATED PARAMETERS)
C---
C--- MAXITP -- FIRST DIMENSION OF "TL" AND "WORK" ARRAYS AS SPECIFIED
C---           IN THE CALLING PROGRAM (LIMIT ON NUMBER OF ITERATED
C---           PARAMETERS)
C---
C--- WORK   -- A DOUBLE PRECISION ARRAY OF AT LEAST 16N ELEMENTS, WHERE
C---           N IS THE NUMBER OF ITERATED PARAMETERS
C---
C--- WRKNLC -- A DOUBLE PRECISION ARRAY OF AT LEAST 4NC ELEMENTS, WHERE
C---           NC IS THE NUMBER OF NON-LINEAR CONSTRAINTS
C---
C----------------------------------------------------------------------
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION XALL(NALL),ITP(NALL),SE(NALL),BND(MAXPAR,2)
      DIMENSION TL(MAXITP,*),WORK(MAXITP,16)
      DIMENSION CC(NC),IAC(NC),WRKNLC(NC,4)
C
C WRITE INITIAL INFORMATION TO PROLIX
C
      IF (IVERB .GT. 0) WRITE (IP,1000)
 1000 FORMAT(1X,11('<>='),'<> ALMINI <>',11('=<>'),/)
C
      IF (IVERB .GT. 1) THEN
         WRITE (IP,1010) H,TRUPB
         IF (INLC .EQ. 0) THEN
            WRITE (IP,1030) TOL
         ELSE
            WRITE (IP,1020) RATE,SCL
            WRITE (IP,1030) TOL,EPSCC
         END IF
         WRITE (IP,1040) INLC,IHESS,IVAR,IHX,IQOB,IBND,IVERB,MAXIT
      END IF
C
      IF (IVERB .GT. 2) THEN
         IF (INLC .EQ. 0) WRITE (IP,1110) INLC
         IF (INLC .EQ. 1) WRITE (IP,1111) INLC
         IF (INLC .EQ. 2) WRITE (IP,1112) INLC
         IF (IHESS.EQ. 0) WRITE (IP,1120) IHESS
         IF (IHESS.EQ. 1) WRITE (IP,1121) IHESS
         IF (IHESS.EQ. 2) WRITE (IP,1122) IHESS
         IF (IVAR .EQ. 0) WRITE (IP,1130) IVAR
         IF (IVAR .NE. 0) WRITE (IP,1131) IVAR
         IF (IHX  .EQ. 0) WRITE (IP,1140) IHX
         IF (IHX  .NE. 0) WRITE (IP,1141) IHX
         IF (IQOB .EQ. 0) WRITE (IP,1150) IQOB
         IF (IQOB .NE. 0) WRITE (IP,1151) IQOB
         IF (IBND .EQ. 0) WRITE (IP,1160) IBND
         IF (IBND .NE. 0) WRITE (IP,1161)
      END IF
C
 1010 FORMAT(' DIFFERENTIATION INTERVAL =',1P,G13.5,
     +        '  TRUNCATION UPPER BOUND =',   G13.5)
 1020 FORMAT(' CONSTRAINT RATE VARIABLE =',1P,G13.5,
     +        '  UPDATE FACTOR FOR SG   =',   G13.5)
 1030 FORMAT(' TOLERANCE ON PTG         =',1P,G13.5,
     +        '  TOLERANCE ON CC        =',   G13.5)
 1040 FORMAT(' INLC=',I1,'   IHESS=',I1,'   IVAR=',I1,'   IHX=',I1,
     +   '   IQOB=',I1,'   IBND=',I1,'   IVERB=',I1,'   MAX ITER=',I4,/)
 1110 FORMAT(' INLC= ',I1,'  CHECKING OF NON-LINEAR CONSTRAINTS',
     +   ' DISABLED')
 1111 FORMAT(' INLC= ',I1,'  NON-LINEAR INEQUALITY CONSTRAINTS',
     +   ' ENABLED')
 1112 FORMAT(' INLC= ',I1,'  NON-LINEAR EQUALITY CONSTRAINTS ENABLED')
 1120 FORMAT(' IHESS=',I1,'  INITIAL HESSIAN TAKEN AS IDENTITY MATRIX')
 1121 FORMAT(' IHESS=',I1,'  INITIAL HESSIAN CALCULATED BY FORWARD',
     +   ' DIFFERENCES')
 1122 FORMAT(' IHESS=',I1,'  INITIAL HESSIAN SUPPLIED BY CALLING',
     +   ' PROCEDURE')
 1130 FORMAT(' IVAR= ',I1,'  CALCULATION OF STANDARD ERRORS DISABLED')
 1131 FORMAT(' IVAR= ',I1,'  CALCULATION OF STANDARD ERRORS ENABLED')
 1140 FORMAT(' IHX=  ',I1,'  DIFFERENTIATION INTERVAL CONSTANT')
 1141 FORMAT(' IHX=  ',I1,'  DIFFERENTIATION INTERVAL FUNCTION OF',
     +   ' PARAMETERS')
 1150 FORMAT(' IQOB= ',I1,'  QUIT ON PARAMETER SET TO BOUND DISABLED')
 1151 FORMAT(' IQOB= ',I1,'  QUIT ON PARAMETER SET TO BOUND ENABLED')
 1160 FORMAT(' IBND= ',I1,'  NO BOUNDARY CONDITIONS SPECIFIED')
 1161 FORMAT(/' BOUNDARY CONDITIONS ON ITERATED PARAMETERS:',
     +     //'  XALL         X            LOWER BOUND     UPPER BOUND',
     +      /'  ----        ---           -----------     -----------')
 1170 FORMAT(1X,I3, ')', 1P,G20.12, 2G16.5)
C
C DIMENSIONS MUST BE AT LEAST 1 IN ORDER TO PARTITION WORKSPACE
C
      IF (NALL .LE. 0 .OR. MAXITP .LE. 0 .OR. NC .LE. 0) THEN
         call intpr('ALMINI: NALL, MAXITP OR NC < 1',-1,0,0)
         STOP
      END IF
C
C MAX NUMBER OF ITERATED+NON-ITERATED PARAMETERS MUST BE AT LEAST NALL
C
      IF (MAXPAR .LE. 0 .OR. (MAXPAR .LT. NALL .AND. IBND .NE. 0) )
     +   THEN
         call intpr('ALMINI: ARRAY DIMENSION "MAXPAR" INADEQUATE',
     +             -1,0,0)
         STOP
      END IF
C
C GET INITIAL VALUES FOR ITERATED PARAMETERS;
C ALSO GET UPPER AND LOWER BOUNDS IF SPECIFIED
C
C NOTE: X() BEGINS AT WORK(1,14), BNDL() BEGINS AT WORK(1,15), AND
C    BNDU() BEGINS AT WORK(1,16) IN WORKSPACE ARRAY
C
      N = 0
      TWOH = H+H
      DO 100 I=1,NALL
         IF (ITP(I) .EQ. 1) THEN
            N = N + 1
            X = XALL(I)
            WORK(N,14) = X
            IF (IBND .NE. 0) THEN
               BNDL = BND(I,1) + TWOH
               BNDU = BND(I,2) - TWOH
               IF (IVERB .GT. 2) THEN
                  WRITE (IP,1170) I, XALL(I), BND(I,1), BND(I,2)
               END IF
               IF (X.LT.BNDL .OR. X.GT.BNDU) THEN
                  call intpr
     +            ('ALMINI: INITIAL X OUTSIDE BOUND -- SEE PROLIX',
     +            -1,0,0)
                  STOP
               END IF
               WORK(N,15) = BNDL
               WORK(N,16) = BNDU
            END IF
         END IF
100   CONTINUE
C
C MINIMIZE THE FUNCTION
C
      CALL ALOUTR(FX,NALL,XALL,ITP,N,
     +   IBND,IHESS,IHX,IQOB,IVERB,MAXIT,IP,H,TOL,TRUPB,
     +   NIT,NFE,PTG,IDG,TL,MAXITP,WORK(1,14),WORK(1,15),WORK(1,16),
     +   WORK(1,1),WORK(1,2),WORK(1,3),WORK(1,4),WORK(1,5),
     +   WORK(1,5),WORK(1,5),WORK(1,6),WORK(1,7),WORK(1,8),
     +   WORK(1,9),WORK(1,10),WORK(1,11),WORK(1,12),WORK(1,13),
     +   INLC,NC,ISCAL,RATE,SCL,EPSCC,FC,FXC,NK,
     +   CC,WRKNLC(1,1),WRKNLC(1,2),WRKNLC(1,3),WRKNLC(1,4),IAC,iop)
C
C GET COVARIANCE MATRIX, IF INDICATED, AND RETURN STANDARD ERRORS
C
      IF (IVAR .EQ. 0) GO TO 900
C
      IF (IDG.EQ.8 .OR. IDG.EQ.9) THEN
         ISTAT = -1
      ELSE
         CALL COVAR(FXC,N,WORK(1,14),H,IHX,TL,MAXITP,IP,IVERB,
     +      XALL,NALL,ITP,NFE,INLC,FX,FC,NC,
     +      CC,WRKNLC(1,1),WRKNLC(1,2),WRKNLC(1,3),IAC,ISTAT,iop)
      END IF
C
      N = 0
      DO 200 I=1,NALL
         IF (ISTAT .EQ. 0 .AND. ITP(I) .EQ. 1) THEN
            N = N + 1
            SE(I) = SQRT( TL(N,N) )
         ELSE
            SE(I) = 0.0
         END IF
200   CONTINUE
C
      IF (ISTAT .NE. 0) IVAR = -1
C
C RETURN ITERATED PARAMETER VALUES IN XALL
C
900   N = 0
      DO 910 I=1,NALL
         IF (ITP(I) .EQ. 1) THEN
            N = N + 1
            XALL(I) = WORK(N,14)
         END IF
910   CONTINUE
C
      IF (IVERB .GT. 0) WRITE (IP,1900) NFE
 1900 FORMAT(/' ** TOTAL FUNCTION EVALUATIONS =',I6)
C
      END
      SUBROUTINE ALOUTR(FX,NALL,XALL,ITP,N,
     +   IBND,IHESS,IHX,IQOB,IVERB,MAXIT,IP,H,TOL,TRUPB,
     +   NIT,NFE,PTG,IDG,TL,MAXITP,X,BNDL,BNDU,
     +   G,GSAVE,Y,P,V,STEPXT,UP1,UP2,UP3,UP4,UP5,UP6,UP7,UP8,UP9,
     +   INLC,NC,ISCAL,RATE,SCL,EPSCC,FC,FXC,NK,
     +   CC,TH,SG,SC,THSAVE,IAC,iop)
C---
C--- PERFORMS OUTER LOOP FOR NON-LINEAR CONSTRAINTS IN ALMINI
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION XALL(NALL),ITP(NALL)
      DIMENSION TL(MAXITP,MAXITP),X(*),BNDL(*),BNDU(*)
      DIMENSION G(*),GSAVE(*),Y(*),P(*),V(*),STEPXT(*)
      DIMENSION UP1(*),UP2(*),UP3(*),UP4(*),UP5(*)
      DIMENSION UP6(*),UP7(*),UP8(*),UP9(*)
      DIMENSION CC(NC),TH(NC),SG(NC),SC(NC),THSAVE(NC),IAC(NC)
C
C INITIALIZATION
C
      NK = 0
      NFE = 0
      NFETOT = 0
C
      IF (INLC .NE. 0) THEN
         CALL ALINIT(FX,N,X,IP,IVERB,XALL,NALL,ITP,NFE,
     +      INLC,NC,ISCAL,FC,FXC,ISWC,CKSAVE,CC,TH,SG,SC,IAC,iop)
      END IF
C
C ------ OUTER ITERATIVE CYCLE STARTS HERE ------
C
100   NK = NK + 1
C
      IF (IVERB .GT. 0) THEN
         IF (INLC .NE. 0) WRITE (IP,1010) NK,ISWC,FX,FC
 1010    FORMAT (/' CYCLE',I3,')  ISWC =',I2,
     +      '  INITIAL FX =',1P,G19.11,'   FC =',G19.11/)
C
         IF (IVERB .GT. 2) THEN
            IF (NK .GT. 1) THEN
               WRITE (IP,1110) (CC(I), I=1,NC)
               WRITE (IP,1120) (TH(I), I=1,NC)
               WRITE (IP,1130) (SG(I), I=1,NC)
               WRITE (IP,1140) (IAC(I), I=1,NC)
C
 1110          FORMAT (' ** CC =',1P,6G12.4)
 1120          FORMAT (' ** TH =',1P,6G12.4)
 1130          FORMAT (' ** SG =',1P,6G12.4)
 1140          FORMAT (' ** CONSTRAINT STATUS:',25I2)
            END IF
         END IF
      END IF
C
C -- GEMINI, WITHOUT NON-LINEAR CONSTRAINTS, BEGINS HERE --
C
      IF (INLC .EQ. 0) THEN
         IF (IVERB .GE. 5)
     +      WRITE (IP,*) 'ALOUTR) INITIAL FUNCTION CALL:'
         CALL ALFUN(FXC,X,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +      INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
      END IF
C
200   CONTINUE
C
C EXIT IF NO ITERATED PARAMETERS
C
      IF (N.LE.0 .OR. N.GT.NALL) GO TO 808
C
C SET INITIAL HESSIAN
C
C READ OR CALCULATE INITIAL HESSIAN AS SPECIFIED BY IHESS, AND STORE
C IN CHOLESKY FACTORIZED FORM LDLT.  TL IS LOWER TRIANGULAR UNIT
C MATRIX, WHILE DIAGONAL OF TL HOLDS RECIPROCALS OF DIAGONAL ELEMENTS.
C
      T = 0.1
      IF (ISWC .NE. 1) THEN
         CALL INIB(FXC,N,X,IHESS,H,IHX,TL,MAXITP,IP,IVERB,
     +      XALL,NALL,ITP,NFE,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
         IF (IHESS .GT. 0) T = 1.0
      END IF
C
C CALL GEM TO PERFORM INNER-LOOP
C
      CALL GEM(FXC,N,IBND,IHX,IVERB,MAXIT,IP,H,TOL,TRUPB,T,
     +   NIT,NFE,PTG,IDG,ICLOSE,TL,MAXITP,X,BNDL,BNDU,
     +   G,GSAVE,Y,P,V,STEPXT,UP1,UP2,UP3,UP4,UP5,UP6,UP7,UP8,UP9,
     +   XALL,NALL,ITP,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
C IF PARAMETER SET TO BOUND, HANDLE AS SPECIFIED BY IQOB
C
      IF (IDG .EQ. 9) THEN
         CALL SETBND(ICLOSE,FXC,NALL,XALL,ITP,N,X,BNDL,BNDU,H,
     +      IP,IVERB,NFE,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,ISTAT,iop)
         IF (IQOB .EQ. 0) THEN
            IHESS = 0
            GO TO 200
         ELSE
            IQOB = ISTAT
         END IF
      END IF
C
C WRITE FINAL OUTPUT FOR INNER PROCEDURE
C
      IF (IVERB .GE. 5)
     +   WRITE (IP,*) 'ALOUTR) FINAL F IN INNER PROCEDURE:'
      CALL ALFUN(FXC,X,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +   INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
      IF (IVERB .GT. 0) WRITE (IP,1200) IDG,NFE,FXC
 1200 FORMAT(1X,79('='),
     +  /' ** IDG =',I2,'  NFE =',I6,'  F =',1P,G19.11)
C
C -- GEMINI, WITHOUT NON-LINEAR CONSTRAINTS, ENDS HERE --
C
      IF (INLC .EQ. 0) GO TO 999
C
C ---------------------------------------------------------------------
C
C OUTER PROCEDURE EXIT CONDITIONS:
C
C 1) EXIT ON FAILED CONVERGENCE OF INNER PROCEDURE (WE ASSUME HERE
C    THAT THE CHOICE FOR THE VALUE OF "TOL" WAS ADEQUATELY LARGE.
C    THIS CHECK MAY HAVE TO BE MODIFIED FOR SOME APPLICATIONS.)
C
      IF (IDG .NE. 4) GO TO 900
C
C 2) EXIT ON CONVERGENCE OF OUTER PROCEDURE, WHERE THE MAXIMUM
C    CONSTRAINT VIOLATION IS LESS THAN EPSILON SPECIFIED BY USER
C    (CONSTRAINTS EXPRESSED IN ORIGINAL SCALE)
C
      CKR = 0.0
      DO 700 I=1,NC
         IF (CC(I) .LT. 0.0 .OR. IAC(I) .EQ. 0) THEN
            CKR = MAX( CKR, ABS(CC(I)) )
         END IF
700   CONTINUE
C
      IF (CKR .LE. EPSCC) GO TO 900
C
C 3) EXIT IF MAXIMUM NUMBER OF OUTER ITERATIONS EXCEEDED
C
      MAXOUT = 10*NC
      IF (NK .GE. MAXOUT) GO TO 810
C
C UPDATE SIGMA AND THETA
C
      CALL ALUPD(NC,RATE,SCL,ISWC,CKSAVE,CC,TH,SG,SC,THSAVE,IAC)
C
C GET THE UPDATED VALUE OF THE AUGMENTED LAGRANGIAN
C
      CALL ALFC(FC,NC,CC,TH,SG,SC,IAC)
C
      FXC = FX + 0.5*FC
C
      NFETOT = NFETOT + NFE
      NFE = 0
C
      GO TO 100
C
C ---------------------------------------------------------------------
C                            EXIT SECTION
C ---------------------------------------------------------------------
C
C ADD TWO MORE TERMINATION CODES:
C
C NO ITERATED PARAMETERS (OR ALL PARAMETERS SET TO BOUNDS)
C
808   IDG = 8
      GO TO 900
C
C MAXIMUM NO. OF ATTEMPTS TO REDUCE CONSTRAINT VIOLATION WAS EXCEEDED
C
810   IDG = 10
      GO TO 900
C
C WRITE FINAL OUTPUT FOR OUTER PROCEDURE
C
900   NFE = NFE + NFETOT
C
      IF (IVERB .GT. 0) THEN
         WRITE (IP,1910) NK,FX,FC
         WRITE (IP,1920) (CC(I), I=1,NC)
 1910    FORMAT (' ** NK =',I3,'  FX =',1P,G19.11,'  FC =',G19.11)
 1920    FORMAT (' ** CC =',1P,6G12.4)
      END IF
C
999   RETURN
      END
      SUBROUTINE ALINIT(FX,N,X,IP,IVERB,XALL,NALL,ITP,NFE,
     +   INLC,NC,ISCAL,FC,FXC,ISWC,CKSAVE,CC,TH,SG,SC,IAC,iop)
C---
C--- INITIALIZATION ASSOCIATED WITH NON-LINEAR CONSTRAINTS
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION X(*)
      DIMENSION XALL(NALL),ITP(NALL)
      DIMENSION CC(NC),TH(NC),SG(NC),SC(NC),IAC(NC)
C
      IF (INLC.LT.0 .OR. INLC.GT.2) THEN
         call intpr('ALMINI (ALINIT): INVALID INLC',-1,0,0)
         STOP
      END IF
C
C INITIALIZE SIGMA, THETA, AND CONSTRAINT SCALE FACTORS
C
      DO 10 I=1,NC
         SG(I) = 2.0
         TH(I) = 0.0
         SC(I) = 1.0
10    CONTINUE
C
C GET INITIAL STATUS OF CONSTRAINTS
C
      IF (IVERB .GE. 5)
     +   WRITE (IP,*) 'ALINIT) FOR INITIAL STATUS OF CONTRAINTS:'
      CALL ALFUN(FXC,X,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +   INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
      DO 100 I=1,NC
         IF (CC(I).GE.0.0 .AND. IAC(I).EQ.1) IAC(I) = 2
100   CONTINUE
C
C IF USER SPECIFIED INLC=2, RE-CLASSIFY ACTIVE INEQUALITY CONTRAINTS
C AS EQUALITIES, AND ALL REMAINING AS PASSIVE
C
      IF (INLC .EQ. 2) THEN
         DO 200 I=1,NC
            IF (CC(I) .LT. 0.0) THEN
               IAC(I) = 0
            ELSE
               IAC(I) = 2
            END IF
200      CONTINUE
      END IF
C
C GET CONSTRAINT SCALE FACTORS AND INITIAL CKSAVE
C (DISABLE SCALING IF ISCAL=1)
C
      IF (ISCAL .EQ. 1 .OR. FX .EQ. 0.0) THEN
         CKSAVE = 10000.
C
      ELSE
         DO 300 I=1,NC
            IF (CC(I) .NE. 0.0) SC(I) = SQRT(ABS(FX)) / ABS(CC(I))
300      CONTINUE
         CKSAVE = SQRT(ABS(FX)) * 1000.0
      END IF
C
C INITIALIZE FLAG TO GET INITIAL HESSIAN
C
      ISWC = 0
C
C GET THE INITIAL VALUE OF THE AUGMENTED LAGRANGIAN
C
      CALL ALFC(FC,NC,CC,TH,SG,SC,IAC)
C
      FXC = FX + 0.5*FC
C
C WRITE TO PROLIX
C
      IF (IVERB .GT. 2) THEN
         WRITE (IP,*)
         WRITE (IP,*)   'UNSCALED CONSTRAINTS:'
         WRITE (IP,1610) (CC(I),I=1,NC)
         WRITE (IP,*)   'SCALE FACTORS:'
         WRITE (IP,1610) (SC(I), I=1,NC)
         WRITE (IP,*)   'SCALED CONSTRAINTS:'
         WRITE (IP,1610) (CC(I)*SC(I), I=1,NC)
         WRITE (IP,*)   'THETA:'
         WRITE (IP,1610) (TH(I)*SC(I), I=1,NC)
         WRITE (IP,*)   'SIGMA:'
         WRITE (IP,1610) (SG(I)*SC(I), I=1,NC)
         WRITE (IP,*)   'CONSTRAINT STATUS:',
     +                  '(0=EQUALITY,1=INEQUALITY,2=INACTIVE)'
         WRITE (IP,1620) (IAC(I),I=1,NC)
      END IF
C
 1610 FORMAT (1X,1P,5G15.5)
 1620 FORMAT((1X,5(I8,7X)))
C
      END
      SUBROUTINE ALUPD(NC,RATE,SCL,ISWC,CKSAVE,CC,TH,SG,SC,THSAVE,IAC)
C---
C--- LOGIC TO MODIFY SIGMA AND THETA FOR NON-LINEAR CONSTRAINTS
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION CC(NC),TH(NC),SG(NC),SC(NC),THSAVE(NC),IAC(NC)
C
C UPDATE STATUS OF INEQUALITY CONSTRAINTS
C
      DO 100 I=1,NC
         IF (IAC(I).EQ.2 .AND. CC(I).LT.0.0) IAC(I) = 1
         IF (IAC(I).EQ.1 .AND. CC(I).GE.0.0) IAC(I) = 2
100   CONTINUE
C
C FIND CONSTRAINT VIOLATION (CONSTRAINTS SCALED)
C
      CK = 0.0
      DO 200 I=1,NC
         IF (IAC(I) .LE. 1) CK  = MAX( CK,  ABS(CC(I)*SC(I)) )
200   CONTINUE
C
C CHOOSE METHOD OF UPDATE
C
      IF (CK.LE.CKSAVE*RATE .OR. (CK.LT.CKSAVE .AND. ISWC.EQ.0)) THEN
C
C UPDATE A) USED WHEN ALL ACTIVE CONSTRAINTS SATISFY SPECIFIED
C RATE OF CONSTRAINT REDUCTION  (IN THIS CASE, B-MATRIX OBTAINED
C IN PREVIOUS CYCLE CAN BE CARRIED THROUGH TO NEXT CYCLE)
C
         DO 300 I=1,NC
            THSAVE(I) = TH(I)
            CS = CC(I)*SC(I)
            IF (IAC(I) .EQ. 1 .AND. CS .GT. TH(I)) THEN
               TH(I) = 0.0
            ELSE
               TH(I) = TH(I) - CS
            END IF
300      CONTINUE
         CKSAVE = CK
         ISWC = 1
C
      ELSE
C
C RETAIN CKSAVE IF CONSTRAINT VIOLATION WAS NOT REDUCED;
C IF CHOICE OF UPDATE A PROVED UNSUCCESSFUL, TRY AGAIN USING UPDATE B
C
         IF (CK .GE. CKSAVE) THEN
            CK = CKSAVE
C
            IF (ISWC .EQ. 1) THEN
               DO 410 I=1,NC
                  TH(I) = THSAVE(I)
410            CONTINUE
            END IF
         END IF
C
C UPDATE B) ONE OR MORE CONSTRAINTS FAILED TO SATISFY SPECIFIED
C RATE OF CONSTRAINT REDUCTION.  INCREASE SIGMA BY THE CONSTANT
C FACTOR SCL AND ADJUST THETA SUCH THAT SG*TH REMAINS UNCHANGED.
C (B-MATRIX WILL BE RE-INITIALIZED IN NEXT CYCLE)
C
         DO 400 I=1,NC
            IF (IAC(I) .LE. 1) THEN
               IF (ABS(CC(I)*SC(I)) .GE. CKSAVE*RATE) THEN
                  SG(I) = SG(I) * SCL
                  TH(I) = TH(I) / SCL
               END IF
            END IF
400      CONTINUE
         CKSAVE = CK
         ISWC = 0
      END IF
C
      END
      SUBROUTINE ALFC(FC,NC,CC,TH,SG,SC,IAC)
C---
C--- GETS THE TERM FC, INVOLVING NON-LINEAR CONSTRAINTS,
C--- USED IN THE CALCULATION OF THE AUGMENTED LAGRANGIAN
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION CC(NC),TH(NC),SG(NC),SC(NC),IAC(NC)
C
      FC = 0.0
C
C LOOP OVER ALL ACTIVE CONSTRAINTS
C
      DO 100 I=1,NC
         CCMTH = CC(I)*SC(I) - TH(I)
         IF (IAC(I).EQ.0 .OR. CCMTH.LT.0.0) THEN
            FC = FC + SG(I)*CCMTH*CCMTH
         END IF
100   CONTINUE
C
      END
      SUBROUTINE GEM(FXC,N,IBND,IHX,IVERB,MAXIT,IP,H,TOL,TRUPB,T,
     +   NIT,NFE,PTG,IDG,ICLOSE,TL,MAXITP,X,BNDL,BNDU,
     +   G,GSAVE,Y,P,V,STEPXT,UP1,UP2,UP3,UP4,UP5,UP6,UP7,UP8,UP9,
     +   XALL,NALL,ITP,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C---
C--- PERFORMS INNER-ITERATION LOOP:
C---
C---  1. GETS GRADIENT
C---  2. UPDATES LDLT
C---  3. INCREMENTS ITERATION COUNT
C---  4. GETS NEW SEARCH DIRECTION
C---  5. CHECKS BOUNDS ON PARAMETERS
C---  6. PERFORMS LINEAR SEARCH
C---
C--- IDG IS THE TERMINATION CODE RETURNED (SEE BELOW)
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION TL(MAXITP,MAXITP),X(*),BNDL(*),BNDU(*)
      DIMENSION G(*),GSAVE(*),Y(*),P(*),V(*),STEPXT(*)
      DIMENSION UP1(*),UP2(*),UP3(*),UP4(*),UP5(*)
      DIMENSION UP6(*),UP7(*),UP8(*),UP9(*)
      DIMENSION XALL(NALL),ITP(NALL)
      DIMENSION CC(NC),TH(NC),SG(NC),SC(NC),IAC(NC)
C
C DEFINE DOUBLE-PRECISON CONSTANTS:
C
      DATA ONE,TENTH /1.0, 0.1D0/
C
C INITIALIZE ITERATION COUNT, AND SET A FLAG TO
C START WITH CHOICE OF FORWARD DIFFERENTIATION
C
      NIT = 0
      IDIF = 1
C
C INITIALIZE CHANGE IN FUNCTION VALUE
C
      FSMF = 0.0
C
C GET INITIAL GRADIENT
C
10    CALL FDERIV(FXC,X,N,IDIF,H,IHX,G,
     +   XALL,NALL,ITP,NFE,IP,IVERB,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
C INCREMENT ITERATION COUNT AND GET SEARCH DIRECTION BY FORWARD AND
C BACKWARD SUBSTITUTION
C
100   NIT = NIT + 1
C
C A TERMINATION CRITERION MAY BE SET ON THE NORM SQRT(GTG)
C
      GTG = 0.0
      DO 110 I=1,N
         GTG = GTG + G(I)*G(I)
110   CONTINUE
C     GTG = SQRT(GTG)
C     IF (GTG .LE. TOLG) GO TO 805
C
C FORWARD SUBSTITUTION TO SOLVE L*V=-G FOR V
C
      DO 220 I=1,N
         V(I) = -G(I)
         DO 210 J=1,I-1
            V(I) = V(I) - TL(I,J)*V(J)
210      CONTINUE
220   CONTINUE
C
C BACKWARD SUBSTITUTION TO SOLVE LT*P=DINV*V
C
      DO 240 J=N,1,-1
         P(J) = V(J) / TL(J,J)
         DO 230 I=J+1,N
            P(J) = P(J) - TL(I,J)*P(I)
230      CONTINUE
240   CONTINUE
C
C WRITE TO PROLIX
C
      IF (IVERB .GT. 1) THEN
         WRITE (IP,1300) NIT,T,NFE,FXC
         WRITE (IP,1301) (X(I), I=1,N)
         IF (IVERB .GT. 2) WRITE (IP,1302) (G(I), I=1,N)
         IF (IVERB .GT. 2) WRITE (IP,1303) (P(I), I=1,N)
      END IF
C
 1300 FORMAT(1X,79('-'),
     +      /' ITERATION',I4,'      T =',1P,G14.5,
     +         '    NFE = ',I5,'    F =',G20.12)
 1301 FORMAT(' X:',1P,5G14.6)
 1302 FORMAT(' G:',1P,5G14.6)
 1303 FORMAT(' P:',1P,5G14.6)
C
C CHECK IF DIRECTION OF SEARCH IS DOWNHILL
C
C TERMINATE IF EXCESSIVE CANCELLATION LEADS TO SOME P(I) TO BE ZERO
C THIS TEST APPLIED ONLY IF BOUNDARY CONDITIONS SPECIFIED
C
      IF (GTG .LE. 0.0) GO TO 805
C
      PTGTMP = 0.0
      DO 400 I=1,N
         IF (IBND.GT.0 .AND. P(I).EQ.0.0) GO TO 807
         PTGTMP = PTGTMP + P(I)*G(I)
400   CONTINUE
      IF (PTGTMP .GE. 0.0) GO TO 801
      PTG = PTGTMP
C
C IF PTG HAS MET SOME TOLERANCE TOL, THE PROCEDURE IS ENDED
C
      IF (ABS(PTG) .LE. TOL) THEN
         IF (IVERB .GT. 1) WRITE (IP,1400) FSMF,PTG
 1400    FORMAT(1X,'FSMF=',1P,G10.3,' PTG=',G12.5)
         CALL FLUSH(IP)
         GO TO 804
      END IF
C
C CHECK FOR MAXIMUM NUMBER OF ITERATIONS
C
      IF (NIT .GE. MAXIT) GO TO 806
C
C GET READY TO CARRY OUT LINEAR SEARCH. GET MINIMUM ACCEPTABLE
C STEP, TMIN, AND INITIAL ESTIMATE OF STEP SIZE T
C
      IF (NIT .GT. 1) THEN
         XPMIN = 1.0D30
         DO 500 I=1,N
            HXDH = ONE
            IF (IHX.EQ.1) HXDH = MAX( ABS(X(I)), TENTH)
            XPMIN = MIN( ABS(HXDH/P(I)), XPMIN )
500      CONTINUE
         TMIN = 0.5 * XPMIN * H / TRUPB
C
         T = -2.0 * FSMF / PTG
         IF (T.LE.0.0 .OR. T.GT.1.0 .OR. IDIF.EQ.2) T = 1.0
C
      ELSE
         TMIN = 0.0
      END IF
C
C CHECK BOUNDS IF ANY
C
      TBND = 0.0
      IF (IBND .GT. 0) THEN
         CALL CHKBND(N,X,BNDL,BNDU,P,T,TBND,H,ICLOSE)
         IF (ICLOSE .NE. 0) GO TO 809
      END IF
C
      IF (IVERB .GT. 1) WRITE (IP,1600) FSMF,PTG,T,TMIN,TBND
 1600 FORMAT(1X,'FSMF=',1P,G10.3,' PTG=',G12.5,
     +         ' INIT T=',G9.2,' TMIN=',G9.2,' TBND=',G9.2)
       CALL FLUSH(IP)
C
C ENTER LINEAR SEARCH
C
      FSAVE = FXC
C
      CALL STEP(FXC,N,X,T,TMIN,TBND,IBND,IP,IVERB,STEPXT,P,
     +   XALL,NALL,ITP,NFE,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,ISTAT,iop)
C
      FSMF = FSAVE - FXC
C
      IF (ISTAT .EQ. -1) GO TO 802
      IF (ISTAT .EQ. -2) GO TO 800
C
C SAVE OLD GRADIENT IN GSAVE, THEN
C GET CURRENT GRADIENT BY EITHER FORWARD OR CENTRAL DIFFERENCES
C
      DO 600 I=1,N
         GSAVE(I) = G(I)
600   CONTINUE
C
      CALL FDERIV(FXC,X,N,IDIF,H,IHX,G,
     +   XALL,NALL,ITP,NFE,IP,IVERB,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
C PREPARATION TO UPDATE LDLT
C
      YTP = 0.0
      DO 610 I=1,N
         Y(I) = G(I) - GSAVE(I)
         YTP = YTP + Y(I)*P(I)
610   CONTINUE
C
C CALL UPDATING ROUTINE AT THIS ITERATION ONLY IF YTP POSITIVE
C
      IF (YTP .GT. 0.0) THEN
         CALL UPDATE(N,TL,MAXITP,G,GSAVE,Y,
     +      UP1,UP2,UP3,UP4,UP5,UP6,UP7,UP8,UP9,T,ISTAT)
         IF (ISTAT .NE. 0) GO TO 803
      END IF
C
      GO TO 100
C
C----------------------------------------------------------------------
C
C RESTART WITH IDIF=2 (CENTRAL DIFFERENCES)
C
700   IDIF = 2
      IF (IVERB .GT. 1) WRITE (IP,1700) IDG
 1700 FORMAT( 30X,'----------------------------',
     +       /30X,'SWITCH TO CENTRAL DIFFERENCE',
     +       /30X,'----------------------------',
     +       /1X,'SWITCH OCCURED BECAUSE: IDG=',I1)
      GO TO 10
C
C ---------------------------------------------------------------------
C                            EXIT SECTION
C ---------------------------------------------------------------------
C
C MAXIMUM POSSIBLE ACCURACY REACHED
800   IDG = 0
      IF (IDIF .EQ. 1) GO TO 700
      GO TO 999
C
C SEARCH DIRECTION NOT DOWNHILL ANYMORE (PTG POSITIVE)
801   IDG = 1
      IF (IDIF .EQ. 1) GO TO 700
      GO TO 999
C
C ACCUMULATION OF ROUNDING ERRORS PREVENTS FURTHER PROGRESS
C (LIMIT ON POSSIBLE ACCURACY REACHED, CAN'T MAKE T ANY SMALLER)
802   IDG = 2
      IF (IDIF .EQ. 1) GO TO 700
      GO TO 999
C
C ALL SIGNIFICANT DIGITS LOST THROUGH CANCELLATION IN OPTIMAL
C CONDITIONING (DIVISION BY ZERO WOULD HAVE OCCURED)
803   IDG = 3
      GO TO 999
C
C SPECIFIED TOLERANCE ON NORMALIZED GRADIENT WAS MET (PTG < TOL)
804   IDG = 4
      GO TO 999
C
C ("SPECIFIED TOLERANCE ON GRADIENT WAS MET" CRITERION NOT USED)
C GRADIENT NUMERICALLY ZERO
805   IDG = 5
      GO TO 999
C
C MAXIMUM NUMBER OF ITERATIONS REACHED
806   IDG = 6
      GO TO 999
C
C EXCESSIVE CANCELLATION IN GRADIENT CALCULATION
807   IDG = 7
      GO TO 999
C
C NO ITERATED PARAMETERS
808   IDG = 8
      GO TO 999
C
C PARAMETER WAS SET TO A BOUND
809   IDG = 9
      GO TO 999
C
999   RETURN
      END
      SUBROUTINE FDERIV(FXC,X,N,IDIF,H,IHX,G,
     +   XALL,NALL,ITP,NFE,IP,IVERB,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C---
C--- RETURNS GRADIENT VECTOR G OF FUNCTION WITH RESPECT TO X
C---
C--- INPUT ARGUMENTS:
C---   FXC   -- FUNCTION VALUE OF GIVEN X VECTOR
C---   IDIF  -- CONTROLS METHOD OF DIFFERENTIATION AS FOLLOWS:
C---            1 = FORWARD NUMERICAL DIFFERENTIATION
C---            2 = CENTRAL NUMERICAL DIFFERENTIATION
C---   H     -- DIFFERENTIATION INTERVAL
C---   IHX   -- NON-ZERO MEANS SCALE H TO PARAMETER VALUE
C---
C--- NOTE:  IF ANALYTIC DERIVITIES ARE KNOWN, THE USER MAY REPLACE
C---        THIS SUBROUTINE WITH HIS OWN.
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION X(*),G(*)
      DIMENSION XALL(NALL),ITP(NALL)
      DIMENSION CC(NC),TH(NC),SG(NC),SC(NC),IAC(NC)
C
      DATA TENTH /0.1D0/
C
      IF (IDIF .EQ. 1) THEN
C     FORWARD DIFFERENCE
         DO 110 I=1,N
            HX = H
            IF (IHX.EQ.1) HX = H*MAX( ABS(X(I)), TENTH)
            XSAVE = X(I)
            X(I) = X(I) + HX
            IF (IVERB .GE. 5)
     +         WRITE (IP,*) 'FDERIV) FORWARD DIFFERENCE: I=', I
            CALL ALFUN(FXPH,X,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +         INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
            G(I) = (FXPH-FXC) / HX
            X(I) = XSAVE
110      CONTINUE
      ELSE
C
C     CENTRAL DIFFERENCE
         DO 120 I=1,N
            HX = H
            IF (IHX.EQ.1) HX = H*MAX( ABS(X(I)), TENTH)
            XSAVE = X(I)
            X(I) = X(I) + HX
            IF (IVERB .GE. 5)
     +         WRITE (IP,*) 'FDERIV) CENTRAL DIFFERENCE 1: I=', I
            CALL ALFUN(FXPH,X,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +         INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
            X(I) = XSAVE - HX
            IF (IVERB .GE. 5)
     +         WRITE (IP,*) 'FDERIV) CENTRAL DIFFERENCE 2: I=', I
            CALL ALFUN(FXMH,X,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +         INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
            G(I) = (FXPH - FXMH) / (HX+HX)
            X(I) = XSAVE
120      CONTINUE
      END IF
C
      END
      SUBROUTINE UPDATE(N,TL,MAXITP,G,GSAVE,Y,
     +   WTIL,ZTIL,W,Z,WTJP1,ZTJP1,S,DP,D,T,ISTAT)
C---
C--- GOLDFARB METHOD FOR VARIABLE METRIC B-MATRIX UPDATING
C--- UPDATE OF FACTORIZED B-MATRIX BY RANK TWO MODIFICATION
C--- IN REAL PRODUCT FORM WITH FORMULA (3.13) OR (3.16) , DEPENDING
C--- WHETHER NONE OR AT LEAST ONE LAMBA(J)**2 IS GREATER THAN 4.
C--- OPTIMAL CONDITIONING OF DAVIDON ALSO INCORPORATED
C---
C--- **** NOTE **** GSTAR IS IN ARRAY G WHILE G IS IN ARRAY GSAVE,
C--- GSTAR BEING GRADIENT AT IT (NIT+1) AND G GRADIENT AT IT (NIT)
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION TL(MAXITP,MAXITP),D(*),G(*),GSAVE(*),Y(*)
      DIMENSION WTIL(*),ZTIL(*),W(*),Z(*),WTJP1(*),ZTJP1(*),S(*),DP(*)
C
      DOUBLE PRECISION NU,MUJ,LAMBJ,LAMBJ2
      data alpha /0./
C
      IF (N .LE. 1) GO TO 700
C
C GET DIAGONALS OF TL IN D
C
      DO 10 I=1,N
         D(I) = TL(I,I)
10    CONTINUE
C
C SOLVES (3.10) AND (3.11) FOR WTIL AND ZTIL
C
      DO 120 K = 1,N
         WTIL(K) = -GSAVE(K)
         ZTIL(K) = Y(K)
         DO 110 I=1,K-1
C
C SAVE TL LOWER TRIANG IN UPPER TRIANG OF ARRAY TL, IN CASE A SWITCH TO
C (3.16) SHOULD BE REQUIRED
C
            TL(I,K) = TL(K,I)
            WTIL(K) = WTIL(K) - TL(K,I)*WTIL(I)
            ZTIL(K) = ZTIL(K) - TL(K,I)*ZTIL(I)
110      CONTINUE
120   CONTINUE
C
C GET SCALARS B,C AND D, DENOTED SB,SC,SD HERE
C
      SB = 0.0
      SC = 0.0
      SD = 0.0
      DO 130 I=1,N
         SB = SB + T*ZTIL(I)*WTIL(I)/D(I)
         SC = SC + T*T*WTIL(I)*WTIL(I)/D(I)
         SD = SD + ZTIL(I)*ZTIL(I)/D(I)
130   CONTINUE
C
C OPTIMAL CONDITIONING OF UPDATE, ACCORDING TO THEOREM 3 OF
C DAVIDON (1975)
C
C OPTIMAL CONDITIONING MAY BE DISABLED BY BRANCHING TO STMT 300 HERE
C IN SUCH CASE, UPDATE IS DFP-UPDATE FOR B (ALPHA=0)
C
CCC      GO TO 300
C
C PREVENTING UNDERFLOW
C IF ANY OF SB,SC,SD IS SMALLER THAN 1.E-10, USE SR1-UPDATE FOR B
C
      FBCD = MIN(SB,SC,SD)
      IF (FBCD .LE. 1.D-10) GO TO 200
C
      FBCD = 2.0 * SC * (SD/SB) / (SC+SD)
200   IF (FBCD .LT. 1.0) THEN
C
C SR1 UPDATE
C
c200     ALPHA = -1.0
         ALPHA = -1.0
         IF (SC .EQ. SB .OR.
     +       SB .EQ. SD .OR.
     +       SD-2.*SB+SC .EQ. 0.0) GO TO 800
         SA = 1. / ( SQRT(ABS(SC-SB)) * SQRT(ABS(SB-SD)) )
         THET1 =    - ((SD-SB)*SA+1.) / (SD-2.*SB+SC)
         THET2 = SA + (SA*(SB-SC)+1.) / (SD-2.*SB+SC)
         GO TO 310
      END IF
C
C RANK TWO UPDATE
C
C GET ALPHA
C
      AA = SB/SC - 2.*(SD/SB) + SD/SC
      BB = SB/SC - 1.
      CC = 1. - SB/SD
      DEL2 = BB*BB - AA*CC
C
C IF DISCRIMINANT NEGATIVE OR EQUAL ZERO, TAKE ALPHA EQUAL TO ZERO
C
      IF (DEL2 .LE. 1.D-08) GO TO 300
C
      DEL = SQRT(DEL2)
      ALPH1 = (-BB+DEL) / AA
      ALPH2 = (-BB-DEL) / AA
C
C FOR NOW, ALWAYS CHOOSE ROOT OF SMALLEST MODULUS
C
      IF (ABS(ALPH1) .LT. ABS(ALPH2)) THEN
         ALPHA = ALPH1
      ELSE
         ALPHA = ALPH2
      END IF
C
C IF ALPHA VERY SMALL, ALPHA TAKEN EQUAL TO ZERO ( DFP-UPDATE FOR B)
C
      IF (ABS(ALPHA) .LE. 1.D-05) GO TO 300
C
C GET SA
C
      SA = (ALPHA+1.)*(ALPHA+1.) + SC/SB
     +   - ALPHA*ALPHA*(SC/SB)*(SD/SB)
     +   - 1.0 + ALPHA*ALPHA*(SD/SB)
      IF (SA .LE. 0.0) THEN
         SA = 0.0
      ELSE
         SA = 1. / (SQRT(SA)*SB)
      END IF
C
C GET THET1 AND THET2 FOR NON TRIVIAL ALPHA
C
      RDIV = 1. / (ALPHA*ALPHA*SD + 2.*ALPHA*SB + SC)
      THET1 = -(SA*(ALPHA*(ALPHA*SD+SB)) + 1.) * RDIV
      THET2 = SA + (ALPHA*SA*(SC+ALPHA*SB) - ALPHA) * RDIV
      GO TO 310
C
C TAKE ALPHA EQUAL TO ZERO
C
300   ALPHA = 0.0
C
C THIS IS DFP-UPDATE FOR B
C
      SA = 1. / ( SQRT(SB)*SQRT(SC) )
      THET1 = -1.0 / SC
      THET2 = SA
C
C GET W AND Z AS GIVEN JUST AFTER (3.11)
C
310   DO 320 I=1,N
         W(I) = T*WTIL(I) + ALPHA*ZTIL(I)
         Z(I) = T*THET1*WTIL(I) + THET2*ZTIL(I)
320   CONTINUE
C
C READY TO APPLY GOLDFARB RECURRENCE 3 TO SOLVE CONCURRENTLY
C (3.13) , (3.12) BEING ALSO SOLVED SIMULTANEOUSLY
C
C GET THE S(J) FIRST (P.802,TOP) AND INITIALIZE
C
      NP1 = N+1
      NM1 = N-1
      S(N) = 0.0
      DO 330 K=1,NM1
         J = NP1-K
         JM1 = J-1
         S(JM1) = S(J) + W(J)*W(J)/D(J)
330   CONTINUE
      NU = 1.0
      ETA = 0.0
C
C INITIALIZE
C
      ISWUP = 1
400   CONTINUE
      IF (ISWUP .EQ. 1) THEN
         DO 410 I=1,N
            WTJP1(I) = -GSAVE(I)
            ZTJP1(I) = Y(I)
410      CONTINUE
      ELSE
         DO 420 I=1,N
            WTIL(I) = -T*G(I) + ALPHA*Y(I)
            ZTIL(I) = -T*THET1*G(I) + THET2*Y(I)
420      CONTINUE
      END IF
C
      DO 590 J=1,NM1
         IF (ISWUP .EQ. 1) THEN
            DO 510 K=J+1,N
               WTJP1(K) = WTJP1(K) - WTIL(J)*TL(K,J)
               ZTJP1(K) = ZTJP1(K) - ZTIL(J)*TL(K,J)
510         CONTINUE
         ELSE
            DO 520 K=J+1,N
               WTJP1(K) = WTIL(K) - W(J)*TL(K,J)
               ZTJP1(K) = ZTIL(K) - Z(J)*TL(K,J)
520         CONTINUE
         END IF
C
C     RECURRENCE 3 TO GET AJ,BJ,ETC...
C
         AJ = NU*Z(J) - ETA*W(J)
         THJ = 1.0 + AJ*W(J)/D(J)
         LAMBJ2 = THJ*THJ + AJ*AJ*S(J)/D(J)
         IF (ISWUP.EQ.1 .AND. LAMBJ2.GT.10.) THEN
C
C     SWITCH TO (3.16) UPDATE
C
            ISWUP = 2
            DO 540 K=2,N
               KM1 = K-1
               DO 530 I=1,KM1
                  TL(K,I) = TL(I,K)
530            CONTINUE
540         CONTINUE
            GO TO 400
         END IF
C
C UPDATES DPLUS(J)
C
         DP(J) = D(J)*LAMBJ2
         LAMBJ = SQRT(LAMBJ2)
C
C TAKES SIGN OF LAMBDA OPPOSITE TO THAT OF THETA
C
         IF (THJ .GT. 0.0) LAMBJ = -LAMBJ
         MUJ = THJ - LAMBJ
         BJ = THJ*W(J) + AJ*S(J)
         GAMLJ = BJ*NU/(LAMBJ2*D(J))
         BETLJ = (AJ-BJ*ETA) / (LAMBJ2*D(J))
C
C NOTE THAT GAMLJ AND BETLJ STAND FOR GAMMA TILDA AND BETA TILDA
C RESPECTIVELY IN (3.14)
C
         NU = -NU/LAMBJ
         ETA = -(ETA+AJ*AJ/(MUJ*D(J)))/LAMBJ
C
C UPDATES J-TH COLUMN OF TL
C
         IF (ISWUP .LE. 1) THEN
            DO 550 K=J+1,N
               TL(K,J) = TL(K,J) + T*(BETLJ+THET1*GAMLJ)*WTJP1(K) +
     +            (ALPHA*BETLJ+THET2*GAMLJ)*ZTJP1(K)
550         CONTINUE
         ELSE
            DO 560 K=J+1,N
               TL(K,J) = TL(K,J)/LAMBJ2 + BETLJ*WTIL(K) + GAMLJ*ZTIL(K)
               WTIL(K) = WTJP1(K)
               ZTIL(K) = ZTJP1(K)
560         CONTINUE
         END IF
C
590   CONTINUE
      AJ = NU*Z(N) - ETA*W(N)
      LAMBJ = 1.0 + AJ*W(N)/D(N)
C
C NOTICE THAT THIS IS ACTUALLY LAMBDA(N), NEEDED TO UPDATE D(N)
C
      DP(N) = D(N)*LAMBJ*LAMBJ
C
C TRANSFER DP IN D
C
      DO 600 I=1,N
         TL(I,I) = DP(I)
600   CONTINUE
      GO TO 900
C
C HANDLE N=1
C
700   TL(1,1) = -Y(1) * TL(1,1) / (T * GSAVE(1))
      GO TO 900
C
C COMES HERE IF DIVISION BY ZERO WOULD OCCUR IN OPTIMAL CONDITIONING
C
800   ISTAT = -1
      GO TO 999
C
900   ISTAT = 0
999   RETURN
      END
      SUBROUTINE STEP(FXC,N,X,T,TMIN,TBND,IBND,IP,IVERB,XT,P,
     +   XALL,NALL,ITP,NFE,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,ISTAT,iop)
C---
C--- PERFORMS LINE SEARCH ALONG DIRECTION P
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION X(*),XT(*),P(*)
      DIMENSION XALL(NALL),ITP(NALL)
      DIMENSION CC(NC),TH(NC),SG(NC),SC(NC),IAC(NC)
C
      DATA TENTH,CURV /0.1D0, 0.75D0/
C
C OBTAIN FT=F(X+T*P)
C
      DO 10 I=1,N
         XT(I) = X(I) + T*P(I)
10    CONTINUE
      IF (IVERB .GE. 5)
     +   WRITE (IP,*) 'STEP) LINE SEARCH (PROJECTED MIN):'
      CALL ALFUN(FT,XT,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +   INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
C TEST SLOPE OF LINE FROM F TO FT
C
      IF (FXC .LE. FT) GO TO 200
C
C ----- DECREASING, HENCE T DOUBLED -----
C
100   DO 110 I=1,N
         X(I) = XT(I)
110   CONTINUE
C
      TWOT = T+T
C
C CHECK BOUNDS IF ANY
C
      IF (IBND .GT. 0 .AND. TWOT .GE. TBND) THEN
         FXC = FT
         IF (IVERB .GT. 1) WRITE (IP,1100)
 1100    FORMAT(1X,10('*'),'ACTIVE BOUNDARY CONSTRAINT',10('*'))
         GO TO 900
      END IF
C
C NO ACTIVE BOUNDARY CONSTRAINT
C
      DO 120 I=1,N
         XT(I) = X(I) + T*P(I)
120   CONTINUE
      IF (IVERB .GE. 5)
     +   WRITE (IP,*) 'STEP) LINE SEARCH (T DOUBLED):'
      CALL ALFUN(F2T,XT,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +   INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
C TEST SLOPE OF LINE FROM FT TO F2T
C TRANSFER TO EXIT SECTION IF MINIMUM BRACKETED
C
      IF (F2T .GE. FT) THEN
         FXC = FT
         GO TO 900
      END IF
C
C F, FT, F2T MONOTONIC DECREASING, CHECK CURVATURE
C
C CHECK IF CONCAVE
C
      IF (FXC-FT .LT. FT-F2T .OR.
C
C CHECK IF CONVEX STEEP OR CONVEX FLAT
C
     +    FT-F2T .GE. CURV*(FXC-FT)) THEN
C
C CONVEX STEEP OR CONCAVE
C
         T = TWOT
         FT = F2T
         GO TO 100
      END IF
C
C CONVEX FLAT, TRANSFER TO EXIT SECTION
C
      FXC = F2T
      DO 130 I=1,N
         X(I) = XT(I)
130   CONTINUE
      T = TWOT
C
      GO TO 900
C
C ----- ASCENDING LINE FROM F TO FT -----
C
C T WILL BE HALVED IF SOME CONDITIONS TESTED HERE ARE MET
C
200   CONTINUE
C
C COMES HERE IF NO PROGRESS YET MADE, SWITCH TO CENTRAL DIFFERENCE
C HAS ALREADY OCCURRED, AND LIMIT ON POSSIBLE ACCURACY MAY HAVE BEEN
C REACHED.  IN FACT, WILL EXIT IF T LESS THAN TMIN
C
      IF (T .LT. TMIN .OR. T .LE. 0.0) GO TO 810
C
C T IS HALVED
C
      T = .5*T
      DO 210 I=1,N
         XT(I) = X(I) + T*P(I)
210   CONTINUE
      IF (IVERB .GE. 5)
     +   WRITE (IP,*) 'STEP) LINE SEARCH (T HALVED):'
      CALL ALFUN(FT2,XT,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +   INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
C CHECK IF MINIMUM IS BRACKETED. IF YES, GO TO EXIT SECTION. IF NOT,
C WILL FIND SMALLER T VALUE IN NEXT BOX
C
      IF (FT2 .LT. FXC) THEN
C
C MINIMUM BRACKETED
C
         FXC = FT2
         DO 220 I=1,N
            X(I) = XT(I)
220      CONTINUE
         IF (T .LT. TMIN) GO TO 820
         GO TO 900
      END IF
C
C MINIMUM NOT BRACKETED, F, FT2, FT MONOTONIC INCREASING
C P BEING A DESCENT DIRECTION, THERE MUST BE A SMALLER T FOR WHICH F IS
C DECREASED, THAT LIES BETWEEN T AND T/2
C
C CHECK CURVATURE
C
      IF (FT-FT2 .LE. FT2-FXC) THEN
C
C CONCAVE
C
         SCAL = .1D0
      ELSE
C
C CONVEX, PARABOLIC INTERPOLATION PERFORMED
C
         SCAL = 1.0 + 0.5*(FXC-FT)/((FXC-FT2)+(FT-FT2))
         SCAL = MAX( TENTH, SCAL )
      END IF
C
C RESTART WITH SMALLER T
C
      T = SCAL*T
      DO 230 I=1,N
         XT(I) = X(I) + T*P(I)
230   CONTINUE
      IF (IVERB .GE. 5)
     +   WRITE (IP,*) 'STEP) LINE SEARCH (RESTART WITH SMALLER T):'
      CALL ALFUN(FT,XT,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +   INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
C IF MINIMUM NOT BRACKETED, CONTINUE SEARCH BY RETURNING TO 200
C IF MINIMUM BRACKETED, GO TO EXIT SECTION
C
      IF (FT .GE. FXC) GO TO 200
C
      FXC = FT
      DO 240 I=1,N
         X(I) = XT(I)
240   CONTINUE
C
      IF (T .LT. TMIN) GO TO 820
C
C----------------------------------------------------------------------
C
C EXIT SECTION
C
900   ISTAT = 0
      RETURN
C
C T < TMIN AND MINIMUM NOT BRACKETED
C (ACCUMULATION OF ROUNDING ERRORS PREVENTS FURTHER PROGRESS)
C
810   ISTAT = -1
      RETURN
C
C T < TMIN AND MINIMUM BRACKETED
C (MAXIMUM POSSIBLE ACCURACY REACHED)
C
820   ISTAT = -2
      END
      SUBROUTINE CHKBND(N,X,BNDL,BNDU,P,T,TBND,H,ISTAT)
C---
C--- CHECK BOUNDARY CONDITIONS ON PARAMETERS
C---
C--- HANDLES SETTING A PARAMETER TO A BOUND; AND
C--- DETERMINES TBND, THE MAXIMUM STEPSIZE T ADMISSIBLE
C---
C--- ISTAT: IF NON-ZERO, INDICATES WHICH PARAMETER WAS SET TO A BOUND
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION X(*),BNDL(*),BNDU(*),P(*)
      data iclose /0/
C
C CHECK CLOSENESS TO BOUND PRIOR TO DETERMINING TBOUND
C
      CLB = X(1) - BNDL(1)
      DO 100 I=1,N
C        LOWER BOUND:
         CHECK = X(I) - BNDL(I)
         IF (CHECK .LE. CLB) THEN
            CLB = CHECK
            ICLOSE = I
         END IF
C
C        UPPER BOUND:
         CHECK = BNDU(I) - X(I)
         IF (CHECK .LT. CLB) THEN
            CLB = CHECK
            ICLOSE = I
         END IF
100   CONTINUE
C
C CHECK FOR A PARAMETER RIDING A BOUND IN SUCCESSIVE ITERATIONS,
C IN WHICH CASE IT WILL BE FIXED TO ITS BOUND.  WE ASSUME THAT
C THE INITIAL VALUE OF X(I) WAS GREATER THAN BNDL(I)+2.*H AND LOWER
C THAN BNDU(I)-2.*H
C
      IF (CLB .LE. 0.01D0*H) GO TO 800
C
C FIND MAXIMUM STEPSIZE T ADMISSIBLE
C
      TBND = 1.D05
      DO 200 I=1,N
C        LOWER BOUND:
         TEQ = (BNDL(I)-X(I)) / P(I)
         IF (TEQ .LT. TBND .AND. TEQ .GT. 0.0) TBND = TEQ
C
C        UPPER BOUND:
         TEQ = (BNDU(I)-X(I)) / P(I)
         IF (TEQ .LT. TBND .AND. TEQ .GT. 0.0) TBND = TEQ
200   CONTINUE
C
C INSURE THAT INITIAL T-VALUE KEEPS SEARCH WITHIN PERMISSIBLE DOMAIN
C
      IF (T*(2.+H) .GE. TBND) T = TBND*(.5-H)
C
      ISTAT = 0
      RETURN
C
C INDICATE PARAMETER TO SET TO BOUND
C
800   ISTAT = ICLOSE
      END
      SUBROUTINE SETBND(ICLOSE,FXC,NALL,XALL,ITP,N,X,BNDL,BNDU,H,
     +   IP,IVERB,NFE,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,ISTAT,iop)
C---
C--- X(ICLOSE), CLOSE TO A BOUND, IS SET TO ITS BOUND.  ITERATION
C--- WILL RESTART AT THE BEGINNING, WITH X(ICLOSE) DROPPED FROM
C--- THE ITERATED SET.
C---
C--- ISTAT IS RETURNED AS THE LOCATION OF X(ICLOSE) IN XALL
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION X(*),BNDL(*),BNDU(*)
      DIMENSION XALL(NALL),ITP(NALL)
      DIMENSION CC(NC),TH(NC),SG(NC),SC(NC),IAC(NC)
CCC      DIMENSION TL(MAXITP,MAXITP)
C
C FIND X(ICLOSE) IN XALL
C
      K = 0
      DO 100 I=1,NALL
         II = I
         IF (ITP(I) .EQ. 1) THEN
            K = K + 1
            IF (K .EQ. ICLOSE) GO TO 110
         END IF
100   CONTINUE
      call intpr
     +('ALMINI (SETBND): INTERNAL ERROR - VALUE OF "ICLOSE" INVALID',
     +-1,0,0)
      STOP
C
110   CONTINUE
C
C SET PARAMETER TO APPROPRIATE BOUND
C
      BNDLO = XALL(II) - BNDL(ICLOSE)
      BNDHI = BNDU(ICLOSE) - XALL(II)
C
      TWOH = H+H
      IF (BNDLO .LT. BNDHI) THEN
         XALL(II) = BNDL(ICLOSE) - TWOH
      ELSE
         XALL(II) = BNDU(ICLOSE) + TWOH
      END IF
C
C REMOVE X(ICLOSE) FROM THE ITERATED SET
C
      DO 200 I=ICLOSE,N-1
         X(I) = X(I+1)
         BNDL(I) = BNDL(I+1)
         BNDU(I) = BNDU(I+1)
200   CONTINUE
C
      ITP(II) = 0
      N = N - 1
CC
CC REMOVE ROW=COL=ICLOSE FROM FACTORIZED APPROXIMATE HESSIAN
CC
CC (THIS SECTION OF CODE IS CURRENTLY DISABLED WITH "IHESS=0" IN
CC RESTART LOGIC OF CALLING ROUTINE, PENDING DISCUSSION WITH JML... SR)
CC
C      DO 330 I=ICLOSE,N
C         DO 310 J=1,ICLOSE-1
C            TL(J,I) = TL(J,I+1)
C310      CONTINUE
C         DO 320 J=ICLOSE,N
C            TL(J,I) = TL(J+1,I+1)
C320      CONTINUE
C330   CONTINUE
C
C GET NEW CURRENT FUNCTION VALUE
C
      IF (IVERB .GE. 5)
     +   WRITE (IP,*) 'STEP) LINE SEARCH (FINAL):'
      CALL ALFUN(FXC,X,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +   INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
C WRITE A MESSAGE TO PROLIX
C
      IF (IVERB .GT. 0) WRITE (IP,1000)
 1000 FORMAT(1X,10('*'),' A VARIABLE WAS SET TO A BOUND ',10('*')/)
C
      ISTAT = II
      END
      SUBROUTINE INIB(FXC,N,X,IHESS,H,IHX,TL,MAXITP,IP,IVERB,
     +   XALL,NALL,ITP,NFE,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C---
C--- OBTAINS INITIAL APPROXIMATION TO HESSIAN, AS SPECIFIED
C--- BY VARIABLE IHESS.  RESULT, IN CHOLESKY FACTORIZED FORM,
C--- IS RETURNED IN TL(I,J), I.GE.J
C---
C--- ARGUMENTS:
C---  FXC   -- VALUE OF F(X) (INPUT)
C---  IHESS -- CODED AS FOLLOWS (INPUT)
C---     0=IDENTITY MATRIX:  CRUDE APPROXIMATION, (FAST BUT WILL
C---        RESULT IN MORE ITERATIONS TO CONVERGENCE)
C---     1=FINITE DIFFERENCE:  MOST ACCURATE METHOD, BUT WILL TAKE
C---        ORDER N-SQUARED FUNCTION EVALUATIONS TO CALCULATE
C---     2=SUPPLIED BY USER (UNFACTORIZED) IN TL(I,J), I.GE.J
C---  MAXITP-- DIMENSION OF MATRIX TL IN CALLING PROGRAM (INPUT)
C---  TL    -- FACTORIZED HESSIAN MATRIX RETURNED
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION X(*),TL(MAXITP,MAXITP)
      DIMENSION XALL(NALL),ITP(NALL)
      DIMENSION CC(NC),TH(NC),SG(NC),SC(NC),IAC(NC)
C
      IF (MAXITP .LT. N) THEN
         call intpr
     +   ('ALMINI (INIB): ARRAY DIMENSION "MAXITP" INADEQUATE',-1,0,0)
         STOP
      END IF
C
C INITIAL B-MATRIX IS IDENTITY MATRIX IF IHESS=0
C
      IF (IHESS .EQ. 0) THEN
         GO TO 800
C
C GET B-MATRIX BY FINITE DIFFERENCE IF IHESS=1
C
      ELSE IF (IHESS .EQ. 1) THEN
         CALL HESSN(FXC,N,X,H,IHX,TL,MAXITP,IP,IVERB,
     +      XALL,NALL,ITP,NFE,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
C B-MATRIX IS ALREADY THERE IF IHESS=2
C
      ELSE IF (IHESS .NE. 2) THEN
         call intpr('ALMINI (INIB): INVALID VALUE FOR IHESS',-1,0,0)
         STOP
      END IF
C
C CARRY OUT CHOLESKY DECOMPOSITION IN-PLACE
C
      DO 130 I=1,N
         TLII = TL(I,I)
         IF (TLII .LE. 0.0) GO TO 210
         DO 120 J=I+1,N
            TLJI = TL(J,I) / TLII
            DO 110 K=J,N
               TL(K,J) = TL(K,J) - TLJI*TL(K,I)
110         CONTINUE
            TL(J,I) = TLJI
120      CONTINUE
130   CONTINUE
C
C INDICATE WHETHER OR NOT B-MATRIX WAS FOUND TO BE POSITIVE DEFINITE
C
      IF (IVERB .GT. 0) WRITE (IP,1200)
 1200 FORMAT(/10X,'FACTORIZATION OF B-MATRIX HAS SUCCEEDED',/)
      GO TO 999
C
210   IF (IVERB .GT. 0) WRITE (IP,1210)
 1210 FORMAT(/10X,'FACTORIZATION OF B-MATRIX HAS FAILED',/)
C
C IF FACTORIZATION FAILED, B WAS NOT P.D.; START WITH A DIAGONAL
C APPROXIMATION TO B
C
800   DO 820 I=1,N
         TL(I,I) = 1.0
         DO 810 J=1,I-1
            TL(I,J) = 0.0
810      CONTINUE
820   CONTINUE
C
999   RETURN
      END
      SUBROUTINE HESSN(FXC,N,X,H,IHX,HESS,MAXITP,IP,IVERB,
     +   XALL,NALL,ITP,NFE,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C---
C--- OBTAINS APPROXIMATE HESSIAN BY FINITE DIFFERENCE
C---
C--- ARGUMENTS:
C---  FXC   -- VALUE OF F(X)
C---  H     -- DIFFERENTIATION INTERVAL
C---  IHX   -- NON-ZERO MEANS SCALE H TO PARAMETER VALUE
C---  IP    -- PROLIX FILE
C---  HESS  -- HESSIAN MATRIX RETURNED
C---  MAXITP-- DIMENSION OF HESS (SQUARE)
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION X(*),HESS(MAXITP,MAXITP)
      DIMENSION XALL(NALL),ITP(NALL)
      DIMENSION CC(NC),TH(NC),SG(NC),SC(NC),IAC(NC)
C
      DATA HHMIN / 0.001D0 /, hh /0.1/, hmin /0.1/
C
C GET SCALING FACTOR, AND LOWER LIMIT ON DIFFERENTIATION
C INTERVAL, IF SCALED
C
      IF (IHX .NE. 0) THEN
         HH = MIN( 10.*H, HHMIN)
         HMIN = 0.1D0
      END IF
C
C GET MATRIX OF SECOND DERIVATIVES BY FORWARD DIFFERENCE
C
      DO 200 I=1,N
         TEMP1 = X(I)
         IF (IHX .EQ. 0) THEN
            HI = H
         ELSE
            HI = HH*MAX( X(I), HMIN )
         END IF
C
         X(I) = X(I) + HI
         IF (IVERB .GE. 5)
     +      WRITE (IP,*) 'HESSN) SECOND DERIVATIVES: I=', I
         CALL ALFUN(FI,X,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +      INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
         HESS(I,N) = FI
C
         DO 100 J=1,I
            TEMP2 = X(J)
            IF (IHX .EQ. 0) THEN
               HJ = H
            ELSE
               HJ = HH*MAX( X(J), HMIN )
            END IF
C
            X(J) = X(J) + HJ
            IF (IVERB .GE. 5)
     +         WRITE (IP,*) 'HESSN) SECOND DERIVATIVES: J=', J
            CALL ALFUN(FIJ,X,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +         INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
            FJ = HESS(J,N)
            HESS(I,J) = ((FXC-FI)+(FIJ-FJ)) / (HI*HJ)
            HESS(J,I) = HESS(I,J)
C
            X(J) = TEMP2
100      CONTINUE
         X(I) = TEMP1
200   CONTINUE
C
C WRITE HESSIAN TO PROLIX
C
      IF (IVERB .GT. 2) THEN
         WRITE (IP,1000)
         DO 400 I=1,N
            WRITE (IP,1100) (HESS(J,I), J=1,I)
400      CONTINUE
      END IF
C
 1000 FORMAT(/' HESSIAN OF F(X)')
 1100 FORMAT (1X,1P,7G11.3)
C
      END
      SUBROUTINE COVAR(FXC,N,X,H,IHX,COV,MAXITP,IP,IVERB,
     +   XALL,NALL,ITP,NFE,INLC,FX,FC,NC,CC,TH,SG,SC,IAC,ISTAT,iop)
C---
C--- FINDS VARIANCE-COVARIANCE MATRIX OF ITERATED PARAMETERS
C---
C--- ARGUMENTS:
C---  FX    -- VALUE OF F(X) WITHOUT CONSTRAINTS (INPUT)
C---  H,IHX -- (SEE SUBROUTINE HESSN)
C---  COV   -- COVARIANCE MATRIX (RETURNED)
C---  MAXITP-- DIMENSION OF MATRIX COV
C---  IP    -- PROLIX FILE
C---  ISTAT -- 0=SUCCESSFUL; -1=FAILED (RETURNED)
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION X(*),COV(MAXITP,MAXITP)
      DIMENSION XALL(NALL),ITP(NALL)
      DIMENSION CC(NC),TH(NC),SG(NC),SC(NC),IAC(NC)
C
      IF (MAXITP .LT. N) THEN
         call intpr(
     +'ALMINI (COVAR): ARRAY DIMENSION "MAXITP" INADEQUATE',-1,0,0)
         STOP
      END IF
C
C GET HESSIAN OF F(X) BY FINITE DIFFERENCE, WITHOUT CONSTRAINTS
C
      FXTEMP = FX
      INLCTM = 0
C
      CALL HESSN(FXTEMP,N,X,H,IHX,COV,MAXITP,IP,IVERB,
     +   XALL,NALL,ITP,NFE,INLCTM,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
C RESTORE CONSTRAINTS
C
      IF (IVERB .GE. 5)
     +   WRITE (IP,*) 'COVAR) FINAL F (CONSTRAINTS RESTORED)'
      CALL ALFUN(FXC,X,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +   INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C
C GET COVARIANCE MATRIX BY INVERTING HESSIAN
C
      CALL INVERT(COV,N,MAXITP,DETLOG,ISTAT)
      IF (ISTAT .NE. 0) GO TO 800
C
C WRITE COVARIANCE MATRIX TO PROLIX
C
      IF (IVERB .GT. 1) THEN
         WRITE (IP,1000)
         DO 300 I=1,N
            WRITE (IP,1100) (COV(J,I),J=1,I)
300      CONTINUE
      END IF
C
 1000 FORMAT(/' COVARIANCE MATRIX')
 1100 FORMAT (1X,1P,7G11.3)
      GO TO 999
C
C WRITE MESSAGE IF HESSIAN WAS NUMERICALLY NON POSITIVE-DEFINITE
C
800   IF (IVERB .GT. 0) WRITE (IP,1800)
 1800 FORMAT(/10X,'FACTORIZATION OF B-MATRIX HAS FAILED',/)
C
999   RETURN
      END
      SUBROUTINE ALFUN(FXC,X,N,XALL,NALL,ITP,NFE,IP,IVERB,
     +   INLC,FX,FC,NC,CC,TH,SG,SC,IAC,iop)
C---
C--- INTERFACE BETWEEN ALMINI AND USER THE SUPPLIED FUNCTION SUBROUTINE
C---
C--- FXC = AUGMENTED LAGRANGIAN FUNCTION
C--- FX = THE FUNCTION TO MINIMIZE SUBJECT TO CONSTRAINTS
C--- FC = THE TERM INVOLVING THE CONSTRAINTS IN FXC
C---
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION XALL(NALL),ITP(NALL),X(*)
      DIMENSION CC(NC),TH(NC),SG(NC),SC(NC),IAC(NC)
C
C UPDATE XALL
C
      K = 0
      DO 10 I=1,NALL
         IF (ITP(I) .EQ. 1) THEN
            K = K + 1
            XALL(I) = X(K)
         END IF
10    CONTINUE
      IF (K .NE. N) THEN
         call intpr
     +   ('ALMINI (ALFUN): INTERNAL INITIALIZATION ERROR',-1,0,0)
         STOP
      END IF
C
C HANDLE "EQUAL PARAMETERS":  (UNDOCUMENTED)
C A NEGATIVE VALUE IN ITP(I) INDICATES THAT MULTIPLE ELEMENTS
C IN XALL() CORRESPOND TO A SINGLE ITERATED PARAMETER
C
      DO 20 I=1,NALL
         IF (ITP(I) .LT. 0) XALL(I) = XALL( -ITP(I) )
20    CONTINUE
C
C GET FX USING THE USER SUPPLIED FUNCTION SUBROUTINE
C
C iop = almtest, gemtest, 3a, 3b
      if (iop .eq. 1) then
         CALL FUNalm(FX,XALL,NALL,CC,NC)
      else if (iop. eq. 2) then
         CALL FUNgem(FX,XALL,NALL,CC,NC)
      else if (iop .eq. 3) then
         CALL FUN3a(FX,XALL,NALL,CC,NC)
      else if (iop .eq. 4) then
         CALL FUN3b(FX,XALL,NALL,CC,NC)
      endif
C
      FXC = FX
C
C CALCULATE TERMS INVOLVING CONSTRAINTS, AND THEN FXC
C
      IF (INLC .NE. 0) THEN
         CALL ALFC(FC,NC,CC,TH,SG,SC,IAC)
         FXC = FX + 0.5*FC
      END IF
C
C UPDATE THE FUNCTION COUNT
C
      NFE = NFE + 1
C
C WRITE FUNCTION VALUES TO PROLIX
C
      IF (IVERB .GT. 3) THEN
         IF (INLC .NE. 0) THEN
            WRITE (IP,1100) NFE, FXC, FX, FC
         ELSE
            WRITE (IP,1100) NFE, FXC
         END IF
      END IF
 1100 FORMAT (1X,I9,')  FXC=',1P,G19.12,:,'  FX=',G19.12,' FC=',G15.8)
C
      END
