\name{deconvolve}
\alias{deconvolve}
\title{Richardson-Lucy Deconvolution}
\description{
Performs a modified Richardson-Lucy iteration for the purpose of
estimating incidence from reported incidence or mortality, conditional
on a reporting probability and on a distribution of the time to
reporting.
}
\usage{
deconvolve(x, prob = 1, delay = 1,
           start, tol = 1, iter.max = 32L, complete = FALSE)
}
\arguments{
\item{x}{
  a numeric vector of length \code{n} giving the number of infections or
  deaths reported during \code{n} observation intervals of equal
  duration.}
\item{prob}{
  a numeric vector of length \code{d+n} such that \code{prob[d+i]} is
  the probability that an infection during interval \code{i} is
  eventually reported.  \code{prob} of length 1 is recycled.}
\item{delay}{
  a numeric vector of length \code{d+1} such that \code{delay[j]} is the
  probability that an infection during interval \code{i} is reported
  during interval \code{i+j-1}, given that it is eventually reported.
  \code{delay} need not sum to 1 but must not sum to 0.}
\item{start}{
  a numeric vector of length \code{d+n} giving a starting value for the
  iteration.  \code{start[d+i]} estimates the expected number of
  infections during interval \code{i} that are eventually reported.  If
  missing, then a starting value is generated by padding \code{x} on the
  left and right with \code{d-d0} and \code{d0} zeros, choosing
  \code{d0 = which.max(delay)-1}.}
\item{tol}{
  a tolerance indicating a stopping condition; see the reference.}
\item{iter.max}{
  the maximum number of iterations.}
\item{complete}{
  a logical flag indicating if the result should preserve successive
  updates to \code{start}.}
}
\value{
A list with elements:
\item{value}{
  the result of updating \code{start} \code{iter} times then dividing by
  \code{prob}.  If \code{complete = TRUE}, then \code{value} is a
  \code{(d+n)}-by-\code{(1+iter)} matrix containing \code{start} and the
  \code{iter} successive updates, each divided by \code{prob}.}
\item{chisq}{
  the chi-squared statistics corresponding to \code{value}.}
\item{iter}{
  the number of iterations performed.}
}
%\details{}
%\seealso{}
\references{
Goldstein, E., Dushoff, J., Ma, J., Plotkin, J. B., Earn, D. J. D.,
& Lipsitch, M. (2020).
Reconstructing influenza incidence by deconvolution of daily mortality
time series.
\emph{Proceedings of the National Academy of Sciences U. S. A.},
\emph{106}(51), 21825-21829.
\doi{10.1073/pnas.0902958106}
}
\examples{
\dontshow{
## for R_DEFAULT_PACKAGES=NULL
library(   stats, pos = "package:base", verbose = FALSE)
library(graphics, pos = "package:base", verbose = FALSE)
library(   utils, pos = "package:base", verbose = FALSE)
}
set.seed(2L)
n <- 200L
d <- 50L
p <- 0.1
prob <- plogis(rlogis(d + n, location = qlogis(p), scale = 0.1))
delay <- diff(pgamma(0L:(d + 1L), 12, 0.4))

h <- function (x, a = 1, b = 1, c = 0) a * exp(-b * (x - c)^2)
ans <- floor(h(seq(-60, 60, length.out = d + n), a = 1000, b = 0.001))

x0 <- rbinom(d + n, ans, prob)
x <- tabulate(rep(1L:(d + n), x0) +
              sample(0L:d, size = sum(x0), replace = TRUE, prob = delay),
              d + n)[-(1L:d)]

str(D0 <- deconvolve(x, prob, delay, complete = FALSE))
str(D1 <- deconvolve(x, prob, delay, complete =  TRUE))

matplot(-(d - 1L):n,
        cbind(x0, c(rep(NA, d), x), prob * D0[["value"]], p * ans),
        type = c("p", "p", "p", "l"),
        col = c(1L, 1L, 2L, 4L), pch = c(16L, 1L, 16L, NA),
        lty = c(0L, 0L, 0L, 1L), lwd = c(NA, NA, NA, 3L),
        xlab = "Time", ylab = "Count")
legend("topleft", NULL,
       c("actual", "actual+delay", "actual+delay+deconvolution", "p*h"),
       col = c(1L, 1L, 2L, 4L), pch = c(16L, 1L, 16L, NA),
       lty = c(0L, 0L, 0L, 1L), lwd = c(NA, NA, NA, 3L),
       bty = "n")

plot(0L:D1[["iter"]], D1[["chisq"]], xlab = "Iterations", ylab = quote(chi^2))
abline(h = 1, lty = 2L)
}
