%  File man/ergm.Rd in package ergm, part of the Statnet suite
%  of packages for network analysis, http://statnet.org .
%
%  This software is distributed under the GPL-3 license.  It is free,
%  open source, and has the attribution requirements (GPL Section 7) at
%  http://statnet.org/attribution
%
%  Copyright 2003-2014 Statnet Commons
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\name{ergm}
\alias{ergm}
\alias{ergm.object}
\title{Exponential Family Random Graph Models}
\description{
  \code{\link{ergm}} is used to fit linear exponential random graph
  models (ERGMs), in which
  the probability of a given network, \eqn{y}, on a set of nodes is 
  \eqn{h(y) \exp\{\eta(\theta) \cdot
    g(y)\}/c(\theta)}, where
  \eqn{h(y)} is the reference measure (usually \eqn{h(y)=1}),
  \eqn{g(y)} is a vector of network statistics for \eqn{y},
  \eqn{\eta(\theta)} is a natural parameter vector of the same 
  length (with \eqn{\eta(\theta)=\theta} for most terms), and \eqn{c(\theta)} is the
  normalizing constant for the distribution.
  \code{\link{ergm}} can return either a maximum pseudo-likelihood
  estimate or an approximate maximum likelihood estimator based on a Monte Carlo scheme.
}
\usage{
ergm (formula,
      response=NULL,
      reference=~Bernoulli,
      constraints=~.,
      offset.coef=NULL,
      target.stats=NULL,
      eval.loglik=TRUE,
      estimate=c("MLE", "MPLE"),
      control=control.ergm(),
      verbose=FALSE,
      \dots)
}
\arguments{
  \item{formula}{An \R \code{\link{formula}} object, of the form
    \code{y ~ <model terms>},
    where \code{y} is a \code{\link[network]{network}} object or a matrix that can be
    coerced to a \code{\link[network]{network}}  object.  For the details on the possible
    \code{<model terms>}, see \code{\link{ergm-terms}} and Morris, Handcock and
    Hunter (2008) for binary ERGM terms and
    Krivitsky (2012) for valued ERGM
    terms (terms for weighted edges).  To create a
    \code{\link[network]{network}} object in \R, use the \code{network()} function,
    then add nodal attributes to it using the \code{\%v\%}
    operator if necessary. Enclosing a model term in \code{offset()}
    fixes its value to one specified in \code{offset.coef}.
  }
  \item{response}{\emph{EXPERIMENTAL.} Name of the edge attribute whose value is to be
    modeled. Defaults to \code{NULL} for simple presence or
    absence, modeled via binary ERGM terms. Passing
    anything but \code{NULL} uses valued ERGM terms.} 
  
  \item{reference}{\emph{EXPERIMENTAL.} A one-sided formula specifying
    the reference measure (\eqn{h(y)}) to be used. (Defaults to \code{~Bernoulli}.)
    See help for \link[=ergm-references]{ERGM reference measures} implemented in the
    \code{\link[=ergm-package]{ergm}} package.}
  
  \item{constraints}{A one-sided formula specifying one or more constraints
    on the support of the distribution of the networks being modeled,
    using syntax similar to the \code{formula} argument. Multiple constraints
    may be given, separated by \dQuote{+} operators.
    Together with the model terms in the formula and the reference measure, the constraints
    define the distribution of networks being modeled.

    It is also possible to specify a proposal function directly
    by passing a string with the function's name. In that case,
    arguments to the proposal should be specified through the
    \code{prop.args} argument to \code{\link{control.ergm}}.

    The default is \code{~.}, for an unconstrained model.

    See the \link[=ergm-constraints]{ERGM constraints} documentation for
    the constraints implemented in the \code{\link[=ergm-package]{ergm}}
    package. Other packages may add their own constraints.
    
    Note that not all possible combinations of constraints and reference
    measures are supported.
  }
  \item{offset.coef}{A vector of coefficients for the offset terms.}
  \item{target.stats}{vector of "observed network statistics,"
    if these statistics are for some reason different than the 
    actual statistics of the network on the left-hand side of
    \code{formula}.
    Equivalently, this vector is the mean-value parameter values for the
    model.  If this is given, the algorithm finds the natural
    parameter values corresponding to these mean-value parameters.
    If \code{NULL}, the mean-value parameters used are the observed
    statistics of the network in the formula.
  }
  \item{eval.loglik}{Logical:  For dyad-dependent models, if TRUE, use bridge
    sampling to evaluate the log-likelihoood associated with the
    fit. Has no effect for dyad-independent models.
    Since bridge sampling takes additional time, setting to FALSE may
    speed performance if likelihood values (and likelihood-based
    values like AIC and BIC) are not needed.
  }
  \item{estimate}{If "MPLE," then the maximum pseudolikelihood estimator
    is returned.  If "MLE" (the default), then an approximate maximum likelihood
    estimator is returned.  For certain models, the MPLE and MLE are equivalent,
    in which case this argument is ignored.  (To force MCMC-based approximate
    likelihood calculation even when the MLE and MPLE are the same, see the
    \code{force.main} argument of \code{\link{control.ergm}}. )
  }
  \item{control}{A list of control parameters for algorithm
    tuning. Constructed using \code{\link{control.ergm}}. 
  }
  \item{verbose}{logical; if this is
    \code{TRUE}, the program will print out additional
    information, including goodness of fit statistics.
  }
  \item{\dots}{Additional
    arguments, to be passed to lower-level functions.
  }
}

\value{\code{\link{ergm}} returns an object of class \code{\link{ergm}} that is a list
  consisting of the following elements:
 \item{coef}{The Monte Carlo maximum likelihood estimate
  of \eqn{\theta}, the vector of coefficients for the model
  parameters.}
 \item{sample}{The \eqn{n\times p} matrix of network statistics, 
  where \eqn{n} is the                               
  sample size and \eqn{p} is the number of network statistics specified in the
  model, that is used in the maximum likelihood estimation routine.}
 \item{sample.obs}{As \code{sample}, but for the constrained sample.}
 \item{iterations}{The number of Newton-Raphson iterations required
  before convergence.}
 \item{MCMCtheta}{The value of \eqn{\theta} used to produce the Markov chain
  Monte Carlo sample.  As long as the Markov chain mixes sufficiently
  well, \code{sample} is roughly a random sample from the distribution
  of network statistics specified by the model with the parameter equal
  to \code{MCMCtheta}.  If \code{estimate="MPLE"} then 
  \code{MCMCtheta} equals the MPLE.}
 \item{loglikelihood}{The approximate change in log-likelihood 
 in the last iteration.
  The value is only approximate because it is estimated based 
  on the MCMC random sample.}
 \item{gradient}{The value of the gradient vector of the approximated
  loglikelihood function, evaluated at the maximizer.  This vector
  should be very close to zero.}
 \item{covar}{Approximate covariance matrix for the MLE, based on the inverse
  Hessian of the approximated loglikelihood evaluated at the maximizer.}
 \item{failure}{Logical:  Did the MCMC estimation fail?}
 \item{network}{Original network}
 \item{newnetwork}{The final network at the end of the MCMC
 simulation}
 \item{coef.init}{The initial value of \eqn{\theta}.}
 \item{est.cov}{The covariance matrix of the model statistics in the final MCMC sample.}
 \item{coef.hist, steplen.hist, stats.hist, stats.obs.hist}{
   For the MCMLE method, the history of coefficients, Hummel step lengths, and average model statistics for each iteration..
}
 \item{control}{The control list passed to the call.}
  \item{etamap}{The set of functions mapping the true parameter theta
 to the canonical parameter eta (irrelevant except in a curved exponential
 family model)}
  \item{formula}{The original \code{\link{formula}} entered into the \code{\link{ergm}} function.}
  \item{target.stats}{The target.stats used during estimation (passed through from the Arguments)}
  \item{target.esteq}{Used for curved models to preserve the target mean values of the curved terms. It is identical to target.stats for non-curved models.}
  \item{constrained}{The list of constraints implied by the constraints used by original \code{ergm} call}
   \item{constraints}{Constraints used during estimation (passed through from the Arguments)}
  \item{reference}{The reference measure used during estimation (passed through from the Arguments)}
  \item{estimate}{The estimation method used (passed through from the Arguments).}
  \item{offset}{vector of logical telling which model parameters are to be set
 at a fixed value (i.e., not estimated).}
 
 \item{drop}{If \code{\link[=control.ergm]{control$drop}=TRUE}, a numeric vector indicating which terms were dropped due to to extreme values of the
   corresponding statistics on the observed network, and how:
   \describe{
     \item{\code{0}}{The term was not dropped.}
     \item{\code{-1}}{The term was at its minimum and the coefficient was fixed at
       \code{-Inf}.}
     \item{\code{+1}}{The term was at its maximum and the coefficient was fixed at
       \code{+Inf}.}
   }
 }
 
\item{estimable}{A logical vector indicating which terms could not be
  estimated due to a \code{constraints} constraint fixing that term at a
  constant value.
}

 \item{null.lik}{Log-likelihood of the null model. Valid only for
  unconstrained models.}
 \item{mle.lik}{The approximate log-likelihood for the MLE.
  The value is only approximate because it is estimated based 
  on the MCMC random sample.}

 \item{degeneracy.value}{Score calculated to assess the degree of 
 degeneracy in the model. Only shows when MCMLE.check.degeneracy is TRUE in \code{control.ergm}. }
 \item{degeneracy.type}{Supporting output for \code{degeneracy.value}. Only shows when MCMLE.check.degeneracy is TRUE in \code{control.ergm}. Mainly for internal use.}

  See the method \code{\link{print.ergm}} for details on how
  an \code{\link{ergm}} object is printed.  Note that the
  method \code{\link{summary.ergm}} returns a summary of the
  relevant parts of the \code{\link{ergm}} object in concise summary
  format.
}

\section{Notes on model specification}{
Although each of the statistics in a given model is a summary
statistic for the entire network, it is rarely necessary to
calculate statistics for an entire network
in a proposed Metropolis-Hastings step.
%  The reason for this is that we may assume that each statistic equals
% zero for the observed network (translating each statistic by
% a constant does not change the probability model), and then
% it is only necessary to calculate the changes in the various
%statistics for a proposed Metropolis-Hastings step.  
Thus, for example, if the triangle term is included in the model,
a census of all triangles in the observed network is never
taken; instead, only the change in the number of triangles
is recorded for each edge toggle.

In the implementation of \code{\link{ergm}}, the model is initialized
in \R, then all the model information is passed to a C program
that generates the sample of network statistics using MCMC.
This sample is then returned to \R, which implements a
simple Newton-Raphson algorithm to approximate the MLE.
An alternative style of maximum likelihood estimation is to use a stochastic
approximation algorithm. This can be chosen with the 
\code{control.ergm(style="Robbins-Monro")} option.
%There seems little reason to implement the Newton-Raphson
%algorithm in C since it moves very quickly in \R and
%it is easier to alter the code if it is written in \R.

The mechanism for proposing new networks for the MCMC sampling
scheme, which is a Metropolis-Hastings algorithm, depends on 
two things:  The \code{constraints}, which define the set of possible
networks that could be proposed in a particular Markov chain step,
and the weights placed on these possible steps by the 
proposal distribution.  The former may be controlled using the
\code{constraints} argument described above.  The latter may
be controlled using the \code{prop.weights} argument to the
\code{\link{control.ergm}} function.

The package is designed so that the user could conceivably add additional 
proposal types. 
}

\references{
%Handcock, M.S. (2000) \emph{Progress in Statistical Modeling of Drug User and 
%  Sexual Networks}, Center for Statistics and the Social Sciences,
%     University of Washington.

%Handcock, M. S. (2002)
%    \emph{Degeneracy and inference for social network models}
%    Paper presented at the Sunbelt XXII International Social
%Network Conference in New Orleans, LA.

Admiraal R, Handcock MS (2007).
 \pkg{networksis}: Simulate bipartite graphs with fixed
  marginals through sequential importance sampling.
 Statnet Project, Seattle, WA.
 Version 1. \url{statnet.org}.

Bender-deMoll S, Morris M, Moody J (2008).
 Prototype Packages for Managing and Animating Longitudinal
  Network Data: \pkg{dynamicnetwork} and \pkg{rSoNIA}.
 \emph{Journal of Statistical Software}, 24(7).
 \url{http://www.jstatsoft.org/v24/i07/}.

Butts CT (2006).
 \pkg{netperm}: Permutation Models for Relational Data.
 Version 0.2, \url{http://erzuli.ss.uci.edu/R.stuff}.

Butts CT (2007).
 \pkg{sna}: Tools for Social Network Analysis.
 Version 1.5, \url{http://erzuli.ss.uci.edu/R.stuff}.

Butts CT (2008).
 \pkg{network}: A Package for Managing Relational Data in \R.
 \emph{Journal of Statistical Software}, 24(2).
 \url{http://www.jstatsoft.org/v24/i02/}.

Butts CT, with help~from David~Hunter, Handcock MS (2007).
 \pkg{network}: Classes for Relational Data.
 Version 1.2, \url{http://erzuli.ss.uci.edu/R.stuff}.

Goodreau SM, Handcock MS, Hunter DR, Butts CT, Morris M (2008a).
 A \pkg{statnet} Tutorial.
 \emph{Journal of Statistical Software}, 24(8).
 \url{http://www.jstatsoft.org/v24/i08/}.

Goodreau SM, Kitts J, Morris M (2008b).
 Birds of a Feather, or Friend of a Friend? Using Exponential
  Random Graph Models to Investigate Adolescent Social Networks.
 \emph{Demography}, 45, in press.

Handcock, M. S. (2003)
    \emph{Assessing Degeneracy in Statistical Models of Social Networks},
    Working Paper \#39, 
Center for Statistics and the Social Sciences,
University of Washington.
\url{www.csss.washington.edu/Papers/wp39.pdf}

Handcock MS (2003b).
 \pkg{degreenet}: Models for Skewed Count Distributions Relevant
  to Networks.
 Statnet Project, Seattle, WA.
 Version 1.0, \url{statnet.org}.

Handcock MS, Hunter DR, Butts CT, Goodreau SM, Morris M (2003a).
 \pkg{ergm}: A Package to Fit, Simulate and Diagnose
  Exponential-Family Models for Networks.
 Statnet Project, Seattle, WA.
 Version 2, \url{statnet.org}.

Handcock MS, Hunter DR, Butts CT, Goodreau SM, Morris M (2003b).
 \pkg{statnet}: Software Tools for the Statistical Modeling of
  Network Data.
 Statnet Project, Seattle, WA.
 Version 2, \url{statnet.org}.

Hunter, D. R. and Handcock, M. S. (2006)
    \emph{Inference in curved exponential family models for networks},
   Journal of Computational and Graphical Statistics.

Hunter DR, Handcock MS, Butts CT, Goodreau SM, Morris M (2008b).
 \pkg{ergm}: A Package to Fit, Simulate and Diagnose
  Exponential-Family Models for Networks.
 \emph{Journal of Statistical Software}, 24(3).
 \url{http://www.jstatsoft.org/v24/i03/}.

Krivitsky PN (2012). Exponential-Family Random Graph Models for Valued
Networks. \emph{Electronic Journal of Statistics}, 2012, 6,
1100-1128. \href{http://dx.doi.org/10.1214/12-EJS696}{\code{doi:10.1214/12-EJS696}}
 
Morris M, Handcock MS, Hunter DR (2008).
 Specification of Exponential-Family Random Graph Models:
  Terms and Computational Aspects.
 \emph{Journal of Statistical Software}, 24(4).
 \url{http://www.jstatsoft.org/v24/i04/}.
 
Snijders, T.A.B. (2002),
Markov Chain Monte Carlo Estimation of Exponential Random Graph Models.
Journal of Social Structure.
Available from 
\url{http://www.cmu.edu/joss/content/articles/volume3/Snijders.pdf}.
 
}

\seealso{network, \%v\%, \%n\%, \code{\link{ergm-terms}}, \code{\link{ergmMPLE}},
  \code{\link{summary.ergm}}, \code{\link{print.ergm}}}
\examples{
\donttest{
#
# load the Florentine marriage data matrix
#
data(flo)
#
# attach the sociomatrix for the Florentine marriage data
# This is not yet a network object.
#
flo
#
# Create a network object out of the adjacency matrix
#
flomarriage <- network(flo,directed=FALSE)
flomarriage
#
# print out the sociomatrix for the Florentine marriage data
#
flomarriage[,]
#
# create a vector indicating the wealth of each family (in thousands of lira) 
# and add it as a covariate to the network object
#
flomarriage \%v\% "wealth" <- c(10,36,27,146,55,44,20,8,42,103,48,49,10,48,32,3)
flomarriage
#
# create a plot of the social network
#
plot(flomarriage)
#
# now make the vertex size proportional to their wealth
#
plot(flomarriage, vertex.cex=flomarriage \%v\% "wealth" / 20, main="Marriage Ties")
#
# Use 'data(package = "ergm")' to list the data sets in a
#
data(package="ergm")
#
# Load a network object of the Florentine data
#
data(florentine)
#
# Fit a model where the propensity to form ties between
# families depends on the absolute difference in wealth
#
gest <- ergm(flomarriage ~ edges + absdiff("wealth"))
summary(gest)
#
# add terms for the propensity to form 2-stars and triangles
# of families 
#
gest <- ergm(flomarriage ~ kstar(1:2) + absdiff("wealth") + triangle)
summary(gest)

# import synthetic network that looks like a molecule
data(molecule)
# Add a attribute to it to mimic the atomic type
molecule \%v\% "atomic type" <- c(1,1,1,1,1,1,2,2,2,2,2,2,2,3,3,3,3,3,3,3)
#
# create a plot of the social network
# colored by atomic type
#
plot(molecule, vertex.col="atomic type",vertex.cex=3)

# measure tendency to match within each atomic type
gest <- ergm(molecule ~ edges + kstar(2) + triangle + nodematch("atomic type"),
  control=control.ergm(MCMC.samplesize=10000))
summary(gest)

# compare it to differential homophily by atomic type
gest <- ergm(molecule ~ edges + kstar(2) + triangle
                              + nodematch("atomic type",diff=TRUE),
  control=control.ergm(MCMC.samplesize=10000))
summary(gest)
}
}
\keyword{models}
