% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EDR_metrics.R
\name{EDR_metrics}
\alias{EDR_metrics}
\alias{dDis}
\alias{dBD}
\alias{dEve}
\title{Metrics of Ecological Dynamic Regimes}
\usage{
dDis(
  d,
  d.type,
  trajectories,
  states = NULL,
  reference,
  w.type = "none",
  w.values,
  ...
)

dBD(d, d.type, trajectories, states = NULL, ...)

dEve(d, d.type, trajectories, states = NULL, w.type = "none", w.values, ...)
}
\arguments{
\item{d}{Symmetric matrix or object of class \code{\link{dist}} containing the
dissimilarities between each pair of states of all trajectories in the EDR or
the dissimilarities between each pair of trajectories. To compute dDis, \code{d}
needs to include the dissimilarities between all states/trajectories and the
states/trajectory of reference.}

\item{d.type}{One of \code{"dStates"} (if \code{d} contains state dissimilarities) or
\code{"dTraj"} (if \code{d} contains trajectory dissimilarities).}

\item{trajectories}{Vector indicating the trajectory or site corresponding to
each entry in \code{d}.}

\item{states}{Only if \code{d.type} = \code{"dStates"}. Vector of integers indicating the
order of the states in \code{d} for each trajectory.}

\item{reference}{Vector of the same class as \code{trajectories} and length equal
to one, indicating the reference trajectory to compute dDis.}

\item{w.type}{Method used to weight individual trajectories:
\itemize{
\item \code{"none"}: All trajectories are considered equally relevant (default).
\item \code{"length"}: Trajectories are weighted by their length, calculated as the
sum of the dissimilarities between every pair of consecutive states. \code{d} must
contain dissimilarities between trajectory states and \code{d.type} = \code{"dStates"}.
\item \code{"size"}: Trajectories are weighted by their size, calculated as the number
of states forming the trajectory. \code{d} must contain dissimilarities between
trajectory states and \code{d.type} = \code{"dStates"}.
\item \code{"precomputed"}: Trajectories weighted according to different criteria.
}}

\item{w.values}{Only if \code{w.type} = \code{"precomputed"}. Numeric vector of length
equal to the number of trajectories containing the weight of each trajectory.}

\item{...}{Only if \code{d.type} = \code{"dStates"}. Further arguments to calculate
trajectory dissimilarities. See \code{\link[ecotraj:trajectorymetrics]{ecotraj::trajectoryDistances()}}.}
}
\value{
\itemize{
\item \code{dDis()} returns the value of dynamic dispersion for a given trajectory
taken as a reference.
\item \code{dBD()} returns the value of dynamic beta diversity.
\item \code{dEve()} returns the value of dynamic evenness.
}
}
\description{
Set of metrics to analyze the distribution and variability of trajectories
in Ecological Dynamic Regimes (EDR), including dynamic dispersion (dDis), dynamic
beta diversity (dBD), and dynamic evenness (dEve).
}
\details{
\strong{Dynamic dispersion (\code{dDis()})}

\emph{dDis} is calculated as the average dissimilarity between each trajectory in an
EDR and a target trajectory taken as reference (Sánchez-Pinillos et al., 2023).

\eqn{
dDis = \frac{\sum_{i=1}^{m}d_{i\alpha}}{m}
}

where \eqn{d_{i\alpha}} is the dissimilarity between trajectory \eqn{i} and
the trajectory of reference \eqn{\alpha}, and \eqn{m} is the number of trajectories.

Alternatively, it is possible to calculate a weighted mean of the dissimilarities
by assigning a weight to each trajectory.

\eqn{
dDis = \frac{\sum_{i=1}^{m}w_{i}d_{i\alpha}}{\sum_{i=1}^{m}w_{i}}
}

where \eqn{w_{i}} is the weight assigned to trajectory \eqn{i}.

\strong{Dynamic beta diversity (\code{dBD()})}

\emph{dBD} quantifies the overall variation of the trajectories in an EDR and is
equivalent to the average distance to the centroid of the EDR
(De Cáceres et al., 2019).

\eqn{
dBD = \frac{\sum_{i=1}^{m-1}\sum_{j=i+1}^{m}d_{ij}^{2}}{m(m-1)}
}

\strong{Dynamic evenness (\code{dEve()})}

\emph{dEve} quantifies the regularity with which an EDR is filled by the individual
trajectories (Sánchez-Pinillos et al., 2023).

\eqn{
dEve = \frac{\sum_{l=1}^{m-1}\min(\frac{d_{ij}}{\sum_{l=1}^{m-1}d_{ij}}, \frac{1}{m-1}) - \frac{1}{m-1}}{1-\frac{1}{1-1}}
}

where \eqn{d_{ij}} is the dissimilarity between trajectories \eqn{i} and \eqn{j}
linked in a minimum spanning tree by the link \eqn{l}.

Optionally, it is possible to weight the trajectories of the EDR. In that case,
\emph{dEve} becomes analogous to the functional evenness index proposed by Villéger
et al. (2008).

\eqn{
dEve_{w} = \frac{\sum_{l=1}^{m-1}\min(\frac{EW_{ij}}{\sum_{l=1}^{m-1}EW_{ij}}, \frac{1}{m-1}) - \frac{1}{m-1}}{1-\frac{1}{1-1}}
}

where \eqn{EW_{ij}} is the weighted evenness:

\eqn{
EW_{ij} = \frac{d_{ij}}{w_i + w_j}
}
}
\examples{
# Data to compute dDis, dBD, and dEve
dStates <- EDR_data$EDR1$state_dissim
dTraj <- EDR_data$EDR1$traj_dissim
trajectories <- paste0("T", EDR_data$EDR1$abundance$traj)
states <- EDR_data$EDR1$abundance$state

# Dynamic dispersion taking the first trajectory as reference
dDis(d = dTraj, d.type = "dTraj", trajectories = unique(trajectories),
         reference = "T1")

# Dynamic dispersion weighting trajectories by their length
dDis(d = dStates, d.type = "dStates", trajectories = trajectories, states = states,
         reference = "T1", w.type = "length")

# Dynamic beta diversity using trajectory dissimilarities
dBD(d = dTraj, d.type = "dTraj", trajectories = unique(trajectories))

# Dynamic evenness
dEve(d = dStates, d.type = "dStates", trajectories = trajectories, states = states)

# Dynamic evenness considering that the 10 first trajectories are three times
# more relevant than the rest
w.values <- c(rep(3, 10), rep(1, length(unique(trajectories))-10))
dEve(d = dTraj, d.type = "dTraj", trajectories = unique(trajectories),
         w.type = "precomputed", w.values = w.values)

}
\references{
De Cáceres, M, Coll L, Legendre P, Allen RB, Wiser SK, Fortin MJ, Condit R &
Hubbell S. (2019). Trajectory analysis in community ecology. Ecological
Monographs.

Sánchez-Pinillos, M., Kéfi, S., De Cáceres, M., Dakos, V. 2023. Ecological Dynamic
Regimes: Identification, characterization, and comparison. \emph{Ecological Monographs}.
\url{doi:10.1002/ecm.1589}

Villéger, S., Mason, N.W.H., Mouillot, D. (2008) New multidimensional functional
diversity indices for a multifaced framework in functional ecology. Ecology.
}
\author{
Martina Sánchez-Pinillos
}
