\name{format.earth}
\alias{format.earth}
\title{Format earth objects}
\description{
Return a string representing an \code{\link{earth}} expression.
}
\usage{
\method{format}{earth}(x = stop("no 'x' argument"),
       style = "h", decomp = "anova", digits = getOption("digits"),
       use.names = TRUE, colon.char = ":", \dots)
}
\arguments{
  \item{x}{
    An \code{\link{earth}} object.
    This is the only required argument.
  }
  \item{style}{
    Formatting style. One of\cr
\code{"h"} (default) more compact\cr
\code{"pmax"} for those who prefer it and for compatibility with old versions of earth\cr
\code{"max"} is the same as \code{"pmax"} but prints \code{max} rather than \code{pmax}\cr
\code{"C"} C style expression with zero based indexing\cr
\code{"bf"} basis function format
  }
  \item{decomp}{One of\cr
\code{"anova"}  (default) order the terms using the "anova decomposition",
i.e., in increasing order of interaction\cr
\code{"none"}   order the terms as created during the earth forward pass.\cr
  }
  \item{digits}{
    Number of significant digits.
    The default is \code{getOption(digits)}.
  }
  \item{use.names}{One of\cr
    \code{TRUE} (default), use variable names if available.\cr
    \code{FALSE} use names of the form \code{x[,1]}.
  }
  \item{colon.char}{
    Change colons in the returned string to colon.char.
    Default is ":" (no change).
    Specifying \code{colon.char="*"} can be useful in some contexts to change
    names of the form \code{x1:x2} to \code{x1*x2}.
  }
  \item{\dots}{
     Unused, but provided for generic/method consistency.
  }
}
\value{
A character representation of the \code{earth} model.

If there are multiple responses, \code{format.earth} will return multiple strings.

If there are embedded GLM model(s), the strings for the GLM model(s)
come after the strings for the standard \code{earth} model(s).
}
\note{
The FAQ section in the package vignette gives precise details of the
\code{"anova"} ordering.

Using \code{format.earth}, perhaps after hand editing the returned string,
you can create an alternative to \code{predict.earth}.
For example:
\preformatted{as.func <- function(object, digits = 8, use.names = FALSE, ...)
  eval(parse(text=paste(
    "function(x){\n",
    "if(is.vector(x))\n",
    "  x <- matrix(x, nrow = 1, ncol = length(x))\n",
    "with(as.data.frame(x),\n",
    format(object, digits = digits, use.names = use.names, style = "pmax", ...),
    ")\n",
    "}\n", sep = "")))

earth.mod <- earth(Volume ~ ., data = trees)
my.func <- as.func(earth.mod, use.names = FALSE)
my.func(c(10,80))             # returns 16.84
predict(earth.mod, c(10,80))  # returns 16.84
}
Note that with \code{pmax} the R expression generated by
\code{format.earth} can handle multiple cases.
Thus the expression is consistent with the way \code{predict}
functions usually work in R --- we can give \code{predict} multiple
cases (i.e., multiple rows in the input matrix) and it will return a
vector of predicted values.
\preformatted{ % trick to force a paragraph break TODO why is this needed?
}
The  earth package also provides a function \code{format.lm}.
It has arguments as follows\cr
\code{format.lm(x, digits=getOption("digits"), use.names=TRUE, colon.char=":")}\cr
(Strictly speaking, \code{format.lm} doesn't belong in the earth package.) Example:
\preformatted{
lm.mod <- lm(Volume ~ Height*Girth, data = trees)
cat(format(lm.mod, colon.char="*"))

# yields:
#    69.4
#    -  1.30 * Height
#    -  5.86 * Girth
#    + 0.135 * Height*Girth
}
}
\seealso{
  \code{\link{earth}},
  \code{\link{pmax}},
}
\examples{
earth.mod <- earth(Volume ~ ., data = trees)
cat(format(earth.mod))

# yields:
#    37.9
#    -  3.92 * h(16-Girth)
#    +   7.4 * h(Girth-16)
#    + 0.484 * h(Height-75)

cat(format(earth.mod, style="pmax")) # default formatting style prior to earth version 1.4

# yields:
#    37.9
#    -  3.92 * pmax(0,     16 -  Girth)
#    +   7.4 * pmax(0,  Girth -     16)
#    + 0.484 * pmax(0, Height -     75)

cat(format(earth.mod, style="C"))

# yields (note zero based indexing):
#  37.927
#    -  3.9187 * max(0,   16 - x[0])
#    +  7.4011 * max(0, x[0] -   16)
#    + 0.48411 * max(0, x[1] -   75)

cat(format(earth.mod, style="bf"))

# yields:
#    37.9
#    -  3.92 * bf1
#    +   7.4 * bf2
#    + 0.484 * bf3
#
#     bf1  h(16-Girth)
#     bf2  h(Girth-16)
#     bf3  h(Height-75)
}
\keyword{models}
